// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//
#ifndef __AtomCache_h__
#define __AtomCache_h__

#ifndef __Refcount_h__
#include "Refcount.h"
#endif

#ifndef __SP_h__
#include "SP.h"
#endif

#ifndef __AtomConfiguration_h__
#include "AtomConfiguration.h"
#endif

#ifndef __Dynavec_h__
#include "Dynavec.h"
#endif

#ifndef __myassert_h__
#include "myassert.h"
#endif

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

template <class AtomClass> class AtomCache
// Atoms are a flyweight, so when we are creating an atom we have to 
// reuse an old one if one is available.
// We use an array to store this cache, which is fast, except it has to be
// thrown away in the rare case when because of changing circumstances the
// scheme for computing indexes into the array changes.  This class
// encapsulates the caching scheme and it looks at the generation count of the
// AtomConfiguration to know when to discard the cache.
//
// This class includes a dynamically allocated pointer.  We will need that to
// be deallcoated before the leak detector runs, so we can't just put an
// AtomCache into a static variable.  Instead we have to put a pointer to an
// AtomCache into a static variable and register a deallocator that frees the
// pointer.
  : public Refcount
{
  int m_generation;
  Dynavec <SP<AtomClass> > *m_poVec;
  // You can't copy a cache.  Better to make a new one.
  AtomCache (const AtomCache &ac); // deliberately unimplemented
  void freshCache () {
    m_poVec = NEW (Dynavec <SP<AtomClass> > ());
  }
public:
  AtomCache ()
    : m_generation (-1)
  {
    freshCache ();
  }
  ~AtomCache () {
    assert (m_poVec);
    delete m_poVec;
    m_poVec = 0;
  }
  // Return a place that either has the cached atom, or has a null pointer and
  // we can put our atom to cache there.
  SP<AtomClass> &cacheref (AtomConfiguration *ac) {
    if (m_generation != ac->generation ()) {
      delete m_poVec;
      freshCache ();
      m_poVec->extendTo (ac->getDifferentAtoms ());
      m_generation = ac->generation ();
    }
    return (*m_poVec) [ac->myArrayPos ()];
  }
};
#endif
