// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __math_h__
#include <math.h>
#define __math_h__
#endif

#ifndef __Factory_h__
#include "Factory.h"
#endif

#ifndef __ScaleConfiguration_h__
#include "ScaleConfiguration.h"
#endif

#ifndef __myassert_h__
#include "myassert.h"
#endif

#ifndef __Action_h__
#include "Action.h"
#endif

#ifndef __FactoryTable_h__
#include "FactoryTable.h"
#endif

#ifndef __String_h__
#include "String.h"
#endif

#ifndef __Canvas_h__
#include "Canvas.h"
#endif

#ifndef __TopLevel_h__
#include "TopLevel.h"
#endif

#ifndef __Event_h__
#include "Event.h"
#endif

#ifndef __RecursiveSlotValue_h__
#include "RecursiveSlotValue.h"
#endif

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

#ifndef __ButtonSlotValue_h__
#include "ButtonSlotValue.h"
#endif

#ifndef __Vec3_h__
#include "Vec3.h"
#endif

#ifndef __KeyboardMouseEvent_h__
#include "KeyboardMouseEvent.h"
#endif

namespace {

  typedef Event::Button Button;

  class TranslateOrScaleConfiguration
    : public ScaleConfiguration
  {
    static String shiftKeyName () {
      return "Shift key";
    }
  public:
    // true if we're scaling, false if we're translating.
    bool m_scaling;
    // The x and y coordinates where the mouse originally went down.
    int m_x;
    int m_y;
    Float m_origScale;
    bool m_firstTime;
    TranslateOrScaleConfiguration ()
      // Not obvious that the initial value for m_scaling matters, but
      // determinism is good in general.
      : m_scaling (false),
	m_firstTime (true)
    {
      initializeSlot (shiftKeyName (),
		      NEW (ButtonSlotValue (NEW (KeyboardMouseEvent()),
					    KeyboardMouseEvent::SHIFT)));
    }
    Button getShift () const {
      return ButtonSlotValue::getSlot (this, shiftKeyName());
    }

    void setShift (Button b) {
      ButtonSlotValue::setSlot (this, shiftKeyName(), b);
    }

    SP<Configuration> copy () const {
      return NEW (TranslateOrScaleConfiguration (*this));
    }

    bool hasTheSameTypeAs (const Configuration *c) const {
      return 0 != dynamic_cast <const TranslateOrScaleConfiguration *> (c);
    }
  };

  class TranslateOrScale
    : public Factory
  {
  public:
    TranslateOrScale ()
      : Factory ("TranslateOrScale")
    {}
    SP<Configuration> defaultConfiguration () const {
      SP<TranslateOrScaleConfiguration> result =
	NEW (TranslateOrScaleConfiguration ());
      // Want a smaller value than the default, since this is per-pixel and the
      // default is good for one IntelliMouse wheel bump.
      result->setScale (1.001);
      return &*result;
    }
    SP<Configurable> makeIt (const SP<Configuration> c) const {
      SP<TranslateOrScaleConfiguration> conf =
	dynamic_cast <TranslateOrScaleConfiguration *> (&*c);
      assert (conf);
      SP<const KeyboardMouseEvent> e =
	dynamic_cast <const KeyboardMouseEvent *>
	(&*conf->getTopLevel()->getEvent());
      assert (e);
      if (conf->m_firstTime) {
	// The first time.
	// NOTE that we are replacing conf with a copy of itself here.  After
	// this, changes to conf will communicate with our next invocation
	// during this grab, but won't affect the initial values next time this
	// command is run from the event dispatcher.  It is particularly
	// important not to change the initial value of the m_firstTime flag.
	conf = NEW (TranslateOrScaleConfiguration (*conf));
	conf->m_firstTime = false;
	conf->m_x = e->getX();
	conf->m_y = e->getY();
	SP<RecursiveSlotValue> rsv = NEW (RecursiveSlotValue ("Action", this));
	rsv->setConfiguration (&*conf);
	conf->getTopLevel()->grab (rsv);
      }
      SP<Canvas> vo = conf->getTopLevel()->getCanvas();
      const int newX = e->getX();
      const int newY = e->getY();
      const int dx = newX - conf->m_x;
      const int dy = newY - conf->m_y;
      if (e->isButtonDown (conf->getShift ())) {
	// Scale.
	// FIXME Scale around the (x,y) coordinates of the mouse.
	// Now I have code for that, so no excuse for it to still be wrong!
	vo->setScale (vo->getScale () * pow (conf->getScale (), dy));
      } else {
	// Translate.
	Vec3 OScreen = vo->transformWorldToLHScreen (vo->getOrigin ());
	// Checking that we got the sign right -- if the mouse moves up and to
	// the right, then we want the scene to move up and to the right, which
	// means the origin moves down and to the left. 
	OScreen -= Vec3 (dx, dy, 0);
	vo->setOrigin (vo->transformLHScreenToWorld (OScreen));
      }
      conf->m_x = newX;
      conf->m_y = newY;
      // No errors possible.
      return NEW (Action ());
    }
  };
  const bool useless =
  (FactoryTable::store ("KeyboardMouseAction", NEW (TranslateOrScale ())),
   true);
} // namespace
