// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __TransformConfiguration_h__
#include "TransformConfiguration.h"
#endif

#ifndef __Vec3_h__
#include "Vec3.h"
#endif

#ifndef __String_h__
#include "String.h"
#endif

#ifndef __Vec3SlotValue_h__
#include "Vec3SlotValue.h"
#endif

#ifndef __QuaternionSlotValue_h__
#include "QuaternionSlotValue.h"
#endif

#ifndef __Matrix4_h__
#include "Matrix4.h"
#endif

#ifndef __Quaternion_h__
#include "Quaternion.h"
#endif

#ifndef __TopLevel_h__
#include "TopLevel.h"
#endif

#ifndef __SP_h__
#include "SP.h"
#endif

#ifndef __Canvas_h__
#include "Canvas.h"
#endif

#ifndef __CursorBallManager_h__
#include "CursorBallManager.h"
#endif

#ifndef __SpaceOrbEvent_h__
#include "SpaceOrbEvent.h"
#endif

#ifndef __CursorBall_h__
#include "CursorBall.h"
#endif

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

namespace {
  String translationName () {
    return "Translation";
  }
  String centerName () {
    return "Center of rotation";
  }
  String rotationName () {
    return "Rotation";
  }
}

TransformConfiguration::TransformConfiguration () {
  initializeSlot (translationName (), NEW (Vec3SlotValue (Vec3 (0,0,0))));
  initializeSlot (centerName (), NEW (Vec3SlotValue (Vec3 (0, 0, 0))));
  initializeSlot (rotationName (), NEW (QuaternionSlotValue (Quaternion ())));
}

Vec3 TransformConfiguration::getTranslation () const {
  return Vec3SlotValue::getSlot (this, translationName());
}

void TransformConfiguration::setTranslation (const Vec3 &v) {
  Vec3SlotValue::setSlot (this, translationName(), v);
}

Vec3 TransformConfiguration::getCenter () const {
  return Vec3SlotValue::getSlot (this, centerName ());
}

void TransformConfiguration::setCenter (const Vec3 &v) {
  Vec3SlotValue::setSlot (this, centerName(), v);
}

Quaternion TransformConfiguration::getRotation () const {
  return QuaternionSlotValue::getSlot (this, rotationName ());
}

void TransformConfiguration::setRotation (const Quaternion &v) {
  QuaternionSlotValue::setSlot (this, rotationName(), v);
}

Matrix4 TransformConfiguration::getMatrix () {
  Vec3 center = getCenter ();
  // Read this from right to left.
  return Matrix4::translate (getTranslation ()) * Matrix4::translate (center)
    * getRotation () * Matrix4::translate (-center);
}

void TransformConfiguration::parseOrbEvent (const SpaceOrbEvent *e) {
  SP<TopLevel> top = getTopLevel ();
  assert (top);
  SP<Canvas> canvas = top->getCanvas ();
  setTranslation (e->getTranslation () * canvas->getScale ());
  setCenter (canvas->transformWorldToRHScreen
	     (CursorBallManager::getCursorBall
	      (top->getSceneGraph ())->getPosition ()));
  setRotation (e->getRotation());
}

bool TransformConfiguration::hasTheSameTypeAs (const Configuration *c) const {
  return 0 != dynamic_cast <const TransformConfiguration *> (c);
}

SP<Configuration> TransformConfiguration::copy () const {
  return NEW (TransformConfiguration (*this));
}
