// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

// For strerror.
#ifndef  __string_h__
#include <string.h>
#define __string_h__
#endif

// For errno.
#ifndef __errno_h__
#include <errno.h>
#define __errno_h__
#endif

#ifndef __TypedFactory_h__
#include "TypedFactory.h"
#endif

#ifndef __SaveMovieConfiguration_h__
#include "SaveMovieConfiguration.h"
#endif

#ifndef __FactoryTable_h__
#include "FactoryTable.h"
#endif

#ifndef __SP_h__
#include "SP.h"
#endif

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

#ifndef __Action_h__
#include "Action.h"
#endif

#ifndef __NullObject_h__
#include "NullObject.h"
#endif

extern "C" {
#ifndef __ppm_h__
#include <ppm.h>
#define __ppm_h__
#endif
}

// Need stdio.h because the ppm library uses FILE *'s.
#ifndef __stdio_h__
#include <stdio.h>
#define __stdio_h__
#endif

#ifndef __String_h__
#include "String.h"
#endif

#ifndef __myassert_h__
#include "myassert.h"
#endif

#ifndef __Canvas_h__
#include "Canvas.h"
#endif

#ifndef __FloatUtil_h__
#include "FloatUtil.h"
#endif

#ifndef __Color_h__
#include "Color.h"
#endif

#ifndef __SceneGraph_h__
#include "SceneGraph.h"
#endif

#ifndef __TopLevel_h__
#include "TopLevel.h"
#endif

namespace {
  class MovieSaver
    : public NullObject
  {
    String m_fileName;
    mutable int m_frames;
    int m_skip;
    mutable int m_thisSkip;
    FILE *m_file;
    // Next one is a nonempty string if an error occurred while opening.  Other
    // errors just generate a message.
    String m_error;
    // The width and height of the last frame we wrote, so we can report it at
    // the end. 
    mutable int m_width;
    mutable int m_height;
  public:
    MovieSaver (const String &fileName, int frames, int skip)
      : m_fileName (fileName), m_frames (frames), m_skip (skip),
	m_thisSkip (0), m_width (-1), m_height (-1)
    {
      assert (frames > 0);
      m_file = fopen (&*fileName, "w");
      if (!m_file) {
	m_error = (String ("Could not open ")+fileName+" for writing: "
		   +strerror (errno));
      }
    }

    String getError () const {
      return m_error;
    }

    bool hasAnimationEver () const {
      return true;
    }

    // By returning false, if the scene isn't moving, we don't record any
    // frames.  This is what we probably want.
    bool hasAnimationNow (const SceneGraph *sg, int index) const {
      (void) sg; (void) index;
      return false;
    }
    
    template <class Pixel> void animateFrameTemplate
    (Canvas *canvas, SceneGraph *sg, int index) const
    {
      if (m_file && m_frames > 0) {
	if (0 == m_thisSkip) {
	  m_thisSkip = m_skip;
	  m_frames --;
	  const int rows = canvas->getHeight ();
	  m_height = rows;
	  const int cols = canvas->getWidth ();
	  m_width = cols;
	  // No point in more than 8 bits per color.
	  const pixval maxval = FloatUtil::min (PPM_MAXMAXVAL, 255);
	  // Allocate and free the array every time.  This way, if the window
	  // is resized while the movie is being saved, the right thing happens
	  // without any special cases.
	  pixel** array = ppm_allocarray (cols, rows);
	  assert (sizeof(Pixel) == canvas->pixelBytes ());
	  Pixel *frameBuffer = (Pixel *) (canvas->getFrameBuffer ());
	  const int pixelsPerLine = canvas->getFrameBufferPixelsPerLine ();
	  for (int row = 0; row < rows; row++) {
	    pixel *toPix = array [row];
	    Pixel *fromPix = ((Pixel *) frameBuffer) + row * pixelsPerLine;
	    for (int col = 0; col < cols; col++) {
	      Color c = canvas->unPackColor (*fromPix);
	      fromPix++;
	      PPM_ASSIGN (*toPix,
			  (int)(c[0]*maxval), (int)(c[1]*maxval),
			  (int)(c[2]*maxval));
	      toPix++;
	    }
	  }
	  // No defined way to discover wither ppm_writeppm got an error while
	  // writing, such as a full disk.  Bummer.
	  ppm_writeppm (m_file, array, cols, rows, maxval, false);
	  ppm_freearray (array, canvas->getHeight ());
	  // Flush after writing.  The hope is that this will give us a valid
	  // movie if the program is aborted.
	  fflush (m_file);
	} else {
	  m_thisSkip --;
	}
      } else {
	// If I have no file, or no more frames to write, then delete myself.
	sg->deleteObject (index);
	// this may be deallocated at this point, so no code allowed after this
	// point.
      }
    }
    
    void animateFrame (Canvas *canvas, SceneGraph *sg, int index) const {
      if (4 == canvas->pixelBytes ()) {
	animateFrameTemplate <Canvas::Pixel32> (canvas, sg, index);
      } else if (2 == canvas->pixelBytes ()) {
	animateFrameTemplate <Canvas::Pixel16> (canvas, sg, index);
      } else {
	die (String ("Can't write a movie at ") + canvas->pixelBytes() +
	     String (" bytes per pixel"));
      }
    }
    
    ~MovieSaver () {
      ref ();
      if (m_file) {
	if (fclose (m_file)) {
	  message (String ("Could not close ")+m_fileName+" after writing: "+
		   strerror (errno));
	}
	m_file = 0;
	if (-1 == m_width) {
	  assert (-1 == m_height);
	  message ("Wrote no frames to "+m_fileName+".");
	} else {
	  message ("Finished writing to "+m_fileName+".  Last frame was "+
		   (String) m_width+" by "+(String) m_height+".");
	}
      }
      deref ();
    }
  };
  class SaveMovie
    : public TypedFactory <SaveMovieConfiguration, Action>
  {
  public:
    SaveMovie ()
      : TypedFactory <SaveMovieConfiguration, Action> ("SaveMovie")
    {}
    SP<SaveMovieConfiguration> typedDefaultConfiguration () const {
      return NEW (SaveMovieConfiguration ());
    }
    SP<Action> makeIt (SaveMovieConfiguration *conf) const {
      SP<Action> result = NEW (Action ());
      SP<MovieSaver> ms = NEW (MovieSaver (conf->getFileName (),
					   conf->getFrames (),
					   conf->getSkip ()));
      if (0 != ms->getError().size()) {
	result->setProblem (ms->getError ());
      } else {
	conf->getTopLevel()->getSceneGraph()->addObject (ms, 0);
      }
      return result;
    }
  };
  bool useless = (FactoryTable::store ("Action", NEW (SaveMovie ())), true);
}
