/* Welcome2L - Welcome to Linux, ANSI boot logo
 * Copyright (c)1998-1999 Jean-Marc Jacquet <jm@littleigloo.org>  
 * Little Igloo Org  http://www.LittleIgloo.org
 *
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/utsname.h>
#include <sys/stat.h>
#include <locale.h>
#include <time.h>

#include "version.h"
#include "ansi.h"
#include "ansi/welcome1.h"
#include "ansi/welcome2.h"
#include "ansi/xmas.h"


#ifndef FALSE
#define FALSE (0)
#endif
#ifndef TRUE
#define TRUE (!FALSE)
#endif
#define NOTSEEN 2

#define NODENAME uname_buf.nodename
#define RELEASE uname_buf.release
#define MACHINE uname_buf.machine
#define TTY ttyname(0)+5



static struct utsname uname_buf;
static char MEMORY[15];
static char TIME[30];
static char WMESG[25];
static unsigned char want_getty_infos = FALSE;
static unsigned char SCREEN = 1;
static unsigned char last_color = FALSE;
static char WANT_NEWLINE = TRUE;
static int NEW_YEAR = FALSE;
static char XMAS_AUTO = FALSE;
char WSCREEN[10000];



static void
draw_ansi_screen (unsigned char *ansi_screen, int ansi_screen_length, int ansi_screen_width)
{
  register int __f;
  int this_width = ansi_screen_width << 1;

  for (__f = 0; __f < ansi_screen_length; __f += 2)
    {
      if (__f)
	if ((__f % this_width) == 0)
	  ANSI_do_char ('\n');
      ANSI_do_color (ansi_screen[__f + 1]);
      ANSI_do_char (ansi_screen[__f]);
    }
}



static void
draw_welcome_screen (unsigned char *screen, int screen_length, int screen_width,
		     char color1, char color2, char color3,
		     char xnode, char ynode, char xtime, char ytime,
		     char xtty, char ytty,
		     char xwmsg, char ywmsg, char xrel, char yrel,
		     char xmach, char ymach, char xmem, char ymem)
{
  ANSI_screen_init ();

  draw_ansi_screen (screen, screen_length, screen_width);
  ANSI_save_pos ();
  ANSI_do_color (color1);
  ANSI_printxy (xnode, ynode, NODENAME);
  ANSI_do_color (color2);
  if (xtty + ytty)
    {
      if (isatty (0))
	{
	  ANSI_printxy (xtty, ytty, "Logging on ");
	  if (want_getty_infos)
	    ANSI_print ("\\l");
	  else
	    ANSI_print (TTY);
	}
    }

  ANSI_do_color (color3);
  if (xwmsg + ywmsg)
    {
      ANSI_printxy (xwmsg, ywmsg, WMESG);
    }

  if (want_getty_infos)
    {
      if (xtime + ytime)
	{
	  ANSI_do_color (color2);
	  ANSI_printxy (xtime, ytime, "\\d - \\t");
	}
      if (xrel + yrel)
	{
	  ANSI_do_color (color3);
	  ANSI_printxy (xrel, yrel, "\\r");
	}
      if (xmach + ymach)
	{
	  ANSI_do_color (color3);
	  ANSI_printxy (xmach, ymach, "\\m");
	}
    }
  else
    {
      if (xtime + ytime)
	{
	  ANSI_do_color (color2);
	  ANSI_printxy (xtime, ytime, TIME);
	}
      if (xrel + yrel)
	{
	  ANSI_do_color (color3);
	  ANSI_printxy (xrel, yrel, RELEASE);
	}
      if (xmach + ymach)
	{
	  ANSI_do_color (color3);
	  ANSI_printxy (xmach, ymach, MACHINE);
	}
    }

  if (xmem + ymem)
    ANSI_printxy (xmem, ymem, MEMORY);
}



static void
draw_this_screen (unsigned char this_screen)
{
  switch (this_screen)
    {
    default:
    case 1:
      {
	char buf[80];
	draw_welcome_screen (WELCOME1, WELCOME1_LENGTH, WELCOME1_WIDTH,
			     0x08, 0x08, 0x08,
			     18, 11, 7, 12, 7, 13,
			     7, 15, 16, 16, 16, 17, 16, 18);
	ANSI_do_color (0x19);
	sprintf (buf, "%s V%s (c)LittleIgloo.Org", PROG_NAME, PROG_VERSION);
	ANSI_printxy (0, 22, buf);
      }
      break;

    case 2:
      {
	char buf[80];
	draw_welcome_screen (WELCOME2, WELCOME2_LENGTH, WELCOME2_WIDTH,
			     0x09, 0x01, 0x09,
			     18, 11, 7, 12, 7, 13,
			     7, 15, 16, 16, 16, 17, 16, 18);
	ANSI_do_color (0x19);
	sprintf (buf, "%s %s (c)LittleIgloo.Org", PROG_NAME, PROG_VERSION);
	ANSI_printxy (0, 22, buf);
      }
      break;


    case 3:
      draw_welcome_screen (XMAS, XMAS_LENGTH, XMAS_WIDTH,
			   0x04, 0x04, 0x04,
			   15, 12, 0, 0, 0, 0,
			   0, 0, 24, 14, 24, 15, 24, 16);
      ANSI_do_color (0x04);
      ANSI_printxy (18, 18, "We wish you a");

      if (!NEW_YEAR)
	{
	  ANSI_do_color (0x8F);
	  ANSI_printxy (16, 19, "MERRY CHRISTMAS");
	}
      else
	{
	  char what_year[50];
	  sprintf (what_year, "HAPPY NEW YEAR %i", NEW_YEAR);
	  ANSI_do_char ('n');
	  ANSI_do_color (0x8F);
	  ANSI_printxy (13, 19, what_year);
	}
      break;
    }

  ANSI_restore_pos ();
  ANSI_screen_close ();
  printf (WSCREEN);
}



static void
prog_infos ()
{
  printf ("\033[H\033[J");
  printf ("%s%s V%s By %s\n%sCopyright (c)%s %s\n%sCompiled on %s %s by %s@%s (%s)\n%s\n",
	  "\033[1;33m", PROG_NAME, PROG_VERSION, PROG_AUTHOR,
	  "\033[36m", VERSION_DATE, LITTLEIGLOO,
	  "\033[0;36m", COMPILE_DATE, COMPILE_TIME, COMPILE_BY, COMPILE_HOST, CC_VERSION,
	  "\033[0;37mDistributed under the terms of the GNU General Public License.\033[m\n");
}



static void
help_usage (void)
{
  prog_infos ();
  printf ("ERROR: Unrecognized Argument.\nType : %s -help for usage information.\n\n", PROG_NAME);
  exit (1);
}


static void
print_option (char *this_option, char *this_option_arg, char *this_message)
{
  printf ("\033[0;36m  \033[1m-%s\033[1;37m%s\n\033[0;36m  %s\n",
	  this_option, this_option_arg, this_message);
}

static void
usage (void)
{
  prog_infos ();
  printf ("\033[1;36m");

  printf ("OPTIONS:\n\033[0;36m");
  print_option ("scr", "xx", "will display the screen number xx");
  print_option ("msg", "\"text of message\"", "will show the msg given as the distribution info");
  print_option ("getty", "", "will display more accurate information when launched at boot time\n  (via an init script).");
  print_option ("nolf", "", "will not add an empty line to the ansi screen produced.");
  print_option ("lcol", "xx", "will change the prompt color to the color number xx");

  printf ("\033[0;37m  COLOR NUMBERS ARE: %s\n", "\033[0;34m1 \033[32m2 \033[36m3 \033[31m4 \033[35m5 \033[33m6 \033[37m7 \033[1;30m8 \033[34m9 \033[32m10 \033[36m11 \033[31m12 \033[35m13 \033[33m14 \033[37m15");

  printf ("\n\033[1;36mEXAMPLE:\n\033[0;36m");
  printf ("  ./Welcome2L -getty -msg\"RedHat Linux\" -lcol14");
  printf ("\n\n\033[1;36mHTTP://WWW.LITTLEIGLOO.ORG\033[0;36m");

  printf ("\033[m\n\n");
  exit (0);
}



int
main (int argc, char *argv[])
{
  struct stat stat_buf;
  time_t timep;
  struct tm *timeptr;
  char SEE_ARG = FALSE;

  WMESG[0] = 0;

  if (argc > 1)
    SEE_ARG = NOTSEEN;
  while (argc--)
    {
      if (strstr (argv[argc], "-help"))
	usage ();
      if (strstr (argv[argc], "-msg"))
	{
	  strncpy (WMESG, argv[argc] + 4, sizeof (WMESG));
	  SEE_ARG = TRUE;
	}
      if (strstr (argv[argc], "-lcol"))
	{
	  last_color = atoi (argv[argc] + 5);
	  if (last_color > 15)
	    last_color = 15;
	  SEE_ARG = TRUE;
	}
      if (strstr (argv[argc], "-nolf"))
	{
	  WANT_NEWLINE = FALSE;
	  SEE_ARG = TRUE;
	}
      if (strstr (argv[argc], "-getty"))
	{
	  want_getty_infos = TRUE;
	  SEE_ARG = TRUE;
	}

      if (strstr (argv[argc], "-scr"))
	{
	  SCREEN = atoi (argv[argc] + 4);
	  SEE_ARG = TRUE;
	}

      if (strstr (argv[argc], "-xmas"))
	{
	  SCREEN = 2;
	  SEE_ARG = TRUE;
	}

      if (strstr (argv[argc], "-xmasauto"))
	{
	  SCREEN = 2;
	  SEE_ARG = TRUE;
	}

    }
  if (SEE_ARG == NOTSEEN)
    help_usage ();


  setlocale (LC_ALL, "");

  uname (&uname_buf);

  if (strlen (NODENAME) > 15)
    *(strchr (NODENAME, '.')) = 0;

  timep = time (NULL);
  timeptr = localtime (&timep);
  strftime (TIME, sizeof (TIME), "%a %d %b %Y - %X", timeptr);

  if ((timeptr->tm_mon < 2) || (timeptr->tm_mon == 11 && timeptr->tm_mday > 25))
    NEW_YEAR = timeptr->tm_year + 1900 + (timeptr->tm_mon == 11 && timeptr->tm_mday > 25);

  if ((timeptr->tm_mon != 0) && (timeptr->tm_mon != 11) && (SCREEN == 2) && (XMAS_AUTO))
    SCREEN = 1;			/* Ross Younger's patch - no Xmas decorations outside of Jan/Dec. (enable it with -xmasauto) */


  stat ("/proc/kcore", &stat_buf);
  sprintf (MEMORY, "%ld Kb", stat_buf.st_size / 1024);

  draw_this_screen (SCREEN);

  if (last_color)
    {
      char attr = 0;
      if (last_color > 8)
	{
	  attr++;
	  last_color -= 8;
	}
      printf ("\033[%i;3%im", attr, ANSI_col_convert (last_color));
    }

  if (WANT_NEWLINE)
    printf ("\n");

  return 0;
}
