/*
 * task.c --
 *
 *      Provides task control services for the MTOS-UX simulator.
 *
 * Copyright (C) 1995, 1996 Daniel Wu.
 * Distributed under the terms of the GNU Library General Public
 * License.
 *
 * This file is part of SimTos.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation (version 2).
 *
 * This library is distributed "AS IS" in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with Pthreads; see the file COPYING.  If not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA
 * 02139, USA.
 *
 * Author: Daniel Wu (dwu@linus.socs.uts.edu.au)
 *
 */


#include <stdlib.h>
#include <math.h>
#include <assert.h>
#include "os_mtosux.h"
#include "symtab.h"
#include "task.h"
#include "os_timer.h"


/*
 *--------------------------------------------------------------
 *
 * crtsk --
 *
 *	Creates a MTOS task.
 *
 * Results:
 *      If the function is succesful, returns the identifier of
 *      the created task. Otherwise, returns DUPTSK if the task
 *      already existed; BADPRM if the parameters are invalid;
 *      or QUEFUL if there is insufficient resources to create
 *      the task.
 *
 *--------------------------------------------------------------
 */

uxid_t	crtsk(struct tcd* tcdptr)
{
    OS_TSK_TYPE* tsk_p;    

    /*
     * Detect and report invalid parameters
     */
    if (tcdptr == NULL)
        return BADPRM;        

    if (tcdptr->lang != C)
        return BADLNG;

    if (tcdptr->apct > DAY)
        return BADTIM;
    
    /*
     * Check to see if task already exists
     */
    if (gettid(tcdptr->key) != BADPRM)
    {
        return DUPTSK;
    }
    
    /*
     * Create a tcb and insert into tasks table
     */
    tsk_p = (OS_TSK_TYPE*) malloc(sizeof(OS_TSK_TYPE));
    if (tsk_p != NULL)
    {
        tsk_p->symtype = sym_task;
        tsk_p->mtos_tcd = (struct tcd*) malloc(sizeof(struct tcd));        
        if (tsk_p->mtos_tcd != NULL)
        {                
            *tsk_p->mtos_tcd = *tcdptr;

            /*
             * Create local event flag group for the task
             */
            if (os_init_efg(&tsk_p->efgs) != NOERR)
            {
                free(tsk_p);

                return QUEFUL;
            }

            pthread_cond_init(&tsk_p->pcond, NULL);        
            pthread_mutex_init(&tsk_p->pmutex, NULL);        
            
            return add_sym(tcdptr->key, tsk_p);
        }
    }

    return QUEFUL;       
}

/*
 *--------------------------------------------------------------
 *
 * dltsk --
 *
 *	Deletes the requesting task.
 *
 * Results:
 *      None.
 *
 *--------------------------------------------------------------
 */

void dltsk(long retarg)
{

    /*
     *	Delete symbol table entries.
     */
    pthread_t    pthread_curr = pthread_self();
    uxid_t       cur_tid      = gettid(0);    

    assert(cur_tid != BADPRM);
    
    /*
     *	Return stack and tcb memory to the OS.
     */
    lock_sym(cur_tid);    
    delete_sym(cur_tid);
    
    /*
     * Terminate the thread
     */
    pthread_detach(&pthread_curr);
    pthread_exit((any_t)retarg);
}

/*
 *--------------------------------------------------------------
 *
 * start --
 *
 *	Starts the given task.
 *
 * Results:
 *      If the function is successful, NOERR is returned.
 *      Otherwise, BADPRM if any invalid parameter is detected;
 *      or QUEFUL if there is insufficient internal resources to
 *      process requests.
 *
 *--------------------------------------------------------------
 */

long start(uxid_t tid, ULONG pty, UPTR arg, ULONG* stabfr, qual_t qual)
{

    OS_TSK_TYPE*   tsk_p;    
    struct tcd*    tcd_ptr;
    int            tsk_eff_pty;
    pthread_attr_t tsk_attr;
    pthread_t      pthread_run;
    pthread_attr_t pthread_attr;          
    int            thread_prio;

    
    /*
     *	Make sure task has been created.
     */
    if ((tsk_p = id2sym(tid)) == NULL)
    {
        *stabfr = BADPRM;
        return BADPRM;
    }      
    
    lock_sym(tid);
    
    tcd_ptr = tsk_p->mtos_tcd;

    pthread_run = pthread_self();    
    pthread_attr_init(&pthread_attr);    
    pthread_getschedattr(pthread_run, &pthread_attr);
    thread_prio = pthread_attr_getprio(&pthread_attr);

    /*
     *	Compute task effective priority.
     */
    switch(PBASIS(pty))
    {
        
      case LRGPTY:
        
          /*
           * LRGPTY:
           *   Use the greater of the task's inherent
           *   priority and the priority of the requesting
           *   task.
           */
          if (pthread_run != NULL && thread_prio >= tcd_ptr->ipr)
          {
              tsk_eff_pty = thread_prio;
          }
          else
          {
              tsk_eff_pty = tcd_ptr->ipr;
          }
          break;
          
      case GVNPTY:
          
          /*
           * GVNPTY:
           *   Use the priority passed to start().
           */
          tsk_eff_pty = PVALUE(pty);
          break;
    
      case CURPTY:
      
          /*
           * CURPTY:
           *   Use the priority of the requesting task.
           */
          tsk_eff_pty = thread_prio;
          break;
    
      case INHPTY:

          /*
           * INHPTY:
           *   Use the task's inherent priority.
           */          
          tsk_eff_pty = tcd_ptr->ipr;
          break;

      default:

          /*
           * If we ever get here, something really bad has
           * happened.
           */
          *stabfr = BADPRM;
          return BADPRM;

    }

    /*
     *	Add the task to the run queue, and swap.
     */

    /*
    ** Start the thread
    */
    pthread_create(&tsk_p->lwp_pcb,
                   NULL,
                   (pthread_func_t)tcd_ptr->ep,
                   arg);

    /*
    ** Set priority of task
    */
    pthread_getschedattr(tsk_p->lwp_pcb, &tsk_attr);    
    pthread_attr_setprio(&tsk_attr, tsk_eff_pty);
    pthread_setschedattr(tsk_p->lwp_pcb, tsk_attr);

    release_sym(tid);
    
    /*
     *	wait for the thread to complete
     */    
    if (CBASIS(qual) == CTERM)
    {
        pthread_join(tsk_p->lwp_pcb, (any_t*)stabfr);

        return (long)*stabfr;
    }    
    
    return NOERR;
}


/*
 *--------------------------------------------------------------
 *
 * pause --
 *
 *	Pause for the given timer interval.
 *
 * Results:
 *      If the function is successful, returns NOERR.
 *      Otherwise, BADPRM if any invalid parameter is detected;
 *      or QUEFUL if there is insufficient internal resources to
 *      process the request.
 *
 *--------------------------------------------------------------
 */

long os_pause(pthread_cond_t* cond, pthread_mutex_t* mutex, qual_t interval)
{
    int             unit     = interval / MS;
    int             val      = interval % MS; 
    long            retval   = 0;
    long            time;
    struct timeval  curtime;
    struct timespec abstime;

    if (interval > 0)
    {        
        time = val * pow(10, unit-1) * 1000;
        gettimeofday(&curtime, NULL);
        curtime.tv_sec += time / 1000000L;
        curtime.tv_usec += time % 1000000L;
        if (curtime.tv_usec >= 1000000L)
        {
            curtime.tv_sec++;
            curtime.tv_usec = curtime.tv_usec - 1000000L;
        }
        TIMEVAL_TO_TIMESPEC(&curtime, &abstime);
        
        pthread_mutex_lock(mutex);    
        if (pthread_cond_timedwait(cond, mutex, &abstime) != -1)
            retval = -1;
        pthread_mutex_unlock(mutex);
    }
    
    return retval;    
}


long ux_pause(qual_t interval)
{    
    uxid_t          cur_tid  = gettid(0);
    OS_TSK_TYPE*    tsk_p    = id2sym(cur_tid);
    long            retval   = NOERR;
    
    assert(cur_tid != BADPRM);

    if (os_pause(&tsk_p->pcond, &tsk_p->pmutex, interval) == -1)
        retval = TIMCAN;
    
#ifdef DEBUG
    /* debugging */
{
    struct timeval  newtime;
    gettimeofday(&newtime, NULL);
    
    printf("## pthread=%p, tv_sec diff=%ld, tv_usec diff=%ld\n",
     	pthread_self(),
        ((long)newtime.tv_sec-(long)curtime.tv_sec),
        ((long)newtime.tv_usec-(long)curtime.tv_usec));
}
#endif

    return retval;
}

/*
 *--------------------------------------------------------------
 *
 * gettid --
 *
 *      Retrieve a tid for a given key
 *
 * Results:
 *      If the function is successful, returns the identifier of
 *      the task, or BADPRM if not.
 *
 *--------------------------------------------------------------
 */

uxid_t gettid(key_t key)
{
    OS_TSK_TYPE* tsk_p;        
    pthread_t    this_thread;
    uxid_t       i      = 0;
    uxid_t       retid  = -1;
    
    if (key == 0L)
    {
        for (i=get_symtab_base(); i<get_symtab_size(); i++)
        {
            if ((tsk_p = id2sym(i)) != NULL)
            {
                lock_sym(i);

                if (tsk_p->symtype == sym_task)
                {                
                    this_thread = pthread_self();            
                    if (pthread_equal(tsk_p->lwp_pcb, this_thread))
                        retid = i;
                }

                release_sym(i);

                if (retid != -1)
                    return retid;                
            }            
        }
        
        return BADPRM;
    }
    
    return key2id(key);        
}

/*
 *--------------------------------------------------------------
 *
 * getkey --
 *
 *      Retrieve a key for a given tid
 *
 * Results:
 *      If the function is successful, returns the key of
 *      the task, or BADPRM if not.
 *
 *--------------------------------------------------------------
 */

key_t getkey(uxid_t tid)
{
    return id2key(tid);    
}


void exit(long	retarg)
{
    pthread_t    pthread_run = pthread_self(); 
    uxid_t       cur_tid     = gettid(0);    
    OS_TSK_TYPE* tsk_p       = id2sym(cur_tid);

    if (tsk_p != NULL)
        tsk_p->lwp_pcb = NULL; /* disociate the mtos TCD with the thread */
    pthread_detach(&pthread_run);
    pthread_exit((any_t)retarg);
}


long getdad(		/* get address of data segments */
	  UPTR		buf	/* addr of buffer to receive seg addresses */
	)
{
    uxid_t       cur_tid = gettid(0L);
    OS_TSK_TYPE* tsk_p = id2sym(cur_tid);


    lock_sym(cur_tid);    
    *((UPTR*)buf) = tsk_p->mtos_tcd->ida;
    release_sym(cur_tid);    

    return NOERR;    
}

UWORD	
	setpty(		/* set priority of task */
	  uxid_t	tid,	/* task identifier */
	  ULONG		basis,	/* basis: USEVAL or ADDVAL */
	  long		value	/* value to be used or added */
	)
{
    pthread_t      tsk_thread = NULL;
    OS_TSK_TYPE*   tsk_p      = NULL;
    pthread_attr_t tsk_attr;    
    int            tsk_pty;
    int            curr_pty;

    /*
     *  Get the task identifier of the requesting task
     */
    if (tid == 0)
    {
        /*
         * The request is called within inient
         */
        if ((tid = gettid(0L)) == BADPRM)
        {
            fprintf(stderr, "check setpty error\n");
            tsk_thread = pthread_self();            
        }        
    }

    if (tsk_thread == NULL)
    {
        
        /*
         *  Make sure task has been created.
         */
        if ((tsk_p = id2sym(tid)) == NULL)
        {
            return BADPRM;
        }

        lock_sym(tid);        
        tsk_thread = tsk_p->lwp_pcb;        
    }
    
    pthread_attr_init(&tsk_attr);
    pthread_getschedattr(tsk_thread, &tsk_attr);    
    curr_pty = pthread_attr_getprio(&tsk_attr);    
    
    switch(basis)
    {
      case USEVAL:
        tsk_pty = value;        
        break;

      case ADDVAL:
        tsk_pty = curr_pty + value;        
        break;

      default:
        if (tsk_p != NULL)
            release_sym(tid);
        return BADPRM;        
    }

    if (tsk_pty < 0)
        tsk_pty = 0;    
    if (tsk_pty > 255)
        tsk_pty = 255;
    
    pthread_attr_setprio(&tsk_attr, tsk_pty);    
    pthread_setschedattr(tsk_thread, tsk_attr);

    if (tsk_p != NULL)
        release_sym(tid);    
    
    return tsk_pty;    
}


long	tstart(		/* transfer start of task */
	  uxid_t	tid	/* task identifier */
	)
{
    uxid_t      cur_tid      = gettid(0L);
    long	stabfd;		/* (dummy) status buffer */

    if (tid == cur_tid)
        return BADPRM;

    /*
    ** Start the gradchild task and wait for its completion
    */
    start(tid, INHPTY, NULL, &stabfd, WAIFIN+CTERM);    

    /*
    ** Terminate current task but not delete it
    */
    exit(stabfd);

    return NOERR;
}

typedef struct {
    uxid_t tid;
    qual_t interval;
} OS_RTSK_TYPE;

static void restart(OS_RTSK_TYPE* arg)
{
    pthread_cond_t  cond;
    pthread_mutex_t mutex;
    long            stabfr;
    
    pthread_cond_init(&cond, NULL);        
    pthread_mutex_init(&mutex, NULL);
    os_pause(&cond, &mutex, arg->interval);    
    pthread_mutex_destroy(&mutex);
    pthread_cond_destroy(&cond);
    
    start(arg->tid, INHPTY, NULL, &stabfr, WAIFIN+CTERM);
    free(arg);
}

void    trmrst(         /* terminate task with restart after given interval */
          long          retarg, /* return argument */
          qual_t        intvlb  /* time interval + STRTIM or TRMTIM */
        )
{
    qual_t         pautim = intvlb & ~TRMTIM; /* mask the high bit */
    OS_RTSK_TYPE*  rtsk = malloc(sizeof(OS_RTSK_TYPE));
    pthread_t      restart_thread;
    pthread_attr_t attr;    

    if (pautim < MS || pautim > 8*MS)
        pautim = 0;

    rtsk->tid = gettid(0);
    rtsk->interval = pautim;
    
    pthread_attr_init(&attr);
    pthread_attr_setprio(&attr, 255);
    pthread_create(&restart_thread, &attr, (pthread_func_t)restart, rtsk);
    exit(retarg);
}

long	canpau(		/* cancel pause */
	  uxid_t	tid	/* task identifier */
	)
{
    OS_TSK_TYPE* tsk_p = id2sym(tid);

    if (tsk_p == NULL)
        return BADPRM;

    pthread_cond_signal(&tsk_p->pcond);

    return NOERR;    
}
