/*
 * msb.c --
 *
 *      Provides message buffer services for the MTOS-UX simulator.
 *
 * Copyright (C) 1995, 1996 Daniel Wu.
 * Distributed under the terms of the GNU Library General Public
 * License.
 *
 * This file is part of SimTos.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation (version 2).
 *
 * This library is distributed "AS IS" in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with Pthreads; see the file COPYING.  If not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA
 * 02139, USA.
 *
 * Author: Daniel Wu (dwu@linus.socs.uts.edu.au)
 *
 */

#include <stdlib.h>
#include "msb.h"


/*
 *--------------------------------------------------------------
 *
 * crmsb --
 *
 *	Create message buffer queue
 *
 * Results:
 *      The identifier of the newly created message buffer.
 *
 *--------------------------------------------------------------
 */

uxid_t crmsb(key_t key,	ULONG attr)
{

    /*
     *  Only MSBGBL is supported
     */
    if (GLS(attr) != MSBGBL)
    {
	/*
	 * crmsb: only MSBGBL supported
	 */
        
	return BADPRM;
    }
    else
    {	
	if ((attr & ~MSBGBL) < MIN_SIZE && (attr & ~MSBGBL) > MAX_SIZE)
	{
	    return BADPRM;
	}
	else
	{
	    int msgq_size = NUMMSB(attr);
	    
	    /*
	    ** Default value for the message queue length
	    */
	    if (msgq_size == 0)
		msgq_size = 64;
	    
            return add_msb(key, msgq_size);
        }            
    }
}


/*
 *--------------------------------------------------------------
 *
 * putmsb --
 *
 *	Puts a message into the beginning of a message buffer.
 *
 * Results:
 *      Returns NOERR if successful, otherwise QUEFUL.
 *
 *--------------------------------------------------------------
 */

long putmsb(uxid_t msbid, UPTR msg)
{
    return send_msg(msbid, msg, FALSE);
}


/*
 *--------------------------------------------------------------
 *
 * putmse --
 *
 *	Puts a message into the end of a message buffer queue.
 *
 * Results:
 *      Returns NOERR if successful, otherwise QUEFUL.
 *
 *--------------------------------------------------------------
 */

long putmse(uxid_t msbid, UPTR msg)
{
    return send_msg(msbid, msg, TRUE);
}


/*
 *--------------------------------------------------------------
 *
 * dlmsb --
 *
 *	Deletes a message buffer.
 *
 * Results:
 *      Returns NOERR if successful, otherwise BADPRM.
 *
 *--------------------------------------------------------------
 */

long dlmsb(uxid_t msbid)
{
    OS_MSB_TYPE* msb_p = id2sym(msbid);

    if (msb_p == NULL)
	return BADPRM;
    
    pthread_cond_destroy(&msb_p->msg_cond);

    return delete_msb(msbid);    
}


/*
 *--------------------------------------------------------------
 *
 * getmsb --
 *
 *	Get a message from a message buffer.
 *
 * Results:
 *      The identifier of the message buffer which corresponds
 *      to the given key.
 *
 *--------------------------------------------------------------
 */

uxid_t getmsb(key_t key)
{
    return key2id(key);    
}

/*
 *--------------------------------------------------------------
 *
 * getmsn --
 *
 *	Get a message from a message buffer.
 *
 * Results:
 *      MBEOF is no message is available in the buffer.
 *      Otherwise, NOERR.
 *
 *--------------------------------------------------------------
 */

long getmsn(uxid_t msbid, UPTR dstadr)
{
    OS_MSB_TYPE* msb_p = id2sym(msbid);

    if (msb_p == NULL || dstadr == NULL)
        return BADPRM;

    lock_sym(msbid);    
    if (msb_p->queue_len > 0)
    {
        *((void**)dstadr) = msb_p->msg_buf[0];
        if (--msb_p->queue_len > 0)
        {
            memcpy(msb_p->msg_buf,
                   &msb_p->msg_buf[1],
                   msb_p->queue_len*sizeof(UPTR));
        }
        msb_p->msg_buf[msb_p->queue_len] = NULL;	
        release_sym(msbid);    

	if (msb_p->queue_len == 0 &&
	    msb_p->status == MSB_DELETED)
	{
	    free(msb_p->msg_buf);
	    
	    return delete_sym(msbid);    
	}

        return NOERR;        
    }
    release_sym(msbid);    
    
    return MBEOF;
}

/*
 *--------------------------------------------------------------
 *
 * getmsw --
 *
 *	Get a message from a message buffer.
 *
 * Results:
 *      NOERR if successful, otherwise BADPRM.
 *
 *--------------------------------------------------------------
 */

long getmsw(uxid_t msbid, UPTR dstadr)
{
    OS_MSB_TYPE*     msb_p      = id2sym(msbid);

    if (msb_p == NULL || dstadr == NULL)
        return BADPRM;

    if (msb_p->queue_len == 0)
    {
       pthread_mutex_t  mutex;

       pthread_mutex_init(&mutex, NULL);
       pthread_mutex_lock(&mutex);
       pthread_cond_wait(&msb_p->msg_cond, &mutex);
    }
    
    return getmsn(msbid, dstadr);
}

/*
 *--------------------------------------------------------------
 *
 * add_msb --
 *
 *	Create a message buffer.
 *
 * Results:
 *      The identifier of the message buffer if successful.
 *      Otheriwse, QUEFUL.
 *
 *--------------------------------------------------------------
 */

uxid_t add_msb(key_t key, int sz)
{
    OS_MSB_TYPE*        msb_p;    
    uxid_t              retval;
        
    if ((retval = getmsb(key)) != BADPRM)
        return retval;                    

    msb_p = (OS_MSB_TYPE*) malloc(sizeof(OS_MSB_TYPE));
    msb_p->symtype = sym_msb;
    msb_p->max_queue_sz = sz;
    msb_p->queue_len = 0;
    msb_p->status = MSB_NORMAL;
    msb_p->msg_buf = (UPTR*) malloc(msb_p->max_queue_sz*sizeof(UPTR));
    if (pthread_cond_init(&msb_p->msg_cond, NULL) == 0)
    {        
       retval = add_sym(key, msb_p);
    }
    else
    {
       free(msb_p); 
       retval = QUEFUL;    
    }
    
    return retval;
}

/*
 *--------------------------------------------------------------
 *
 * add_msb --
 *
 *	Delete a message buffer.
 *
 * Results:
 *      NOERR if successful, otheriwse BADPRM.
 *
 *--------------------------------------------------------------
 */

long delete_msb(uxid_t msbid)
{
    OS_MSB_TYPE* msb_p = id2sym(msbid);    

    if (msb_p == NULL)
	return BADPRM;

    if (msb_p->queue_len > 0)
    {
	/*
	 * Setup delete pending
	 */
	msb_p->status = MSB_DELETED;
	
	return NOERR;
    }
    
    free(msb_p->msg_buf);
    
    return delete_sym(msbid);    
}

/*
 *--------------------------------------------------------------
 *
 * send_msg --
 *
 *	Delete a message buffer.
 *
 * Results:
 *      NOERR if successful, otheriwse QUEFUL.
 *
 *--------------------------------------------------------------
 */

long send_msg(uxid_t msbid, UPTR msg, int at_end)
{
    OS_MSB_TYPE* msb_p = id2sym(msbid);
    UPTR         tmp_msg_buf[MAX_QUEUE_SIZE];
    
    if (msb_p == NULL)
        return BADPRM;
    
    if (msb_p->queue_len == msb_p->max_queue_sz)
        return QUEFUL;

    lock_sym(msbid);    
    if (!at_end)
    {
        if (msb_p->queue_len > 0)
        {            
            memcpy(tmp_msg_buf,
                   msb_p->msg_buf,
                   msb_p->queue_len*sizeof(UPTR));
        }
        msb_p->msg_buf[0] = msg;
        if (msb_p->queue_len > 0)
        {            
            memcpy(&msb_p->msg_buf[1],
                   tmp_msg_buf,
                   msb_p->queue_len*sizeof(UPTR));
        }        
        msb_p->queue_len++;        
    }
    else
    {
        msb_p->msg_buf[msb_p->queue_len++] = msg;
    }
    release_sym(msbid);    

    /*
    ** Signal the thread that is waiting for a message
    */
    pthread_cond_signal(&msb_p->msg_cond); 

    /*
    ** Yield current thread to allow blocked threads to run
    */
    pthread_yield(NULL); 
    
    return NOERR;
}
