/*
 * mem.c --
 *
 *      Provides memory management services for the MTOS-UX simulator.
 *
 * Copyright (C) 1995, 1996 Daniel Wu.
 * Distributed under the terms of the GNU Library General Public
 * License.
 *
 * This file is part of SimTos.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation (version 2).
 *
 * This library is distributed "AS IS" in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with Pthreads; see the file COPYING.  If not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA
 * 02139, USA.
 *
 * Author: Daniel Wu (dwu@linus.socs.uts.edu.au)
 *
 */

#include <stdlib.h>
#include <string.h>
#include "os_mtosux.h"
#include "symtab.h"
#include "mem.h"

#undef malloc
#undef free
#undef realloc
#undef calloc

#define ALFLAG  ((unsigned long) 0x98765432)

long bytes_free = GTALEN;


/*
 *--------------------------------------------------------------
 *
 * crcmp --
 *
 *	Create or attach a common memory pool.
 *
 * Results:
 *      The identifier to the newly created common memory pool.
 *
 *--------------------------------------------------------------
 */

uxid_t crcmp(pid_t pid, key_t key, UPTR basadr, ULONG len, ULONG ln2blk)
{
    
    OS_MEM_TYPE* cmp_p;
    ULONG        blk_len;
    ULONG        num_blks;        
    uxid_t       cmp_id;
    
    /*
     *	If the cmp exists and the crcmp parameters match,
     *	return the cpid of the existing cmp (attach).
     */
    if ((cmp_id = getcmp(key)) != BADPRM)
        return cmp_id;
    
    /*
     *	Validate pid.  Only -1 (global) & 0 are accepted.
     */
    if ((pid < -1L) || (pid > 15L))
    {
        /* crcmp: Bad pid */
        return BADPRM;
    }
    else if (pid > 0L)
    {
        perror("crcmp: pid ignored\n");
    }
    
    /*
     *	Validate ln2blk.  1 <= ln2blk <= 63.
     */
    if ((ln2blk < 1L) || (ln2blk > 63L))
    {
        /* crcmp: Bad ln2blk */
        return BADPRM;
    }
    blk_len = 1 << ln2blk;
    
    /*
     *	Validate len.  len >= 2**ln2blk.
     */
    if ((num_blks = len/blk_len) < 1)
    {
        /* crcmp: Pool smaller than block size */
        return BADPRM;
    }
    
    /*
     *	Make a new cpid.
     */
    
    cmp_p = (OS_MEM_TYPE*) malloc(sizeof(OS_MEM_TYPE));
    cmp_p->basadr = basadr;
    cmp_p->num_blks = num_blks;
    cmp_p->num_blks -= ((num_blks-1)/8 + 1)/blk_len + 1;        
    cmp_p->num_avail = cmp_p->num_blks;        
    cmp_p->blk_len = blk_len;
    cmp_p->free_blks = NULL;

    /*
    ** Initialise memory
    */
    memset(basadr, 0x00, cmp_p->num_avail*cmp_p->blk_len);
    
    return add_sym(key, cmp_p);
        
}


/*
 *--------------------------------------------------------------
 *
 * dlcmp --
 *
 *	Deletes a common memory pool.
 *
 * Results:
 *      NOERR if successful. Otherwise BADPRM.
 *
 *--------------------------------------------------------------
 */

long dlcmp(uxid_t cpid)
{
    return delete_sym(cpid);
}


/*
 *--------------------------------------------------------------
 *
 * getcmp --
 *
 *	Get the identifier corresponding to the given common 
 *      memory pool.
 *
 * Results:
 *      The identifier of the memory pool which corresponds
 *      to the given key.
 *
 *--------------------------------------------------------------
 */

uxid_t getcmp(key_t key)
{
    return key2id(key);    
}


/*
 *--------------------------------------------------------------
 *
 * alloc --
 *
 *	Allocates a block of memory from the common buffer pool.
 *
 * Results:
 *      A pointer to the allocated block of memory.
 *
 *--------------------------------------------------------------
 */

UPTR alloc(uxid_t cpid, ULONG numbyt, UPTR* alobfr, qual_t qual)
{

    OS_MEM_TYPE* sym_ptr;
    int          blocks_requested;    
    ULONG*       ptr = NULL;
    size_t       alosiz;
        
    if (alobfr == NULL)
        return (UPTR) BADPRM;
    
    if (cpid == -1L || cpid == 0L)
    {
        if ((bytes_free - (long)numbyt) > 0)
        {
            alosiz = numbyt + 2*sizeof(ULONG);
            if ((*alobfr = malloc(alosiz)) == NULL)
            {
                *alobfr = (void *) QUEFUL;
            }
            else
            {
                ULONG* aloptr = *alobfr;

                memset(*alobfr, 0x55, alosiz);                
                bytes_free -= numbyt;
                *aloptr++ = numbyt;		/* store allocation size */
                *aloptr++ = (ULONG)ALFLAG;	/*   &   flag */
                *alobfr = aloptr;                
            }
        }
        else
        {            
            *alobfr = (void *) QUEFUL;
        }
        
        return *alobfr;
    }
    else if (cpid <= 15L)
    {
        return (*alobfr = (UPTR) BADPRM);
    }
    else if ((sym_ptr = id2sym(cpid)) != NULL)
    {
        UPTR retval;
        
        lock_sym(cpid);
        
        blocks_requested = numbyt / sym_ptr->blk_len;
        if (numbyt % sym_ptr->blk_len)
        {
            blocks_requested++;
        }
        if (blocks_requested > sym_ptr->num_blks)
        {
            /* alloc: numbyts > pool size */
            retval = (*alobfr = (UPTR) BADPRM);
        }
        else
        {                
            if (CMODE(qual) != WAIFIN)
            {
                /* alloc: only WAIFIN supported */
                return (*alobfr = (UPTR) BADPRM);
            }
            
            if (LNUM(qual) & !(LUNITS(qual)))
            {
                retval = (*alobfr = (UPTR) BADPRM);
            }
            else
            {
                if (blocks_requested <= sym_ptr->num_avail)
                {
                    numbyt = blocks_requested * sym_ptr->blk_len;
                    ptr = (ULONG*) malloc(numbyt);
                    
                    if (ptr != NULL)
                    {
                        ULONG* aloptr = ptr;

                        memset(ptr, 0x55, numbyt);                        
                        *aloptr++ = numbyt;
                        *aloptr++ = (ULONG)ALFLAG;
                        *alobfr = aloptr;
                        sym_ptr->num_avail -= blocks_requested;
                    }
                    else
                    {
                        /* alloc: Cannot malloc */
                        *alobfr = (void *) QUEFUL;
                    }
                    
                    retval = *alobfr;
                }                
                else if (LUNITS(qual) && !(LNUM(qual)))	
                {
                           /* 0 of any units is IMONLY */
                    retval = (*alobfr = (UPTR) TIMOUT);
                }
                else
                {
                    /*
                     * alloc: Qualifier not supported
                     *         No retries upon failed allocation,
                     *         assumed timeout
                     */
                    retval = (*alobfr = (UPTR) TIMOUT);
                }
            }
        }            
        release_sym(cpid);
        
        return retval;        
    }
    
    return (*alobfr = (UPTR) BADPRM);
    
}


/*
 *--------------------------------------------------------------
 *
 * dalloc --
 *
 *	Deallocates a block of memory from the common buffer
 *      pool.
 *
 * Results:
 *      NOERR if successful. Otherwise BADPRM.
 *
 *--------------------------------------------------------------
 */

long dalloc(uxid_t cpid, ULONG numbyt, UPTR aloadr)
{
    OS_MEM_TYPE* sym_ptr;
    int          blocks_released;
    long         retval = NOERR;
    size_t       alosiz;
    ULONG*       aloptr;    
    
    if (cpid == -1L || cpid == 0L)
    {
        if (aloadr != NULL)
        {
            aloptr = (ULONG*)aloadr;
            if (*(--aloptr) == ALFLAG)
            {
                *aloptr = 0L;                
                alosiz = *(--aloptr);

                if (alosiz <= numbyt)
                {
                    free(aloptr);
                }
                else
                {
                    numbyt = alosiz - (size_t)numbyt;

                    aloptr = realloc(aloptr,
                                     numbyt+2*sizeof(ULONG));
                    *aloptr++ = numbyt;
                    *aloptr++ = (ULONG)ALFLAG;
                }
                bytes_free += numbyt;                    
                
                return NOERR;                                    
            }        
        }
        
        return BADPRM;         
    }
    else if (cpid <= 15L)
    {
        return BADPRM;
    }
    
    if ((sym_ptr = id2sym(cpid)) == NULL)
    {
        return BADPRM;
    }

    /*
     * If the memory came from a common pool, need to recalculate the
     * free space in the memory pool.
     */
    lock_sym(cpid);
    
    blocks_released = numbyt / sym_ptr->blk_len;
    if (numbyt % sym_ptr->blk_len)
    {
        blocks_released++;
    }
    if (blocks_released == 0)
    {
        retval = BADPRM;        
    }
    else if (blocks_released > sym_ptr->num_blks)
    {
        retval = BADPRM;
    }
    else
    {
        if (aloadr != NULL)
        {
            aloptr = (ULONG*)aloadr;            
            if (*(--aloptr) == ALFLAG)
            {
                *aloptr = 0L;                
                alosiz = *(--aloptr);

                numbyt = blocks_released * sym_ptr->blk_len;                
                if (alosiz == numbyt)
                {
                    free(aloptr);
                }
                else
                {
                    /* reallocate memory area */
                    if ((alosiz-numbyt) > 0)
                        aloadr = realloc(aloptr, alosiz-numbyt);
                    else
                    {
                        release_sym(cpid);                        
                        return BADPRM;
                    }                    
                }
                
                sym_ptr->num_avail += blocks_released;
                retval = NOERR;                                    
            }
            else
                retval = BADPRM;                                    
        }
    }
    
    release_sym(cpid);
    
    return retval;
}



/*
 *--------------------------------------------------------------
 *
 * crfbp --
 *
 *	Create or attach a fixed memory pool.
 *
 * Results:
 *      The identifier to the newly created fixed buffer pool.
 *
 *--------------------------------------------------------------
 */

uxid_t	crfbp(pid_t pid, key_t key, UPTR basadr, ULONG numblk, ULONG lenblk)
{

    OS_MEM_TYPE* sym_ptr;
    OS_MEM_TYPE* new_fbp;
    uxid_t       fbp_id;
    uxid_t       retval = NOERR;
    
    /*
     *	If the fbp exists and the crfbp parameters match,
     *	return the fpid of the existing fbp (attach).
     */
    if ((sym_ptr = key2sym(key)) != NULL)
    {
        fbp_id = key2id(key);
        
        lock_sym(fbp_id);
    
        if (sym_ptr->basadr != basadr ||
            sym_ptr->num_blks != numblk ||
            sym_ptr->blk_len != lenblk)
        {
            retval = BADPRM;
        }
	else
            retval = fbp_id;

        release_sym(fbp_id);
    }
    else
    {            
	/*
         *  Validate pid.  Only -1 (global) & 0 are accepted.
         */
        if ((pid < -1L) || (pid > 15L))
            retval = BADPRM;
        else if (pid > 0L)
            perror("crfbp: pid ignored\n");

	/*
         *  Validate lenblk.  4 <= lenblk 
         */
        if (lenblk < 4)
            retval = BADPRM;

	/*
         *  Validate numblk.  1 <= numblk
         */
        if (numblk < 1)
            retval = BADPRM;

	/*
         *  Make a new fbp.
         */
	if (retval != BADPRM)
	{
	    new_fbp = (OS_MEM_TYPE *) malloc(sizeof(OS_MEM_TYPE));        
	    if (new_fbp != NULL)
	    {
		/*
		 *	Initialize fbp fields.
		 */
		new_fbp->basadr	   = basadr;
		new_fbp->num_blks  = numblk;
		new_fbp->num_avail = numblk;
		new_fbp->blk_len   = lenblk;
		new_fbp->free_blks = NULL;
		
		/*
		 *	Add key to the symbol table.  The symbol entry points
		 *	to the fbp
		 */
		
		retval = add_sym(key, new_fbp);
	    }
	    else
	    {
		/*
		 *  Create failed.  Return QUEFUL.
		 */
		
		retval = QUEFUL;            
	    }
	}
    }
    
    return retval;    
}


/*
 *--------------------------------------------------------------
 *
 * dlfbp --
 *
 *	Delete an fbp.
 *
 * Results:
 *      NOERR if successful. Otherwise BADPRM.
 *
 *--------------------------------------------------------------
 */

long dlfbp(uxid_t fpid)
{

    OS_MEM_TYPE* sym_ptr;
    
    /*
     *	Find fbp in symbol table.
     */
    if ((sym_ptr = id2sym(fpid)) != NULL)
    {        
        /*
         *  Delete symbol table entries and return fbp
         *  memory.
         */
        
        return delete_sym(fpid); 
    }
    else 
    {
        
        /*
         *  No cmp exists for specifed fpid.  Return BADPRM.
         */
	return BADPRM;
    }
    
}


/*
 *--------------------------------------------------------------
 *
 * alofbp --
 *
 *	Allocate a fix buffer pool.
 *
 * Results:
 *      A pointer to the allocated block of memory.
 *
 *--------------------------------------------------------------
 */

UPTR alofbp(uxid_t fpid, UPTR* alobfr, qual_t qual)
{

    OS_MEM_TYPE* sym_ptr;
    
    if ((sym_ptr = id2sym(fpid)) != NULL)
    {

        if (CMODE(qual) != WAIFIN)
        {
            perror("alloc: only WAIFIN suported\n");
        }
        
        if (LNUM(qual) & !(LUNITS(qual)))
        {
            return (UPTR)BADPRM;
        }

        lock_sym(fpid);        
        
        if (sym_ptr->num_avail > 0)
        {
            sym_ptr->num_avail--;
            if ((*alobfr = malloc(sym_ptr->blk_len)) != NULL)
            {
		memset(*alobfr, 0x55, sym_ptr->blk_len);
		release_sym(fpid);		
            }
	    else
	    {
                *alobfr = (UPTR) QUEFUL;
	    }
            
            return (UPTR) *alobfr;
        }
        else if (LUNITS(qual) && !(LNUM(qual)))
        {
            /* 0 of any units is IMONLY */
            
            return (UPTR)QUEFUL;
        }
        else
        {
            perror("alloc: Qualifier not supported\n");            
	}
    }
    
    return (*alobfr = (UPTR)BADPRM);

}


/*
 *--------------------------------------------------------------
 *
 * dalfbp --
 *
 *	Deallocate a fix buffer pool.
 *
 * Results:
 *      NOERR if successful. Otherwise BADPRM.
 *
 *--------------------------------------------------------------
 */

long dalfbp(uxid_t fpid, UPTR aloadr)
{

    OS_MEM_TYPE* sym_ptr;

    if ((sym_ptr = id2sym(fpid)) != NULL)
    {
	lock_sym(fpid);
    
	free(aloadr);
	sym_ptr->num_avail++;

	release_sym(fpid);
    }
    else
    {
        return BADPRM;
    }
	
    return NOERR;

}


/*
 *--------------------------------------------------------------
 *
 * getfbp --
 *
 *	Get the fixed buffer pool corresponding to the given
 *      key.
 *
 * Results:
 *      The identifier of the fixed memory pool which corresponds
 *      to the given key.
 *
 *--------------------------------------------------------------
 */

uxid_t getfbp(key_t key)
{
    return key2id(key);    
}
