/*
 * mbx.c --
 *
 *      Provides mailbox services for the MTOS-UX simulator.
 *
 * Copyright (C) 1995, 1996 Daniel Wu.
 * Distributed under the terms of the GNU Library General Public
 * License.
 *
 * This file is part of SimTos.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation (version 2).
 *
 * This library is distributed "AS IS" in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with Pthreads; see the file COPYING.  If not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA
 * 02139, USA.
 *
 * Author: Daniel Wu (dwu@linus.socs.uts.edu.au)
 *
 */


#include "mbx.h"


/*
 *--------------------------------------------------------------
 *
 * opnmbx --
 *
 *      Opens a mailbox for sending or receiving messages.
 *
 * Results:
 *      If the function is successful, returns the identifier of
 *      the mailbox. Otherwise, QUEFUL.
 *
 *--------------------------------------------------------------
 */

uxid_t opnmbx(key_t key, ULONG mode)
{
    OS_MBX_TYPE*        mbx_p;
    uxid_t              retval;

    if (mode != MBRCV && mode != MBSND)
	return BADPRM;
        
    if ((mbx_p = key2sym(key)) == NULL)
    {
	mbx_p = (OS_MBX_TYPE*) malloc(sizeof(OS_MBX_TYPE));
	mbx_p->symtype = sym_mbx;
	mbx_p->msgq = NULL;
	mbx_p->opnsnd = 0;
	mbx_p->opnrcv = 0;

	if ((pthread_cond_init(&mbx_p->os_efg.efg_set, NULL) == 0) &&
	    (pthread_mutex_init(&mbx_p->os_efg.efg_mutex, NULL) == 0))
	{
	    if (pthread_cond_init(&mbx_p->mbx_cond, NULL) == 0)
	    {        
		retval = add_sym(key, mbx_p);
	    }
	}
	else
	{
	    free(mbx_p); 
	    retval = QUEFUL;
	}
    }
    else
    {
	retval = key2id(key);
    }

    /*
     * Update tallies
     */
    if (mode == MBSND)
	mbx_p->opnsnd++;
    else if (mode == MBRCV)
	mbx_p->opnrcv++;
    
    return retval;	    
}

/*
 *--------------------------------------------------------------
 *
 * clsmbx --
 *
 *      Closes the mailbox in the given mode.
 *
 * Results:
 *      If the function is successful, returns NOERR. Otherwise,
 *      BADPRM.
 *
 *--------------------------------------------------------------
 */

long clsmbx(key_t key, ULONG mode)
{
    OS_MBX_TYPE*        mbx_p;
    long                retval;
    
    if (mode != MBRCV && mode != MBSND)
	return BADPRM;

    if ((mbx_p = key2sym(key)) != NULL)
    {
	if (mbx_p->opnsnd == 0 || mbx_p->opnrcv == 0)
	    return BADPRM;
	
	if (mode == MBSND)
	{
	    if (--mbx_p->opnsnd == 0)
	    {
		pthread_cond_signal(&mbx_p->os_efg.efg_set);
		
		retval = NOERR;
	    }
	    else
		retval = NOTFRE;
	}
	else if (mode == MBRCV)
	{
	    if (--mbx_p->opnrcv == 0)
	    {
		if (mbx_p->opnsnd == 0)
		{
		    OS_PRIENTRY* msg;
		    
		    while ((msg = PriDeleteMax(&mbx_p->msgq)) != NULL)
			free(msg);
		}
		
		retval = NOERR;
	    }
	    else
		retval = NOTFRE;
	}

	if (mbx_p->opnsnd == 0 && mbx_p->opnrcv == 0)
	{
	    /*
	     * Delete mailbox
	     */
	    if (mbx_p->msgq != NULL)
		delete_sym(key);
	}
    }

    return retval;
}

void os_delete_mbx(uxid_t mbid, OS_MBX_TYPE* mbx)
{
    OS_PRIENTRY* msg;
    
    while ((msg = PriDeleteMax(&mbx->msgq)) != NULL)
	free(msg);

    delete_sym(id2key(mbid));
}


/*
 *--------------------------------------------------------------
 *
 * dlmbx --
 *
 *      Deletes the mailbox.
 *
 * Results:
 *      If the function is successful, returns NOERR. Otherwise,
 *      BADPRM.
 *
 *--------------------------------------------------------------
 */

long dlmbx(uxid_t mbid)
{
    OS_MBX_TYPE* mbx;
    long         retval = BADPRM;

    mbx = id2sym(mbid);

    if (mbx != NULL && mbx->symtype == sym_mbx)
    {
	/*
	 * Check if anyone is there are any receive requests
	 */
	if (pthread_mutex_trylock(&mbx->os_efg.efg_mutex) == -1)
	{
	    mbx->opnsnd = mbx->opnrcv = -1; /* mark for deletion */
	    
	    pthread_cond_signal(&mbx->os_efg.efg_set);
	}
	else
	{
	    os_delete_mbx(mbid, mbx);
	}
	
	retval = NOERR;
    }

    return retval;
}


/*
 *--------------------------------------------------------------
 *
 * sndmbx --
 *
 *      Sends a message to a mailbox.
 *
 * Results:
 *      If the function is successful, returns NOERR. Otherwise,
 *      BADPRM.
 *
 *--------------------------------------------------------------
 */


long sndmbx(uxid_t mbid, UPTR srcadr, ULONG prty, ULONG *stabuf, qual_t qual)
{
    OS_MBX_TYPE* mbx;
    long         retval = BADPRM;

    mbx = id2sym(mbid);
    
    if (mbx != NULL && mbx->symtype == sym_mbx && mbx->opnsnd > 0)
    {
	OS_PRIENTRY* msg;
	
	/*
	 * Create and queue the entry
	 */
	msg = (OS_PRIENTRY*) malloc(sizeof(OS_PRIENTRY));

	msg->priority = prty;
	msg->length = *((long*)srcadr);
	msg->entry = malloc(msg->length);
	memcpy(msg->entry, &(((long*)srcadr)[1]), msg->length);
	msg->pred = msg->succ = msg;
	
	PriInsert(&mbx->msgq, msg);

	/*
	 * For all tasks waiting on the event flag
	 */
	pthread_cond_signal(&mbx->os_efg.efg_set);

	/*
	 * Return codes
	 */
	retval = NOERR;	
    }

    *stabuf = retval;		
    
    return retval;
}


/*
 *--------------------------------------------------------------
 *
 * rcvmbx --
 *
 *      Receive a message to a mailbox.
 *
 * Results:
 *      If the function is successful, returns NOERR. Otherwise,
 *      BADPRM.
 *
 *--------------------------------------------------------------
 */

long rcvmbx(uxid_t mbid, UPTR dstadr, ULONG *stabuf, qual_t qual)
{
    OS_MBX_TYPE* mbx;
    long         retval = BADPRM;

    mbx = id2sym(mbid);
    
    if (mbx != NULL && mbx->symtype == sym_mbx && mbx->opnrcv > 0)
    {
	OS_PRIENTRY* msg;
	long         msglen;
	
	while ((msg = PriDeleteMax(&mbx->msgq)) == NULL)
	{
	    pthread_mutex_lock(&mbx->os_efg.efg_mutex);	    
	    pthread_cond_wait(&mbx->os_efg.efg_set,
			      &mbx->os_efg.efg_mutex);

	    if (mbx->opnsnd == 0)
		return MBEOF;

	    if (mbx->opnsnd == -1 && mbx->opnrcv == -1)
	    {
		os_delete_mbx(mbid, mbx);
		
		return MBDLT;
	    }
	}
	
	msglen = (*((long*)dstadr) < msg->length) ?
	                             *((long*)dstadr) : msg->length;
	if (msglen > 0)
	    memcpy(&(((long*)dstadr)[1]), msg->entry, msglen);
	free(msg);

	retval = NOERR;
    }

    *stabuf = retval;
    
    return retval;
}

