#include "runix.h"
#include "net/net.h"
#include "runixd/runixd.h"

/*
 * Offer an fd to the new runixd process spawned by fork(). There are at least
 * 3 ways to do it on different systems:
 *
 * ^oo^ BSD socket sendmsg/recvmsg
 * ^oo^ SVR4 ioctl
 * ^oo^ Linux /proc filesystem
 *
 * As SVR4 also has networking, complete with sendmsg, I didn't actually
 * implement ioctl method.
 */

#ifdef linux

/*
 * Processes running with the same euid  can access each other's /proc/pid#/fd
 * directories.    Unfortunately, changing   uid  gives access   to *all* file
 * descriptors, as well as ptrace().  To minimize the danger, I use a separate
 * process and  close all  file descriptors  but the one   to pass ASAP. Still
 * there is a possible race condition. Enjoy it!  
 *
 * In the spirit  of family practices  of some spiders,  the recieving process
 * kills the sender once it gets the file descriptor.
 */

void givefd(int fd)
{
  ulg handle = htonl((ulg)getpid());
  int i;
  
  dup2(fd, 0);
  for(i = 1; i < FD_SETSIZE; i++) close(fd);

  write(0, &handle, sizeof(ulg));
  
  /*
   * This read makes sure that we go away if the client dies before making
   * RPC call. It's also a nice way to wait for signal.
   */
  read(0, &handle, sizeof(ulg));
}

#else

#include <sys/socket.h>
#include <sys/un.h>

void givefd(int fd)
{
  int s, s0;
  ulg handle;
  static struct sockaddr_un un;
  struct stat st;
  fd_set fds;

  {
    static struct sigaction sa;
    sa.sa_handler = SIG_IGN;
    sigaction(SIGPIPE, &sa, NULL);
  }

  if((s = socket(AF_UNIX, SOCK_STREAM, 0)) < 0) return;

  un.sun_family = AF_UNIX;

  /*
   * Creating intermediate files can be messy as any given name could already
   * exist and be owned by someone else.
   */

  for(handle = random() ;; handle ++) {
    sprintf(un.sun_path, "/tmp/%lX", handle);
    if(stat(un.sun_path, &st) == -1 &&
       bind(s, &un, sizeof(un)) != -1)
      break;
  }

  /* 
   * Should be no race condition, because we didn't listen() yet?
   */
    
  chmod(un.sun_path, 0600);
  listen(s, 5);
  handle = htonl(handle);
  write(fd, &handle, sizeof(ulg));

  /*
   * Wait for connection, but also keep an eye on fd: if it's ready
   * for read, it means the client exited and so should we.
   */

  FD_ZERO(&fds);
  FD_SET(fd, &fds);
  FD_SET(s, &fds);

  while(select(FD_SETSIZE, &fds, NULL, NULL, NULL) == -1)
    if(errno != EINTR) {
    forgetit:;
      unlink(un.sun_path);
      return;
    }

  if(FD_ISSET(s, &fds)) {
    struct iovec io;
    static struct msghdr mhdr;
    struct sockaddr_un peer;
    int psize = sizeof(peer);

    while((s0 = accept(s, &peer, &psize)) == -1)
      if(errno != EINTR)
	goto forgetit;

    io.iov_base = &fd;
    io.iov_len = sizeof(int);
    
    mhdr.msg_iov = &io;
    mhdr.msg_iovlen = 1;

#ifdef NEW_SENDMSG
    /*
     * "Advanced Programming in the UNIX Environment" by W. Richard Stevens
     * says that this is how passing fd's works in BSD 4.3Reno and later. This
     * is completely untested. Good luck!
     */
    {
      static struct {
	struct cmsghdr cm;
	int fd;
      } x;
      
      x.cm.cmsg_len = sizeof(x);
      x.cm.cmsg_level = SOL_SOCKET;
      x.cm.cmsg_type = SCM_RIGHTS;
      x.fd = s;
      
      mhdr.msg_control = &x;
      mhdr.msg_controllen = sizeof(x);
    }
#else
    mhdr.msg_accrights = &fd;
    mhdr.msg_accrightslen = sizeof(int);

    while(sendmsg(s0, &mhdr, 0) == -1 && errno == EINTR);
#endif
  }

  unlink(un.sun_path);
}

#endif
