#include "runix.h"
#include "rpc/runix_prot.h"
#include "clnt/runixlib.h"
#include "preload/c_runix.h"
#include "preload/preload.h"

static char fork_msg[] = 
	"runix/fork: failed to duplicate service descriptor\n";

OPEN_RET i_open OPEN_P(file, flags)
{
  va_list va;
  int mode;

  va_start(va, flags);
  mode = (flags & O_CREAT) ? va_arg(va, int) : 0666;
  va_end(va);
  
  chk_path(file);
  
  if(doremote(file)) {
    OPEN_RET r, newfd;
    r = r_open(pathdup(file), flags, mode);
    if(r == -1) return -1;
    if((newfd = c_dup(dummyfd)) == -1) {
      hide_errno(r_close(r));
      return -1;
    }
    fdmap[newfd] = r;
    return newfd;
  } else {
    leave_critical();
    return c_open(file, flags, mode);
  }
}

CREAT_RET i_creat CREAT_P(file, flags)
{
  chk_path(file);

  if(doremote(file)) {
    CREAT_RET r, newfd;
    r = r_creat(pathdup(file), flags);
    if(r == -1) return -1;
    if((newfd = c_dup(dummyfd)) == -1) {
      hide_errno(r_close(r));
      return -1;
    }
    fdmap[newfd] = r;
    return newfd;
  } else {
    leave_critical();
    return c_creat(file, flags);
  }
}
    
READ_RET i_read READ_P(fd, buf, count)
{
  chk_fd(fd);
  chk_buf(buf);

  if(isremfd(fd))
    return r_read(fdmap[fd], buf, count);
  else {
    leave_critical();
    return c_read(fd, buf, count);
  }
}

WRITE_RET i_write WRITE_P(fd, buf, count)
{
  chk_fd(fd);
  chk_buf(buf);

  if(isremfd(fd))
    return r_write(fdmap[fd], buf, count);
  else {
    leave_critical();
    return c_write(fd, buf, count);
  }
}

CLOSE_RET i_close CLOSE_P(fd)
{
  chk_fd(fd);
  
  if(isremfd(fd)) {
    CLOSE_RET r = r_close(fdmap[fd]);
    hide_errno(close(fd));
    fdmap[fd] = 0;
    return r;
  } else {
    leave_critical();
    return c_close(fd);
  }
}

LSEEK_RET i_lseek LSEEK_P(fd, offset, whence)
{
  chk_fd(fd);
  
  return isremfd(fd) ? r_lseek(fdmap[fd], offset, whence) :
    c_lseek(fd, offset, whence);
}

#ifdef linux

_XSTAT_RET i__xstat _XSTAT_P(ver, file, st)
{
  chk_path(file);
  chk_buf(st);

  return doremote(file) ? r_stat(pathdup(file), st) :
    c__xstat(ver, file, st);
}

_FXSTAT_RET i__fxstat _FXSTAT_P(ver, fd, st)
{
  chk_fd(fd);
  chk_buf(st);

  return isremfd(fd) ? r_fstat(fdmap[fd], st) :
    c__fxstat(ver, fd, st);
}

_LXSTAT_RET i__lxstat _LXSTAT_P(ver, file, st)
{
  chk_path(file);
  chk_buf(st);

  return doremote(file) ? r_lstat(pathdup(file), st) :
    c__lxstat(ver, file, st);
}

#else /* linux */

STAT_RET i_stat STAT_P(file, st)
{
  chk_path(file);
  chk_buf(st);

  return doremote(file) ? r_stat(pathdup(file), st) :
    c_stat(file, st);
}

FSTAT_RET i_fstat FSTAT_P(fd, st)
{
  chk_fd(fd);
  chk_buf(st);

  return isremfd(fd) ? r_fstat(fdmap[fd], st) :
    c_fstat(fd, st);
}

LSTAT_RET i_lstat LSTAT_P(file, st)
{
  chk_path(file);
  chk_buf(st);

  return doremote(file) ? r_lstat(pathdup(file), st) :
    c_lstat(file, st);
}

#endif /*linux*/

CHMOD_RET i_chmod CHMOD_P(path, mode)
{
  chk_path(path);
  
  return doremote(path) ? r_chmod(pathdup(path), mode) : c_chmod(path, mode);
}

FCHMOD_RET i_fchmod FCHMOD_P(fd, mode)
{
  chk_fd(fd);
  
  return isremfd(fd) ? r_fchmod(fdmap[fd], mode) :
    c_fchmod(fd, mode);
}

CHOWN_RET i_chown CHOWN_P(path, uid, gid)
{
  chk_path(path);

  return doremote(path) ?
    r_chown(pathdup(path), uid, gid) : c_chown(path, uid, gid);
}

FCHOWN_RET i_fchown FCHOWN_P(fd, uid, gid)
{
  chk_fd(fd);

  return isremfd(fd) ? r_fchown(fdmap[fd], uid, gid) :
    c_fchown(fd, uid, gid);
}

UMASK_RET i_umask UMASK_P(mask)
{
  hide_errno(r_umask(mask));
  return c_umask(mask);
}

MKDIR_RET i_mkdir MKDIR_P(path, mode)
{
  chk_path(path);
  return doremote(path) ?
    r_mkdir(pathdup(path), mode) : c_mkdir(path, mode);
}

CHDIR_RET i_chdir CHDIR_P(path)
{
  CHDIR_RET r;

  chk_path(path);
  
  if(doremote(path)) {
    r = r_chdir(pathdup(path));
    if(r != -1) remote = 1;
  } else {
    r = c_chdir(path);
    if(r != -1) remote = 0;
  }
  return r;
}

RMDIR_RET i_rmdir RMDIR_P(path)
{
  chk_path(path);
  return doremote(path) ?
    r_rmdir(pathdup(path)) : c_rmdir(path);
}

FCHDIR_RET i_fchdir FCHDIR_P(fd)
{
  FCHDIR_RET r;

  chk_fd(fd);

  if(isremfd(fd)) {
    r = r_fchdir(fdmap[fd]);
    if(r != -1) remote = 1;
  } else {
    r = c_fchdir(fd);
    if(r != -1) remote = 0;
  }
  return r;
}

LINK_RET i_link LINK_P(old, new)
{
  chk_path(old);
  chk_path(new);

  if(doremote(old) != doremote(new)) {
    errno = EXDEV;
    return -1;
  }

  return doremote(old) ? r_link(pathdup(old), pathdup(new)) :
    c_link(old, new);
}

UNLINK_RET i_unlink UNLINK_P(path)
{
  chk_path(path);

  return doremote(path) ? r_unlink(pathdup(path)) : c_unlink(path);
}


RENAME_RET i_rename RENAME_P(old, new)
{
  chk_path(old);
  chk_path(new);

  if(doremote(old) != doremote(new)) {
    errno = EXDEV;
    return -1;
  }

  return doremote(old) ? r_rename(pathdup(old), pathdup(new)) :
    c_rename(old, new);
}

SYMLINK_RET i_symlink SYMLINK_P(old, new)
{
  chk_path(old);
  chk_path(new);

  return doremote(old) ? r_symlink(pathdup(old), pathdup(new)) :
    c_symlink(old, new);
}

READLINK_RET i_readlink READLINK_P(path, buf, bufsiz)
{
  chk_path(path);
  chk_buf(buf);

  return doremote(path) ? r_readlink(pathdup(path), buf, bufsiz) :
    c_readlink(path, buf, bufsiz);
}

GETCWD_RET i_getcwd GETCWD_P(buf, size)
{
  GETCWD_RET r;
  GETCWD_RET r0;
  int siz;

  if(!remote) {
    /*
     * So what do you know next? getcwd, getwd and access can fork off
     * processes, so we better re-enable runix or vfork will bite us.
     */
    leave_critical();
    return c_getcwd(buf, size);
  }

  if(buf) {
    if(!(r = r_getcwd(buf+2, size-2)))
      return NULL;
    r = buf;
  } else {
    if(!(r = r_getcwd(NULL, size)))
      return NULL;
    siz = strlen(r) + 3;
    if(size >= 0 && (size < siz)) {
      free(r);
      errno = ERANGE;
      return NULL;
    } else if(size < 0 && !(r = realloc(r0 = r, siz))) {
      free(r0);
      errno = ERANGE;
      return NULL;
    }
    memmove(r+2, r, siz-2);
  }
  r[0] = '/';
  r[1] = '@';
  return r;
}

GETWD_RET i_getwd GETWD_P(buf)
{
  if(remote)
    return i_getcwd(buf, -1);
  else {
    leave_critical();
    return c_getwd(buf);
  }
}

DUP_RET i_dup DUP_P(fd)
{
  chk_fd(fd);

  if(isremfd(fd)) {
    DUP_RET r, newfd;

    if((r = r_dup(fdmap[fd])) == -1)
      return -1;

    if((newfd = c_dup(dummyfd)) == -1) {
      hide_errno(r_close(r));
      return -1;
    }

    fdmap[newfd] = r;
    return newfd;
  } else
    return c_dup(fd);
}

DUP2_RET i_dup2 DUP2_P(old, new)
{
  chk_fd(old);
  chk_fd(new);
  
  if(c_dup2(old, new) == -1) return -1;

  if(isremfd(old)) {
    DUP2_RET r;
    
    if((r = r_dup(fdmap[old])) == -1) return -1;

    if(isremfd(new))
      hide_errno(r_close(fdmap[new]));
      
    fdmap[new] = r;
  } else if(isremfd(new)) {
    hide_errno(r_close(fdmap[new]));
    fdmap[new] = 0;
  }
  return 0;
}

TRUNCATE_RET i_truncate TRUNCATE_P(path, len)
{
  chk_path(path);
  
  return doremote(path) ?
    r_truncate(pathdup(path), len) : c_truncate(path, len);
}

FTRUNCATE_RET i_ftruncate FTRUNCATE_P(fd, len)
{
  chk_fd(fd);

  return isremfd(fd) ?
    r_ftruncate(fdmap[fd], len) : c_ftruncate(fd, len);
}

FORK_RET i_fork FORK_P()
{
  FORK_RET pid;
  ulg handle;
  int fd, lim;
  struct rlimit rl;

  if((fd = r_runixcmd(NULL, PROC_FORK, &handle)) == -1) {
    errno = EDEADRPC;
    return -1;
  }

  if(r_fork(handle) == -1) {
    c_close(fd);
    errno = EDEADRPC;
    return -1;
  }

  switch(pid = c_fork()) {
  case -1:
    hide_errno(c_close(fd));
    return -1;
  case 0:
    getrlimit(RLIMIT_NOFILE, &rl);
    lim = rl.rlim_cur;
    rl.rlim_cur = rl.rlim_max;
    setrlimit(RLIMIT_NOFILE, &rl);
    if(c_dup2(fd, rpc_fd) == -1) {
      /*
       * Uh-oh! Complaining might be of no use. Suicide instead.
       */
      c_write(2, fork_msg, sizeof(fork_msg)-1);
      kill(getpid(), SIGKILL);
    }
    rl.rlim_cur = lim;
    c_close(fd);
    setrlimit(RLIMIT_NOFILE, &rl);
    return 0;
  default:
    hide_errno(c_close(fd));
    return pid;
  }
}

VFORK_RET i_vfork VFORK_P()
{
  return i_fork();
}

UTIME_RET i_utime UTIME_P(path, buf)
{
  chk_path(path);
  
  return doremote(path) ?
    r_utime(pathdup(path), buf) : c_utime(path, buf);
}

#ifdef linux
_XMKNOD_RET i__xmknod _XMKNOD_P(ver, path, mode, dev)
{
  chk_path(path);

  return doremote(path) ?
    r_mknod(pathdup(path), mode, *dev) : c__xmknod(ver, path, mode, dev);
}

/*
 * BugBugBug: Linux libc 5.x.x fflush'es file descriptors AFTER dlclose on all
 * libraries.
 */

EXIT_RET i_exit EXIT_P(ret)
{
  leave_critical();
  fflush(NULL);
  c_exit(ret);
}

#else

MKNOD_RET i_mknod MKNOD_P(path, mode, dev)
{
  chk_path(path);

  return doremote(path) ?
    r_mknod(pathdup(path), mode, dev) : c_mknod(path, mode, dev);
}

#endif

ACCESS_RET i_access ACCESS_P(path, mode)
{
  chk_path(path);

  if(doremote(path))
    return r_access(pathdup(path), mode);
  else {
    leave_critical();
    return c_access(path, mode);
  }
}

MMAP_RET i_mmap MMAP_P(addr, len, prot, flags, fd, offset)
{
  chk_fd(fd);

  return (
#ifdef MAP_ANON
     !(flags & MAP_ANON) &&
#endif
     isremfd(fd)) ?
       r_mmap(addr, len, prot, flags, fdmap[fd], offset) :
       c_mmap(addr, len, prot, flags, fd, offset);
}

READV_RET i_readv READV_P(fd, vec, count)
{
  chk_fd(fd);
  chk_buf(vec);

  if(isremfd(fd))
    return r_readv(fdmap[fd], vec, count);
  else {
    leave_critical();
    return c_readv(fd, vec, count);
  }
}

WRITEV_RET i_writev WRITEV_P(fd, vec, count)
{
  chk_fd(fd);
  chk_buf(vec);

  if(isremfd(fd))
    return r_writev(fdmap[fd], vec, count);
  else {
    leave_critical();
    return c_writev(fd, vec, count);
  }
}

FSYNC_RET i_fsync FSYNC_P(fd)
{
  chk_fd(fd);

  if(isremfd(fd))
    return r_fsync(fdmap[fd]);
  else {
    leave_critical();
    return c_fsync(fd);
  }
}

    
