#include "runix.h"
#include "rpc/runix_prot.h"
#include "clnt/runixlib.h"

#include "clnt/chk.h"

OPEN_RET r_open OPEN_P(file, flags)
{
  int_ret *r;
  int ret;
  va_list va;
  mode_t mode;

  va_start(va, flags);

  chk_path(file);
  
  rpc_call(ret, r, runix_open_1((char *)file, enc_open(flags), 
			(flags & O_CREAT) ? va_arg(va, int) : 0, clnt));
  va_end(va);

  XDR_FREE(xdr_int_ret, r);
  return ret;
}

CREAT_RET r_creat CREAT_P(file, mode)
{
  return r_open(file, O_WRONLY | O_CREAT | O_TRUNC, mode);
}

READ_RET r_read READ_P(fd, buf, count)
{
  buf_ret *r;
  int ret;

  chk_fd(fd);
  chk_buf(buf);

  rpc_call(ret, r, runix_read_1(fd, count, clnt));

  if(ret != -1)
    memcpy(buf, r -> buf.opq_val, r -> buf.opq_len);

  XDR_FREE(xdr_buf_ret, r);
  return ret;
}

WRITE_RET r_write WRITE_P(fd, buf, count)
{
  int_ret *r;
  opq b;
  int ret;

  chk_fd(fd);
  chk_buf(buf);

  b.opq_len = count;
  b.opq_val = (char *)buf;

  rpc_call(ret, r, runix_write_1(fd, b, count, clnt));
  
  XDR_FREE(xdr_int_ret, r);
  return ret;
}

CLOSE_RET r_close CLOSE_P(fd)
{
  int_ret *r;
  int ret;

  chk_fd(fd);

  rpc_call(ret, r, runix_close_1(fd, clnt));

  XDR_FREE(xdr_int_ret, r);
  return ret;
}

LSEEK_RET r_lseek LSEEK_P(fd, offset, whence)
{
  int_ret *r;
  int ret;

  chk_fd(fd);
  
  rpc_call(ret, r, runix_lseek_1(fd, offset, enc_lseek(whence), clnt));

  XDR_FREE(xdr_int_ret, r);
  return ret;
}

  
STAT_RET r_stat STAT_P(file, st)
{
  stat_ret *r;
  int ret;

  chk_path(file);
  chk_buf(st);

  rpc_call(ret, r, runix_stat_1((char *)file, clnt));
  
  *st = dec_stat(r -> st);
  XDR_FREE(xdr_stat_ret, r);
  return ret;
}

FSTAT_RET r_fstat FSTAT_P(fd, st)
{
  stat_ret *r;
  int ret;

  chk_fd(fd);
  chk_buf(st);

  rpc_call(ret, r, runix_fstat_1(fd, clnt));
  
  *st = dec_stat(r -> st);
  XDR_FREE(xdr_stat_ret, r);
  return ret;
}

LSTAT_RET r_lstat LSTAT_P(file, st)
{
  stat_ret *r;
  int ret;

  chk_path(file);
  chk_buf(st);

  rpc_call(ret, r, runix_lstat_1((char *)file, clnt));
  
  *st = dec_stat(r -> st);
  XDR_FREE(xdr_stat_ret, r);
  return ret;
}

CHMOD_RET r_chmod CHMOD_P(path, mode)
{
  int_ret *r;
  int ret;

  chk_path(path);
  
  rpc_call(ret, r, runix_chmod_1((char *)path, mode, clnt));

  XDR_FREE(xdr_int_ret, r);
  return ret;
}

FCHMOD_RET r_fchmod FCHMOD_P(fd, mode)
{
  int_ret *r;
  int ret;

  chk_fd(fd);
  
  rpc_call(ret, r, runix_fchmod_1(fd, mode, clnt));

  XDR_FREE(xdr_int_ret, r);
  return ret;
}

CHOWN_RET r_chown CHOWN_P(path, uid, gid)
{
  int_ret *r;
  int ret;

  chk_path(path);

  rpc_call(ret, r, runix_chown_1((char *)path, uid, gid, clnt));

  XDR_FREE(xdr_int_ret, r);
  return ret;
}

FCHOWN_RET r_fchown FCHOWN_P(fd, uid, gid)
{
  int_ret *r;
  int ret;

  chk_fd(fd);

  rpc_call(ret, r, runix_fchown_1(fd, uid, gid, clnt));

  XDR_FREE(xdr_int_ret, r);
  return ret;
}

UMASK_RET r_umask UMASK_P(mask)
{
  int_ret *r;
  int ret;

  rpc_call(ret, r, runix_umask_1(mask, clnt));
  
  XDR_FREE(xdr_int_ret, r);
  return ret;
}

MKDIR_RET r_mkdir MKDIR_P(path, mode)
{
  int_ret *r;
  int ret;

  chk_path(path);
  
  rpc_call(ret, r, runix_mkdir_1((char *)path, mode, clnt));

  XDR_FREE(xdr_int_ret, r);
  return ret;
}

CHDIR_RET r_chdir CHDIR_P(path)
{
  int_ret *r;
  int ret;

  chk_path(path);
  
  rpc_call(ret, r, runix_chdir_1((char *)path, clnt));

  XDR_FREE(xdr_int_ret, r);
  return ret;
}

RMDIR_RET r_rmdir RMDIR_P(path)
{
  int_ret *r;
  int ret;

  chk_path(path);

  rpc_call(ret, r, runix_rmdir_1((char *)path, clnt));

  XDR_FREE(xdr_int_ret, r);
  return ret;
}

FCHDIR_RET r_fchdir FCHDIR_P(fd)
{
  int_ret *r;
  int ret;

  chk_fd(fd);

  rpc_call(ret, r, runix_fchdir_1(fd, clnt));

  XDR_FREE(xdr_int_ret, r);
  return ret;
}

LINK_RET r_link LINK_P(old, new)
{
  int_ret *r;
  int ret;

  chk_path(old);
  chk_path(new);

  rpc_call(ret, r, runix_link_1((char *)old, (char *)new, clnt));

  XDR_FREE(xdr_int_ret, r);
  return ret;
}

UNLINK_RET r_unlink UNLINK_P(path)
{
  int_ret *r;
  int ret;

  chk_path(path);

  rpc_call(ret, r, runix_unlink_1((char *)path, clnt));

  XDR_FREE(xdr_int_ret, r);
  return ret;
}

RENAME_RET r_rename RENAME_P(old, new)
{
  int_ret *r;
  int ret;

  chk_path(old);
  chk_path(new);

  rpc_call(ret, r, runix_rename_1((char *)old, (char *)new, clnt));

  XDR_FREE(xdr_int_ret, r);
  return ret;
}

SYMLINK_RET r_symlink SYMLINK_P(old, new)
{
  int_ret *r;
  int ret;

  chk_path(old);
  chk_path(new);

  rpc_call(ret, r, runix_symlink_1((char *)old, (char *)new, clnt));

  XDR_FREE(xdr_int_ret, r);
  return ret;
}

READLINK_RET r_readlink READLINK_P(path, buf, bufsiz)
{
  buf_ret *r;
  int ret;

  chk_path(path);
  chk_buf(buf);

  rpc_call(ret, r, runix_readlink_1((char *)path, clnt));

  if(ret != -1)
    memcpy(buf, r -> buf.opq_val, r -> buf.opq_len);

  XDR_FREE(xdr_buf_ret, r);
  return ret;
}

GETCWD_RET r_getcwd GETCWD_P(buf, size)
{
  str_ret *r;
  int ret;
  int siz;

  if(!(r = runix_getwd_1(clnt))) {
    errno = EDEADRPC;
    return NULL;
  }

  if(r -> xerrno)
    errno = dec_errno(r -> xerrno);

  ret = r -> retcode;

  if(ret == -1) {
  cleanup:;
    XDR_FREE(xdr_str_ret, r);
    return NULL;
  }

  siz = strlen(r -> s) + 1;
  if((long)size >= 0 && siz > size) {
    errno == ERANGE;
    goto cleanup;
  }

  if(!buf) {
    buf = malloc((long)size >= 0 ? size : siz);
    if(!buf) goto cleanup;
  }

  memcpy(buf, r -> s, siz);
  XDR_FREE(xdr_str_ret, r);
  return buf;
}

GETWD_RET r_getwd GETWD_P(buf)
{
  return r_getcwd(buf, -1);
}

DUP_RET r_dup DUP_P(fd)
{
  int_ret *r;
  int ret;

  chk_fd(fd);

  rpc_call(ret, r, runix_dup_1(fd, clnt));

  XDR_FREE(xdr_int_ret, r);
  return ret;
}

TRUNCATE_RET r_truncate TRUNCATE_P(path, len)
{
  int_ret *r;
  int ret;

  chk_path(path);
  
  rpc_call(ret, r, runix_truncate_1((char *)path, len, clnt));

  XDR_FREE(xdr_int_ret, r);
  return ret;
}

FTRUNCATE_RET r_ftruncate FTRUNCATE_P(fd, len)
{
  int_ret *r;
  int ret;

  chk_fd(fd);

  rpc_call(ret, r, runix_ftruncate_1(fd, len, clnt));

  XDR_FREE(xdr_int_ret, r);
  return ret;
}

/*
 * Not really a remote  fork() - just a  wrapper around RPC call. With
 * current program  structure, some work has  to  be done by  files in
 * preload subdirectory.
 */

int r_fork(u_long handle)
{
  int ret;
  int_ret *r;

  rpc_call(ret, r, runix_fork_1(handle, clnt));

  XDR_FREE(xdr_int_ret, r);
  return ret;
}

UTIME_RET r_utime UTIME_P(path, buf)
{
  int ret;
  int_ret *r;
  struct rutimbuf ru, *pru;

  if(buf) {
    ru.actime = buf -> actime;
    ru.modtime = buf -> modtime;
    pru = &ru;
  } else
    pru = NULL;

  chk_path(path);
  rpc_call(ret, r, runix_utime_1(path, pru, clnt));
  
  XDR_FREE(xdr_int_ret, r);
  return ret;
}

MKNOD_RET r_mknod MKNOD_P(path, mode, dev)
{
  int ret;
  int_ret *r;

  chk_path(path);
  rpc_call(ret, r, runix_mknod_1(path, enc_mode(mode), dev, clnt));
  
  XDR_FREE(xdr_int_ret, r);
  return ret;
}

ACCESS_RET r_access ACCESS_P(path, mode)
{
  int ret;
  int_ret *r;

  chk_path(path);
  rpc_call(ret, r, runix_access_1(path, enc_access(mode), clnt));
  
  XDR_FREE(xdr_int_ret, r);
  return ret;
}

READV_RET r_readv READV_P(fd, vec, count)
{
  int cnt, xcnt;
  char *p;

  int i, len = 0;

  chk_fd(fd);
  chk_buf(vec);
  for(i = 0; i < count; i++) {
    chk_buf(vec[i].iov_base);
    len += vec[i].iov_len;
  }

  p = alloca(len);
  
  if((xcnt = r_read(fd, p, len)) == -1)
    return -1;

  for(cnt = xcnt; cnt; cnt -= len) {
    len = MIN(cnt, vec[i].iov_len);
    memcpy(vec[0].iov_base, p, len);
    p += len;
    vec++;
  }

  return xcnt;
}

WRITEV_RET r_writev WRITEV_P(fd, vec, count)
{
  char *p, *p0;

  int i, len = 0;

  chk_fd(fd);
  chk_buf(vec);
  for(i = 0; i < count; i++) {
    chk_buf(vec[i].iov_base);
    len += vec[i].iov_len;
  }

  p0 = p = alloca(len);
  
  for(i = 0; i < count; i++) {
    memcpy(p, vec[i].iov_base, vec[i].iov_len);
    p += vec[i].iov_len;
  }
  
  return r_write(fd, p0, len);
}

FSYNC_RET r_fsync FSYNC_P(fd)
{
  int_ret *r;
  int ret;

  chk_fd(fd);

  rpc_call(ret, r, runix_fsync_1(fd, clnt));

  XDR_FREE(xdr_int_ret, r);
  return ret;
}
