/*
 * Basic Sanity Checks
 * Author: Eric Vought - QLUE Consulting, Inc.
 *	Copyright (C) 1998 QLUE Consulting, Inc.
 * $Id: basic.c,v 3.0 1998/05/21 17:14:03 adfh Exp $
 *
 * Perform (very) basic sanity checking on the functions in the string
 * utilities package.
 */

/*
 * See the file "LICENSE.txt" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */

/* xsnprintf is never tested. */

#include <stdlib.h>
#include <tet_api.h>
#include <assert.h>
#include "qstring.h"

void (*tet_startup)() = NULL;
void (*tet_cleanup)() = NULL;

void tp1();
void tp2();
void tp3();
void tp4();
void tp5();
void tp6();
void tp7();
void tp8();
void tp9();

struct tet_testlist tet_testlist[] = { 
    {tp1, 1},
    {tp2, 2},
    {tp3, 3},
    {tp4, 4},
    {tp5, 5},
    {tp6, 6},
    {tp7, 7},
    {tp8, 8},
    {tp9, 9},
    {NULL, 0}
};

/* ----------------------------- xstrdup */

/* Copy a string and compare it to the original. */
void tp1()
{
    const char* orig = "This is the original string.";
    char* copy = NULL;

    /* Initialize */
    tet_printf("basic:%d xstrdup- copy a string.",
	       tet_thistest );

    /* Test */
    copy = xstrdup(orig);
    
    if (strcmp(orig, copy) == 0)
	tet_result(TET_PASS);
    else
	tet_result(TET_FAIL);

    /* Cleanup */
    free(copy);

} /* tp1 */

/* ----------------------------- QSTR_split */

/* Split a string and verify number of returned pieces. */
void tp2()
{
    const char* orig = "Apple Cobbler|Spitting Cobra|17459907|Perspicacity";
    char** list = NULL;
    int i = 0, num_elements = 0;

    /* Initialize */
    tet_printf("basic:%d QSTR_split- split a string and count pieces.",
	       tet_thistest );

    /* Test */

    num_elements = QSTR_split(orig, '|', &list);

    if (num_elements == 4)
    {
	tet_result(TET_PASS);
    } else {
	tet_result(TET_FAIL);
    }

    /* Cleanup */
    for(i = 0; i < num_elements; i++) free(list[i]);
    free(list);

} /* tp2 */

/* Split a string and verify identity. */
void tp3()
{
    const char* orig = "Apple Cobbler|Spitting Cobra|17459907|Perspicacity";
    char** list = NULL;
    int i = 0, num_elements = 0;

    /* Initialize */
    tet_printf("basic:%d QSTR_split- split a string and count pieces.",
	       tet_thistest );

    /* Test */

    num_elements = QSTR_split(orig, '|', &list);

    tet_result(TET_PASS);

    if (num_elements != 4) 
    {
	tet_result(TET_FAIL);
	goto CLEANUP;
    }

    if (!( (strcmp(list[0], "Apple Cobbler") == 0)
	   && (strcmp(list[1], "Spitting Cobra") == 0)
	   && (strcmp(list[2], "17459907") == 0)
	   && (strcmp(list[3], "Perspicacity") == 0) ))
    {
	tet_result(TET_FAIL);
    }

 CLEANUP:

    for(i = 0; i < num_elements; i++) free(list[i]);
    free(list);

} /* tp3 */

/* ----------------------------- QSTR_rspn */

/* Count the whitespace at the end of a string. */
void tp4()
{
    const char* whitespace = " \t\n\r";

    const char* searchstr = " 378,463.721\t  \n";
    int num_spaces = 0;

    /* Initialize */
    tet_printf( "basic:%d QSTR_rspn- count whitespace at the end of a string.",
	       tet_thistest );

    /* Test */

    num_spaces = QSTR_rspn(searchstr, whitespace);

    if (num_spaces == 4)
    {
	tet_result(TET_PASS);
    } else {
	tet_result(TET_FAIL);
    }

    /* Cleanup */

} /* tp4 */

/* ----------------------------- QSTR_rcspn */

/* Count the non-whitespace at the end of a string. */
void tp5()
{
    const char* whitespace = " \t\n\r";

    const char* searchstr = "  378,463.721\t  890,373.37";
    int num_non_spaces = 0;

    /* Initialize */
    tet_printf( "basic:%d QSTR_rcspn- count non-whitespace at end of string.",
	       tet_thistest );

    /* Test */

    num_non_spaces = QSTR_rcspn(searchstr, whitespace);

    if (num_non_spaces == 10)
    {
	tet_result(TET_PASS);
    } else {
	tet_result(TET_FAIL);
    }

    /* Cleanup */

} /* tp5 */

/* ----------------------------- QSTR_triml */

/* Remove the whitepace from a string and check.. */
void tp6()
{
    const char* whitespace = " \t\n\r";
    const char* trimstr = "\t\t    378,463.721\t\t    \n";
    char* result = NULL;

    /* Initialize */
    tet_printf( "basic:%d QSTR_triml- remove leading whitespace.",
	       tet_thistest );

    /* Test */

    result = QSTR_triml(trimstr, whitespace);

    if (strcmp(result, "378,463.721\t\t    \n") == 0)
    {
	tet_result(TET_PASS);
    } else {
	tet_result(TET_FAIL);
    }

    /* Cleanup */

    free(result);
} /* tp6 */

/* ----------------------------- QSTR_trimr */

/* Remove the whitepace from a string and check.. */
void tp7()
{
    const char* whitespace = " \t\n\r";
    const char* trimstr = "\t\t    378,463.721\t\t    \n";
    char* result = NULL;

    /* Initialize */
    tet_printf( "basic:%d QSTR_triml- remove leading whitespace.",
	       tet_thistest );

    /* Test */

    result = QSTR_trimr(trimstr, whitespace);

    if (strcmp(result, "\t\t    378,463.721") == 0)
    {
	tet_result(TET_PASS);
    } else {
	tet_result(TET_FAIL);
    }

    /* Cleanup */

    free(result);
} /* tp7 */

/* ----------------------------- QSTR_trim */

/* Remove the whitepace from a string and check.. */
void tp8()
{
    const char* whitespace = " \t\n\r";
    const char* trimstr = "\t\t    378,463.721\t\t    \n";
    char* result = NULL;

    /* Initialize */
    tet_printf( "basic:%d QSTR_triml- remove leading whitespace.",
	       tet_thistest );

    /* Test */

    result = QSTR_trim(trimstr, whitespace);

    if (strcmp(result, "378,463.721") == 0)
    {
	tet_result(TET_PASS);
    } else {
	tet_result(TET_FAIL);
    }

    /* Cleanup */
    free(result);
} /* tp8 */

/* ----------------------------- QSTR_glue */

/* Combine four strings and check result */
void tp9()
{
    const char* string1 = "abcd";
    const char* string2 = "efghij";
    const char* string3 = "klmno";
    const char* string4 = "pqrstuvwxyz";

    char* result = NULL;

    /* Initialize */
    tet_printf( "basic:%d QSTR_glue- combine four strings.",
	       tet_thistest );

    /* Test */

    result = QSTR_glue(4, string1, string2, string3, string4);

    if (strcmp(result, "abcdefghijklmnopqrstuvwxyz") == 0)
    {
	tet_result(TET_PASS);
    } else {
	tet_result(TET_FAIL);
    }

    /* Cleanup */
    free(result);
} /* tp9 */
