/*
 * Standard Tests
 * Author: Eric Vought - QLUE Consulting, Inc.
 *	Copyright (C) 1998 QLUE Consulting, Inc.
 * $Id: standard.c,v 3.0 1998/05/21 17:14:02 adfh Exp $
 *
 * Verify that the hashtable behaves reasonably for correct input and
 * normal conditions.
 */

/*
 * See the file "LICENSE.txt" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */

#include <stdlib.h>
#include <tet_api.h>
#include <assert.h>

/* Needed for tmpnam */
#include <stdio.h>

#include "qstring.h"
#include "hashtable.h"
#include "test_data.h"

void (*tet_startup)() = NULL;
void (*tet_cleanup)() = NULL;

void tp1();
void tp2();
void tp3();
void tp4();
void tp5();
void tp6();
void tp7();
void tp8();
void tp9();
void tp10();
void tp11();
void tp12();
void tp13();
void tp14();
void tp15();

struct tet_testlist tet_testlist[] = {
    {tp1, 1},
    {tp2, 2},
    {tp3, 3},
    {tp4, 4},
    {tp5, 5},
    {tp6, 6},
    {tp7, 7},
    {tp8, 8},
    {tp9, 9},
    {tp10, 10},
    {tp11, 11},
    {tp12, 12},
    {tp13, 13},
    {tp14, 14},
    {tp15, 15},
    {NULL, 0}
}; /* tet_testlist */

/*
 * Initialize and fill with some values using HASH_add.
 * Make sure there is enough space that no resize will occur.
 */
void tp1()
{
    Hash_Table hash;
    int i;

    /* Initialize */
    tet_printf("standard:%d Initialize and fill using HASH_add.",
	       tet_thistest );

    hash = HASH_new(64);

    /* Test */

    /* Add 30 values to the hash. */
    for(i = 0; i < (num_strings - 1); i++)
    {
	HASH_add(hash, some_strings[i], some_strings[i+1]);
	HASH_add(hash, tmpnam(NULL), tmpnam(NULL));
    }
    
    if (hash->count == 30)
    {
	tet_result(TET_PASS);
    } else {
	tet_result(TET_FAIL);
    }

    HASH_check_invariants(hash);

    /* Cleanup */
    HASH_delete(hash);

} /* tp1 */

/*
 * Initialize and fill with some values using HASH_set.
 * Make sure there is enough space that no resize will occur.
 */
void tp2()
{
    Hash_Table hash;
    int i;

    /* Initialize */
    tet_printf("standard:%d Initialize and fill using HASH_set.",
	       tet_thistest );

    hash = HASH_new(64);

    /* Test */

    /* Add 30 values to the hash. */
    for(i = 0; i < (num_strings - 1); i++)
    {
	HASH_set(hash, some_strings[i], some_strings[i+1]);
	HASH_set(hash, tmpnam(NULL), tmpnam(NULL));
    }
    
    if (hash->count == 30)
    {
	tet_result(TET_PASS);
    } else {
	tet_result(TET_FAIL);
    }

    HASH_check_invariants(hash);

    /* Cleanup */
    HASH_delete(hash);

} /* tp2 */

/*
 * Verify that values stored by set are there.
 * Make sure there is enough space that no resize will occur.
 */
void tp3()
{
    Hash_Table hash;
    int i;

    /* Initialize */
    tet_printf("standard:%d Verify that values stored by set are there.",
	       tet_thistest );

    hash = HASH_new(64);

    /* Test */

    /* Add 30 values to the hash. */
    for(i = 0; i < (num_strings - 1); i++)
    {
	HASH_set(hash, some_strings[i], some_strings[i+1]);
	HASH_set(hash, tmpnam(NULL), tmpnam(NULL));
    }

    for(i = 0; i < 15; i++)
    {
	if (!HASH_exists(hash, some_strings[i])) tet_result(TET_FAIL);
    }

    tet_result(TET_PASS);
    
    HASH_check_invariants(hash);

    /* Cleanup */
    HASH_delete(hash);

} /* tp3 */


/*
 * Verify that values stored by set are correct.
 * Make sure there is enough space that no resize will occur.
 */
void tp4()
{
    Hash_Table hash;
    int i;

    /* Initialize */
    tet_printf("standard:%d Verify that values stored by set are correct.",
	       tet_thistest );

    hash = HASH_new(64);

    /* Test */

    /* Add 30 values to the hash. */
    for(i = 0; i < (num_strings - 1); i++)
    {
	HASH_set(hash, some_strings[i], some_strings[i+1]);
	HASH_set(hash, tmpnam(NULL), tmpnam(NULL));
    }

    for(i = 0; i < (num_strings - 1); i++)
    {
	if (strcmp(HASH_fetch(hash, some_strings[i]), some_strings[i+1])
		!= 0 )
	{
	    tet_result(TET_FAIL);
	}
    }

    tet_result(TET_PASS);
    
    HASH_check_invariants(hash);

    /* Cleanup */
    HASH_delete(hash);

} /* tp4 */

/*
 * Verify that values stored by add are there.
 * Make sure there is enough space that no resize will occur.
 */
void tp5()
{
    Hash_Table hash;
    int i;

    /* Initialize */
    tet_printf("standard:%d Verify that values stored by set are there.",
	       tet_thistest );

    hash = HASH_new(64);

    /* Test */

    /* Add 30 values to the hash. */
    for(i = 0; i < (num_strings - 1); i++)
    {
	HASH_add(hash, some_strings[i], some_strings[i+1]);
	HASH_add(hash, tmpnam(NULL), tmpnam(NULL));
    }

    for(i = 0; i < (num_strings - 1); i++)
    {
	if (!HASH_exists(hash, some_strings[i])) tet_result(TET_FAIL);
    }

    tet_result(TET_PASS);
    
    HASH_check_invariants(hash);

    /* Cleanup */
    HASH_delete(hash);

} /* tp5 */


/*
 * Verify that values stored by set are correct.
 * Make sure there is enough space that no resize will occur.
 */
void tp6()
{
    Hash_Table hash;
    int i;

    /* Initialize */
    tet_printf("standard:%d Verify that values stored by set are correct.",
	       tet_thistest );

    hash = HASH_new(64);

    /* Test */

    /* Add 30 values to the hash. */
    for(i = 0; i < (num_strings - 1); i++)
    {
	HASH_add(hash, some_strings[i], some_strings[i+1]);
	HASH_add(hash, tmpnam(NULL), tmpnam(NULL));
    }

    for(i = 0; i < 15; i++)
    {
	if (strcmp(HASH_fetch(hash, some_strings[i]), some_strings[i+1])
		!= 0 )
	{
	    tet_result(TET_FAIL);
	}
    }

    tet_result(TET_PASS);
    
    HASH_check_invariants(hash);

    /* Cleanup */
    HASH_delete(hash);

} /* tp6 */

/*
 * Fill with non-unique values using HASH_add.
 */
void tp7()
{
    Hash_Table hash;
    int i,j;

    /* Initialize */
    tet_printf("standard:%d Fill with non-unique values using HASH_add.",
	       tet_thistest );

    hash = HASH_new(300);

    /* Test */

    /* Add 300 values to the hash, 165 unique. */
    for(i = 0; i < 10; i++)
    {
	for(j = 0; j < (num_strings - 1); j++)
	{
	    HASH_add(hash, some_strings[i], some_strings[i+1]);
	    HASH_add(hash, tmpnam(NULL), tmpnam(NULL));
	}
    }
    
    if (hash->count == 300)
    {
	tet_result(TET_PASS);
    } else {
	tet_result(TET_FAIL);
    }

    HASH_check_invariants(hash);

    /* Cleanup */
    HASH_delete(hash);

} /* tp7 */


/*
 * Fill with non-unique values using HASH_set.
 */
void tp8()
{
    Hash_Table hash;
    int i,j;

    /* Initialize */
    tet_printf("standard:%d Fill with non-unique values using HASH_set.",
	       tet_thistest );

    hash = HASH_new(200);

    /* Test */

    /* Add 300 values to the hash, 165 unique. */
    for(i = 0; i < 10; i++)
    {
	for(j = 0; j < (num_strings - 1); j++)
	{
	    HASH_set(hash, some_strings[j], some_strings[j+1]);
	    HASH_set(hash, tmpnam(NULL), tmpnam(NULL));
	}
    }
    
    if (hash->count == 165)
    {
	tet_result(TET_PASS);
    } else {
	tet_result(TET_FAIL);
    }

    HASH_check_invariants(hash);

    /* Cleanup */
    HASH_delete(hash);

} /* tp8 */

/*
 * Non-unique values using HASH_add and resize to remove dups.
 */
void tp9()
{
    Hash_Table hash;
    int i,j;

    /* Initialize */
    tet_printf("standard:%d Non-unique values using HASH_add and resize."
	       "to remove dups",
	       tet_thistest );

    hash = HASH_new(300);

    /* Test */

    /* Add 300 values to the hash, 165 unique. */
    for(i = 0; i < 10; i++)
    {
	for(j = 0; j < (num_strings - 1); j++)
	{
	    HASH_add(hash, some_strings[j], some_strings[j+1]);
	    HASH_add(hash, tmpnam(NULL), tmpnam(NULL));
	}
    }
    
    HASH_resize(hash);

    if (hash->count == 165)
    {
	tet_result(TET_PASS);
    } else {
	tet_result(TET_FAIL);
    }

    HASH_check_invariants(hash);

    /* Cleanup */
    HASH_delete(hash);

} /* tp9 */

/*
 * Verify that add keeps last entered.
 */
void tp10()
{
    Hash_Table hash;
    int i;

    /* Initialize */
    tet_printf("standard:%d Verify that add keeps last entered",
	       tet_thistest );

    hash = HASH_new(30);

    /* Test */

    /* Add 16 identical keys to the hash. */
    for(i = 0; i < num_strings; i++)
    {
	HASH_add(hash, some_strings[0], some_strings[i]);
    }
    
    if (strcmp(HASH_fetch(hash, some_strings[0]), some_strings[15])
		== 0 )
    {
	tet_result(TET_PASS);
    } else {
	tet_result(TET_FAIL);
    }

    HASH_check_invariants(hash);

    /* Cleanup */
    HASH_delete(hash);

} /* tp10 */

/*
 * Verify that add keeps last entered (w/ resize).
 */
void tp11()
{
    Hash_Table hash;
    int i;

    /* Initialize */
    tet_printf("standard:%d Verify that add keeps last entered (w/ resize)",
	       tet_thistest );

    hash = HASH_new(30);

    /* Test */

    /* Add 16 identical keys to the hash. */
    for(i = 0; i < num_strings; i++)
    {
	HASH_add(hash, some_strings[0], some_strings[i]);
    }

    HASH_resize(hash);
    
    if (strcmp(HASH_fetch(hash, some_strings[0]), some_strings[15])
		== 0 )
    {
	tet_result(TET_PASS);
    } else {
	tet_result(TET_FAIL);
    }

    HASH_check_invariants(hash);

    /* Cleanup */
    HASH_delete(hash);

} /* tp11 */

/*
 * Insert strings, call clear, and check.
 */
void tp12()
{
    Hash_Table hash;
    int i;

    /* Initialize */
    tet_printf("standard:%d Insert strings, call clear, and check",
	       tet_thistest );

    hash = HASH_new(30);

    /* Test */

    /* Add 16 keys to the hash. */
    for(i = 0; i < num_strings; i++)
    {
	HASH_add(hash, some_strings[i], some_strings[0]);
    }

    HASH_clear(hash);
    
    for(i = 0; i < num_strings; i++)
    {
	if (HASH_exists(hash, some_strings[i])) tet_result(TET_FAIL);
    }

    tet_result(TET_PASS);

    HASH_check_invariants(hash);

    /* Cleanup */
    HASH_delete(hash);

} /* tp12 */


/*
 * Insert strings, remove them all, and check.
 */
void tp13()
{
    Hash_Table hash;
    int i;

    /* Initialize */
    tet_printf("standard:%d Insert strings, remove, and check",
	       tet_thistest );

    hash = HASH_new(30);

    /* Test */

    /* Add 16 keys to the hash. */
    for(i = 0; i < num_strings; i++)
    {
	HASH_add(hash, some_strings[i], some_strings[0]);
    }

    /* remove them each individually */
    for(i = 0; i < num_strings; i++)
    {
	HASH_remove(hash, some_strings[i]);
    }

    /* make sure that they are not in the table */
    for(i = 0; i < num_strings; i++)
    {
	if (HASH_exists(hash, some_strings[i])) tet_result(TET_FAIL);
    }

    tet_result(TET_PASS);

    HASH_check_invariants(hash);

    /* Cleanup */
    HASH_delete(hash);

} /* tp13 */

/*
 * Add the same key several times, remove it once, and check.
 */
void tp14()
{
    Hash_Table hash;
    int i;

    /* Initialize */
    tet_printf("standard:%d Add duplicates, remove once, and check.",
	       tet_thistest );

    hash = HASH_new(30);

    /* Test */

    /* Add the same key 16 times. */
    for(i = 0; i < num_strings; i++)
    {
	HASH_add(hash, some_strings[0], some_strings[i]);
    }

    /* force a resize to remove dups */
    HASH_resize(hash);

    HASH_remove(hash, some_strings[0]);

    /* make sure that it is not in the table */
    if (HASH_exists(hash, some_strings[0]))
    {
	tet_result(TET_FAIL);
    } else {
	tet_result(TET_PASS);
    }

    HASH_check_invariants(hash);

    /* Cleanup */
    HASH_delete(hash);

} /* tp14 */


/*
 * Add values using set, forcing a resize, and check.
 */
void tp15()
{
    Hash_Table hash;
    int i;

    /* Initialize */
    tet_printf("standard:%d Add values using set, forcing a resize.",
	       tet_thistest );

    hash = HASH_new(5);

    /* Test */

    /* Add 30 values to the hash. */
    for(i = 0; i < (num_strings - 1); i++)
    {
	HASH_set(hash, some_strings[i], some_strings[i+1]);
	HASH_set(hash, tmpnam(NULL), tmpnam(NULL));
    }

    /* make sure the values are still there. */
    for(i = 0; i < (num_strings - 1); i++)
    {
	if (strcmp(HASH_fetch(hash, some_strings[i]), some_strings[i+1])
		!= 0 )
	{
	    tet_result(TET_FAIL);
	}
    }
    tet_result(TET_PASS);

    HASH_check_invariants(hash);

    /* Cleanup */
    HASH_delete(hash);

} /* tp15 */

