/*
 * Resizing Tests
 * Author: Eric Vought - QLUE Consulting, Inc.
 *	Copyright (C) 1998 QLUE Consulting, Inc.
 * $Id: resize.c,v 3.0 1998/05/21 17:14:02 adfh Exp $
 *
 * Verify that the hashtable resizes when and how it is supposed to.
 * These tests are very likely to break if the resizing behavior of the
 * class every changes and were therefore stuck in a separate file.
 */

/*
 * See the file "LICENSE.txt" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */

#include <stdlib.h>
#include <tet_api.h>
#include <assert.h>

/* Needed for tmpnam */
#include <stdio.h>

#include <qstring.h>
#include <hashtable.h>
#include "test_data.h"

void (*tet_startup)() = NULL;
void (*tet_cleanup)() = NULL;

void tp1();
void tp2();
void tp3();
void tp4();
void tp5();
void tp6();
void tp7();
void tp8();
void tp9();

struct tet_testlist tet_testlist[] = {
    {tp1, 1},
    {tp2, 2},
    {tp3, 3},
    {tp4, 4},
    {tp5, 5},
    {tp6, 6},
    {tp7, 7},
    {tp8, 8},
    {tp9, 9},
    {NULL, 0}
}; /* tet_testlist */

/*
 * Overfill with set and make sure that the resize occurs.
 */
void tp1()
{
    Hash_Table hash;
    int i;

    /* Initialize */
    tet_printf("resize:%d Overfill with set and make sure that the"
	       " resize occurs.",
	       tet_thistest );

    hash = HASH_new(10);

    /* Test */

    /* Add 30 values to the hash. */
    for(i = 0; i < (num_strings - 1); i++)
    {
	HASH_set(hash, some_strings[i], some_strings[i+1]);
	HASH_set(hash, tmpnam(NULL), tmpnam(NULL));
    }
    
    if (hash->size == 20)
    {
	tet_result(TET_PASS);
    } else {
	tet_result(TET_FAIL);
    }

    HASH_check_invariants(hash);

    /* Cleanup */
    HASH_delete(hash);

} /* tp1 */


/*
 * Overfill with set and check resize threshold.
 */
void tp2()
{
    Hash_Table hash;
    int i;

    /* Initialize */
    tet_printf("resize:%d Overfill with set and check resize threshold.",
	       tet_thistest );

    hash = HASH_new(10);

    /* Test */

    /* Add 30 values to the hash. */
    for(i = 0; i < (num_strings - 1); i++)
    {
	HASH_set(hash, some_strings[i], some_strings[i+1]);
	HASH_set(hash, tmpnam(NULL), tmpnam(NULL));
    }
    
    if (hash->resize_threshold == 40)
    {
	tet_result(TET_PASS);
    } else {
	tet_result(TET_FAIL);
    }

    HASH_check_invariants(hash);

    /* Cleanup */
    HASH_delete(hash);

} /* tp2 */


/*
 * Overfill with add and make sure that no resize occurs.
 */
void tp3()
{
    Hash_Table hash;
    int i;

    /* Initialize */
    tet_printf("resize:%d Overfill with add and make sure that no"
	       " resize occurs.",
	       tet_thistest );

    hash = HASH_new(10);

    /* Test */

    /* Add 30 values to the hash. */
    for(i = 0; i < (num_strings - 1); i++)
    {
	HASH_add(hash, some_strings[i], some_strings[i+1]);
	HASH_add(hash, tmpnam(NULL), tmpnam(NULL));
    }
    
    if (hash->size == 10)
    {
	tet_result(TET_PASS);
    } else {
	tet_result(TET_FAIL);
    }

    HASH_check_invariants(hash);

    /* Cleanup */
    HASH_delete(hash);

} /* tp3 */

/*
 * Overfill with add, force resize, and check.
 */
void tp4()
{
    Hash_Table hash;
    int i;

    /* Initialize */
    tet_printf("resize:%d Overfill with add, force resize, and check.",
	       tet_thistest );

    hash = HASH_new(10);

    /* Test */

    /* Add 30 values to the hash. */
    for(i = 0; i < (num_strings - 1); i++)
    {
	HASH_add(hash, some_strings[i], some_strings[i+1]);
	HASH_add(hash, tmpnam(NULL), tmpnam(NULL));
    }

    HASH_resize(hash);
    
    if (hash->size == 30)
    {
	tet_result(TET_PASS);
    } else {
	tet_result(TET_FAIL);
    }

    HASH_check_invariants(hash);

    /* Cleanup */
    HASH_delete(hash);

} /* tp4 */

/*
 * Overfill with add, call exists, and check.
 */
void tp5()
{
    Hash_Table hash;
    int i;

    /* Initialize */
    tet_printf("resize:%d Overfill with add, call exists, and check.",
	       tet_thistest );

    hash = HASH_new(10);

    /* Test */

    /* Add 30 values to the hash. */
    for(i = 0; i < (num_strings - 1); i++)
    {
	HASH_add(hash, some_strings[i], some_strings[i+1]);
	HASH_add(hash, tmpnam(NULL), tmpnam(NULL));
    }

    /* Cause a resize as a side effect */
    HASH_exists(hash, some_strings[0]);
    
    if (hash->size == 30)
    {
	tet_result(TET_PASS);
    } else {
	tet_result(TET_FAIL);
    }

    HASH_check_invariants(hash);

    /* Cleanup */
    HASH_delete(hash);

} /* tp5 */

/*
 * Overfill with add, call set, and check.
 */
void tp6()
{
    Hash_Table hash;
    int i;

    /* Initialize */
    tet_printf("resize:%d Overfill with add, call set, and check.",
	       tet_thistest );

    hash = HASH_new(10);

    /* Test */

    /* Add 30 values to the hash. */
    for(i = 0; i < (num_strings - 1); i++)
    {
	HASH_add(hash, some_strings[i], some_strings[i+1]);
	HASH_add(hash, tmpnam(NULL), tmpnam(NULL));
    }

    /* Cause a resize as a side effect */
    HASH_set(hash, "some_random_key", "some_random_value");
    
    if (hash->size == 30)
    {
	tet_result(TET_PASS);
    } else {
	tet_result(TET_FAIL);
    }

    HASH_check_invariants(hash);

    /* Cleanup */
    HASH_delete(hash);

} /* tp6 */

/*
 * Overfill with add, call remove, don't resize.
 */
void tp7()
{
    Hash_Table hash;
    int i;

    /* Initialize */
    tet_printf("resize:%d Overfill with add, call remove, and check.",
	       tet_thistest );

    hash = HASH_new(10);

    /* Test */

    /* Add 30 values to the hash. */
    for(i = 0; i < (num_strings - 1); i++)
    {
	HASH_add(hash, some_strings[i], some_strings[i+1]);
	HASH_add(hash, tmpnam(NULL), tmpnam(NULL));
    }

    /* Doesn't cause a resize as a side effect */
    HASH_remove(hash, some_strings[3]);

    if (hash->size == 10)
    {
	tet_result(TET_PASS);
    } else {
	tet_result(TET_FAIL);
    }

    HASH_check_invariants(hash);

    /* Cleanup */
    HASH_delete(hash);

} /* tp7 */

/*
 * Overfill with add, call fetch, and check.
 */
void tp8()
{
    Hash_Table hash;
    int i;

    /* Initialize */
    tet_printf("resize:%d Overfill with add, call fetch, and check.",
	       tet_thistest );

    hash = HASH_new(10);

    /* Test */

    /* Add 30 values to the hash. */
    for(i = 0; i < (num_strings - 1); i++)
    {
	HASH_add(hash, some_strings[i], some_strings[i+1]);
	HASH_add(hash, tmpnam(NULL), tmpnam(NULL));
    }

    /* Cause a resize as a side effect */
    HASH_fetch(hash, some_strings[2]);
    
    if (hash->size == 30)
    {
	tet_result(TET_PASS);
    } else {
	tet_result(TET_FAIL);
    }

    HASH_check_invariants(hash);

    /* Cleanup */
    HASH_delete(hash);

} /* tp8 */

/*
 * Underfill with add, use resize to shrink.
 */
void tp9()
{
    Hash_Table hash;
    int i;

    /* Initialize */
    tet_printf("resize:%d Underfill with add, use resize to shrink.",
	       tet_thistest );

    hash = HASH_new(60);

    /* Test */

    /* Add 30 values to the hash. */
    for(i = 0; i < (num_strings - 1); i++)
    {
	HASH_add(hash, some_strings[i], some_strings[i+1]);
	HASH_add(hash, tmpnam(NULL), tmpnam(NULL));
    }

    /* Force a resize and shrink the hash */
    HASH_resize(hash);
    
    if (hash->size == 30)
    {
	tet_result(TET_PASS);
    } else {
	tet_result(TET_FAIL);
    }

    HASH_check_invariants(hash);

    /* Cleanup */
    HASH_delete(hash);

} /* tp9 */
