/*
 * Exception Tests
 * Author: Eric Vought - QLUE Consulting, Inc.
 *	Copyright (C) 1998 QLUE Consulting, Inc.
 * $Id: exception_tests.c,v 3.0 1998/05/21 17:14:02 adfh Exp $
 *
 * Verify that bad input and defined error conditions are handled
 * properly. Because some of these tests will terminate in an abort
 * if they work *correctly*, *always* check the results of these tests!
 */

/*
 * See the file "LICENSE.txt" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */

#include <stdlib.h>
#include <tet_api.h>
#include <assert.h>

/* Needed for tmpnam */
#include <stdio.h>

#include <qstring.h>
#include <hashtable.h>
#include "test_data.h"

void (*tet_startup)() = NULL;
void (*tet_cleanup)() = NULL;

void tp1();
void tp2();
void tp3();
void tp4();
void tp5();
void tp6();
void tp7();
void tp8();
void tp9();

struct tet_testlist tet_testlist[] = { 
    {tp1, 1},
    {tp2, 2},
    {tp3, 3},
    {tp4, 4},
    {tp5, 5},
    {tp6, 6},
    {tp7, 7},
    {tp8, 8},
    {tp9, 9},
    {NULL,0}
};


/* Pass a NULL as the value. Should accept silently. */
void tp1()
{
    Hash_Table hash;

    /* Initialize */
    tet_printf("Pass a NULL as the value. Should accept silently.",
	       tet_thistest);

    hash = HASH_new(16);

    /* Test */

    HASH_set(hash, "Foo", NULL);

    if (!HASH_exists(hash, "Foo")) tet_result(TET_FAIL);
    if (HASH_fetch(hash, "Foo") != NULL) tet_result(TET_FAIL);

    tet_result(TET_PASS);

    HASH_check_invariants(hash);

    /* Cleanup */
    HASH_delete(hash);

} /* tp1 */

/* set: Pass a NULL as the key. Should abort on assertion. */
void tp2()
{
    Hash_Table hash;

    /* Initialize */
    tet_printf("set: Pass a NULL as the key. Should abort.",
	       tet_thistest);

    hash = HASH_new(16);

    /* Test */

    HASH_set(hash, NULL, NULL);

    tet_result(TET_FAIL);

    HASH_check_invariants(hash);

    /* Cleanup */
    HASH_delete(hash);

} /* tp2 */

/* add: Pass a NULL as the key. Should abort on assertion. */
void tp3()
{
    Hash_Table hash;

    /* Initialize */
    tet_printf("add: Pass a NULL as the key. Should abort.",
	       tet_thistest);

    hash = HASH_new(16);

    /* Test */

    HASH_add(hash, NULL, NULL);

    tet_result(TET_FAIL);

    HASH_check_invariants(hash);

    /* Cleanup */
    HASH_delete(hash);

} /* tp3 */

/* exists: Pass a NULL as the key. Should abort on assertion. */
void tp4()
{
    Hash_Table hash;

    /* Initialize */
    tet_printf("exists: Pass a NULL as the key. Should abort.",
	       tet_thistest);

    hash = HASH_new(16);

    /* Test */

    HASH_exists(hash, NULL);

    tet_result(TET_FAIL);

    HASH_check_invariants(hash);

    /* Cleanup */
    HASH_delete(hash);

} /* tp4 */

/* remove: Pass a NULL as the key. Should abort on assertion. */
void tp5()
{
    Hash_Table hash;

    /* Initialize */
    tet_printf("remove: Pass a NULL as the key. Should abort.",
	       tet_thistest);

    hash = HASH_new(16);

    /* Test */

    HASH_remove(hash, NULL);

    tet_result(TET_FAIL);

    HASH_check_invariants(hash);

    /* Cleanup */
    HASH_delete(hash);

} /* tp5 */

/* fetch: Pass a NULL as the key. Should abort on assertion. */
void tp6()
{
    Hash_Table hash;

    /* Initialize */
    tet_printf("fetch: Pass a NULL as the key. Should abort.",
	       tet_thistest);

    hash = HASH_new(16);

    /* Test */

    HASH_fetch(hash, NULL);

    tet_result(TET_FAIL);

    HASH_check_invariants(hash);

    /* Cleanup */
    HASH_delete(hash);

} /* tp6 */

/* Initialize a hash with size zero. Should use default size. */
void tp7()
{
    Hash_Table hash;

    /* Initialize */
    tet_printf("remove: Initialize with size zero. Should use default.",
	       tet_thistest);

    hash = HASH_new(0);

    /* Test */

    if (hash->size == HASH_DEFAULT_INITIAL_SIZE)
    {
	tet_result(TET_PASS);
    } else {
	tet_result(TET_FAIL);
    }

    HASH_check_invariants(hash);

    /* Cleanup */
    HASH_delete(hash);

} /* tp7 */

/* Remove non-existant node. Should fail silently. */
void tp8()
{
    Hash_Table hash;

    /* Initialize */
    tet_printf("Remove non-existant node. Should fail silently.",
	       tet_thistest);

    hash = HASH_new(16);

    /* Test */

    HASH_remove(hash, "foo");

    tet_result(TET_PASS);

    HASH_check_invariants(hash);

    /* Cleanup */
    HASH_delete(hash);

} /* tp8 */

/* Fetch non-existant key. Should return NULL. */
void tp9()
{
    Hash_Table hash;

    /* Initialize */
    tet_printf("Fetch non-existant key. Should return NULL.",
	       tet_thistest);

    hash = HASH_new(16);

    /* Test */

    if (HASH_fetch(hash, "foo") == NULL)
    {
	tet_result(TET_PASS);
    } else {
	tet_result(TET_FAIL);
    }

    HASH_check_invariants(hash);

    /* Cleanup */
    HASH_delete(hash);

} /* tp9 */


