/*
 * String Utilities (QCI)
 * Author: Eric Vought
 *	Copyright (C) 1998 QLUE Consulting, Inc.
 * $Id: qstring.h,v 3.2 1998/06/29 00:25:50 adfh Exp $
 *
 *  String utilities package - safe versions of some standard C library
 *      functions and additional string manipulation routines.
 */

/*
 * See the file "LICENSE.txt" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */

#ifndef _QSTRING_H
#define _QSTRING_H

#include <qci_util_config.h>
#include <unistd.h>

/* snprintf replacement */
#if !defined(HAVE_SNPRINTF) || defined(_POSIX_SOURCE)
int snprintf(char* str, size_t n, const char* format, ...);
#endif

/* asprintf replacement */
#ifndef HAVE_ASPRINTF
int vasprintf(char** buffer, const char* format, va_list ap);
int asprintf(char** buffer, const char* format, ...);
#endif

/*
 * Same as strncpy, except that it always null-terminates the dest string.
 */
char* xstrncpy(char* dest, const char* src, size_t n);

/* strdup replacement for primitive systems */
#ifndef HAVE_STRDUP
char* strdup(const char* s);
#endif

/* Safe strdup; exits on memory allocation failure. */
char* xstrdup(const char* s);

/*
 * Substitute for strcasecmp on systems where it's not available
 */
#if !defined(HAVE_STRCASECMP) || defined(_POSIX_SOURCE)
int strcasecmp(const char* a, const char* b);
int strncasecmp(const char* a, const char* b, size_t n);
#endif

int QSTR_split(const char* string, int delim, char*** list);

/* Substitute for strstr on systems where it's not available. */
#ifndef HAVE_STRSTR
char* strstr(const char *haystack, const char *needle);
#endif

/* Substitute for strspn/strcspn on systems where it's not available. */
#ifndef HAVE_STRSPN
size_t strspn(const char* s, const char* accept);
size_t strcspn(const char* s, const char* reject);
#endif

/*
 * Return the number of characters that form an unbroken sequence of
 * characters found in accept, staring from the end of the string.
 * See strspn(3).
 */
size_t QSTR_rspn(const char* string, const char* accept);

/*
 * Return the number of characters that form an unbroken sequence of
 * characters not found in reject, staring from the end of the string.
 * See strcspn(3).
 */
size_t QSTR_rcspn(const char* string, const char* reject);

/*
 * Return a copy of string after removing all characters on the left side
 * that are found in trimchars.
 */
char* QSTR_triml(const char* string, const char* trimchars);

/*
 * Return a copy of string after removing all characters on the right side
 * that are found in trimchars.
 */
char* QSTR_trimr(const char* string, const char* trimchars);

/*
 * Return a copy of string after removing all characters on the right side
 * and the left side that are found in trimchars.
 */
char* QSTR_trim(const char* string, const char* trimchars);

/*
 * DEPRECATED: Use asprintf instead...
 * Paste *num* strings together in a newly allocated buffer.
 * Returns a pointer to the newly allocated space.
 */
char* QSTR_glue(int num, ...);

/*
 * Returns malloc()ed upper-case version of `string'.
 */
char* QSTR_toupper(const char* string);
    
#endif /* ndef _QSTRING_H */
