/*
 * String Utilities (QCI)
 * Author: Eric Vought
 *	Copyright (C) 1998 QLUE Consulting, Inc.
 * $Id: qstring.c,v 3.2 1998/08/22 18:26:16 adfh Exp $
 *
 *  String utilities package - safe versions of some standard C library
 *      functions and additional string manipulation routines.
 */

/*
 * See the file "LICENSE.txt" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */

#include "qci_util_config.h"
#ifndef _POSIX_SOURCE
#define _POSIX_SOURCE 1
#endif

#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <stdarg.h>
#include <ctype.h>

#include "qstring.h"
#include "qci_util.h"


/*
 * Same as strncpy, except that it always null-terminates the dest string.
 */
char* xstrncpy(char* dest, const char* src, size_t n)
{
    strncpy(dest, src, n);
    dest[n-1] = '\0';

    return dest;
}

/*
 * Function: xstrdup
 * Description: Safe strdup
 */
char* xstrdup(const char* s)
{
    char* foo;

    if (!s)
	return NULL;

    foo = (char*) xmalloc(strlen(s)+1);
    strcpy(foo, s);

    return foo;
} /* xstrdup */

int QSTR_split(const char* string, int delim, char*** list)
{
    /* There's always at least one element */
    int num_elements = 1;
    const char* index = NULL;
    const char* next_delim = NULL;
    int element = 0;
    int length = 0;

    assert(string != NULL);

    index = strchr(string, delim);
    while(index != NULL)
    {
	/* increment index so it points past the delimiter. */
	index++;
	num_elements++;
	index = strchr(++index, delim);
    }

    *list = xcalloc(num_elements, sizeof(char*));
    
    index = string;
    element = 0;
    while ((next_delim = strchr(index, delim)))
    {
	/*
	 * allocate space for the string plus the NULL -
	 * we don't copy the delimiter.
	 */
	length = next_delim - index;
	(*list)[element] = (char*) xcalloc(length + 1, sizeof(char));

	strncpy((*list)[element], index, length);

	index = next_delim + 1;
	element++;
    }
    /* extract last piece of the string */
    (*list)[element] = xstrdup(index);

    return num_elements;
} /* split */ 

/* Replacement prototypes for systems missing strspn/strcspn */
#ifndef HAVE_STRSPN
size_t strspn(const char* s, const char* accept);
size_t strcspn(const char* s, const char* reject);
#endif


/*
 * Return the number of characters that form an unbroken sequence of
 * characters found in accept, staring from the end of the string.
 * See strspn(3).
 */
size_t QSTR_rspn(const char* string, const char* accept)
{
    register size_t i = 0, j = 0;
    size_t length = 0;
    bool match = FALSE;
    int num_chars = 0;

    assert(string != NULL);
    assert(accept != NULL);

    length = strlen(string);
    num_chars = strlen(accept);

    /* Find the first character not in accept */
    for(i = length - 1; i >= 0; i--)
    {
	match = FALSE;
	for(j = 0; j < num_chars; j++)
	{
	    if (accept[j] == string[i])
	    {
		match = TRUE;
		break;
	    }
	}
	if (match == FALSE) break;
    }

    return length - (i + 1);
} /* rspn */

/*
 * Return the number of characters that form an unbroken sequence of
 * characters not found in reject, starting from the end of the string.
 * See strcspn(3).
 */
size_t QSTR_rcspn(const char* string, const char* reject)
{
    register size_t i = 0, j = 0;
    size_t length = 0;
    bool match = FALSE;
    int num_chars = 0;

    assert(string != NULL);
    assert(reject != NULL);

    length = strlen(string);
    num_chars = strlen(reject);

    /* Find the first character in reject */
    for(i = length - 1; i >= 0; i--)
    {
	match = FALSE;
	for(j = 0; j < num_chars; j++)
	{
	    if (reject[j] == string[i])
		goto FOUND;
	}
    }

 FOUND:
    return length - (i + 1);
} /* rcspn */

/*
 * Return a copy of string after removing all characters on the left side
 * that are found in trimchars.
 */
char* QSTR_triml(const char* string, const char* trimchars)
{
    size_t offset = 0;

    assert(string != NULL);
    assert(trimchars != NULL);

    offset = strspn(string, trimchars);

    return xstrdup(&string[offset]);
} /* triml */

/*
 * Return a copy of string after removing all characters on the right side
 * that are found in trimchars.
 */
char* QSTR_trimr(const char* string, const char* trimchars)
{
    char* trimmed = NULL;
    size_t length = 0;
    size_t offset = 0;

    assert(string != NULL);
    assert(trimchars != NULL);

    length = strlen(string);
    offset = QSTR_rspn(string, trimchars);

    trimmed = xcalloc((length - offset) + 1, sizeof(char));
    strncpy(trimmed, string, length - offset);

    return trimmed;
} /* trimr */

/*
 * Return a copy of string after removing all characters on the right side
 * and the left side that are found in trimchars.
 */
char* QSTR_trim(const char* string, const char* trimchars)
{
    char * trimmed = NULL;
    size_t new_len = 0;
    size_t length = 0;
    size_t left = 0, right = 0;

    assert(string != NULL);
    assert(trimchars != NULL);

    length = strlen(string);
    left = strspn(string, trimchars);
    right = QSTR_rspn(string, trimchars);

    new_len = length - left - right;
    trimmed = xcalloc(new_len + 1, sizeof(char));
    strncpy(trimmed, &string[left], new_len);

    return trimmed;
} /* trim */


/*
 * Paste *num* strings together in a newly allocated buffer.
 * Returns a pointer to the newly allocated space.
 */
char* QSTR_glue(int num, ...)
{
    va_list ap;
    char** temp_array = NULL;
    char* ret_val = NULL;
    int i = 0;
    int length = 0;

    assert(num > 1);
    
    temp_array = xcalloc(num, sizeof(char*));

    /* Calculate the size */
    va_start(ap, num);

    for (i = 0; i < num; i++)
    {
	temp_array[i] = va_arg(ap, char *);
	length += strlen(temp_array[i]);
    }

    /* Cat together */
    ret_val = xcalloc(length + 1, sizeof(char));

    for (i = 0; i < num; i++)
    {
	strcat(ret_val, temp_array[i]);
    }

    va_end(ap);

    free(temp_array);
    return ret_val;
} /* QSTR_glue */

/*
 * Returns malloc()ed upper-case version of `string'.
 */
char* QSTR_toupper(const char* string)
{
    char* uc_string;
    register const char* c;
    register char* d;

    assert(string != NULL);

    uc_string = (char*) xmalloc(strlen(string) + 1);
    for (c = string, d = uc_string; *c; c++, d++)
	*d = toupper(*c);
    *(++d) = '\0';

    return uc_string;
}
