/*
 * Utilities (QCI)
 * Author: Alex Plotnick
 *	Copyright (C) 1998 QLUE Consulting, Inc.
 * $Id: qci_util.h,v 3.5 1998/08/22 23:38:41 adfh Exp $
 *
 *  Utilities package - safe versions of some standard C library functions.
 */

/*
 * See the file "LICENSE.txt" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */

#ifndef _UTIL_H
#define _UTIL_H

#include <qci_util_config.h>

#include <stdlib.h>

/* Run of the mill boolean type */
#ifndef __cplusplus

#ifndef bool
#define bool int
#endif

#ifndef FALSE
#define FALSE 0
#endif

#ifndef TRUE
#define TRUE 1
#endif

#endif /* __cplusplus */

/* Constants */
const int QCI_MDA_MAXDIMS;

/* Public Variables */

/*
 * Is this process a daemon process? if so, we need to write errors to syslog
 * instead of stderr. DEFAULT is FALSE.
 */
extern bool QCI_is_daemon_proc;

/*
 * If is_daemon_proc is TRUE, what syslog log level do we use for fatal
 * errors. Default is LOG_ERR.
 */
extern int QCI_fatal_error_log_level;

/* Functions */

/*
 * Fatal error.  Print an error message to stderr if is_daemon_proc is false,
 * syslog otherwise, then exit. If writing to syslog, fatal_error_log_level
 * determines the syslog message priority.
 */
void QCI_fatal_error(const char* msg, ...);

/*
 * Initializes a daemon process: forks, creates a new session, forks again,
 * sets is_daemon_proc appropriately, changes
 * the working directory to "/", clears the umask, closes open
 * file descriptors, and initializes syslog. In short, nearly everything
 * that a daemon has to do at startup is taken care of. Does not open and
 * assign anything to stdout, stderr, stdin (e.g. log file, /dev/null, etc.).
 *
 * Based on the example from Stevens, _UNIX_Network_Programming_,
 * 2nd Ed, Vol. 1, p.335.
 */
void QCI_daemonize( const char* prog_name, int syslog_options,
		    int facility );

/*
 * Wrapper over fork(). If fork() fails due to resource exhaustion,
 * call fatal_error().
 */
pid_t xfork();

/* Safe malloc; exits on failure */
void* xmalloc(size_t size);

/* Safe calloc; exits on failure */
void* xcalloc(size_t nmemb, size_t size);

/* Safe realloc; exits on failure */
void* xrealloc(void* ptr, size_t newsize);

/* Allocates space for and duplicates the contents of a buffer. */
void* xmemdup(const void* src, size_t n);

/*
 * Zeros out an area of memory. Replaces the BSD bzero function if not
 * available.
 * PORTABILITY: Many systems won't provide the prototype for bzero if
 * _POSIX_SOURCE is defined.
 */
#if !defined(HAVE_BZERO) || defined(_POSIX_SOURCE)
void bzero(void* s, size_t n);
#endif

/* Replaces the memcmp function if unavailable or broken
 * (some SunOS versions) */
#ifndef HAVE_MEMCMP
int memcmp(const void *s1, const void *s2, size_t n);
#endif

/* Allocates and zeros memory for a structure. Wraps calloc. */
void* QCI_scalloc(size_t n);

/*
 * Dynamically allocate a multi-dimensional array such that it can be
 * deallocated using free.
 * Based heavily on public domain code from Paul Schlyter.
 *
 * *e_size*	: Size of each element.
 * *dims*	: number of dimensions (1 ... MDA_MAXDIMS)
 * *...*	: number of elements in each dimension (integers)
 *
 * Uses checked allocation (xmalloc, etc).
 * Memory is cleared using calloc.
 */
void* QCI_mda_malloc( int esiz, int dims, ... );

/* Wraps getlogin and checks memory errors. */
char* xgetlogin();

/*
 * Wrapper for usleep where not available.
 * NOTE: On Linux, usleep() returns nothing.  On just about everything else,
 * it seems to return 0 on success, and -1 on failure.
 * ALSO NOTE: On OSF 3.2, the man page for usleep(3) says it returns an
 * `unsigned int'.  IT LIES.  Check line 146 of `/usr/include/unistd.h'.
 * It's `void'.
 */
#if !defined(HAVE_USLEEP) || defined(_POSIX_SOURCE)
#if defined __linux__ || defined _OSF_SOURCE
void usleep(unsigned int usec);
#else
int usleep(unsigned int usec);
#endif
#endif /* HAVE_USLEEP */

#endif /* _UTIL_H */
