/*
 * File: util.c
 * Author: Alex Plotnick
 *	Copyright (C) 1998 QLUE Consulting, Inc.
 * $Id: qci_util.c,v 3.3 1998/08/22 22:14:30 adfh Exp $
 *
 * Utilities package
 */

/*
 * See the file "LICENSE.txt" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */

#define _POSIX_SOURCE 1

#include "qci_util_config.h"

#include <stdio.h>
#include <stdlib.h>
#include <syslog.h>
#include <string.h>
#include <stdarg.h>
#include <assert.h>
#include <sys/time.h>
#include <unistd.h>
#include <signal.h>
#include <sys/stat.h>
#include "qci_util.h"
#include "qstring.h"

/* Constants */
#define QCI_MDA_MAXDIMS 8

/* Public Variable Definitions */

bool QCI_is_daemon_proc = FALSE;

bool QCI_fatal_error_log_level = LOG_ERR;


/* Functions */

void QCI_fatal_error(const char* msg, ...)
{
    char* buf = NULL;
    va_list args;

    va_start(args, msg);

    vasprintf(&buf, msg, args);

    if (QCI_is_daemon_proc)
    {
	syslog(QCI_fatal_error_log_level, buf);
    } else {
	/* WIERDNESS: Flush stdout in case stdout and stderr are the same. */
	fflush(stdout);
	fputs("fatal error: ", stderr);
	fputs(buf, stderr);
	fputs("\n", stderr);
    }

    /* Flush all buffers. */
    fflush(NULL);

    va_end(args);

    exit(EXIT_FAILURE);
} /* QCI_fatal_error */

void QCI_daemonize( const char* prog_name, int syslog_options,
		    int facility )
{
    int i;
    pid_t pid;

    if ((pid = xfork()) != 0)
    {
	/* Parent process exits */
	exit(0);
    }

    /* Child continues in background */
    (void) setsid();
    signal(SIGHUP, SIG_IGN);

    if ((pid = xfork()) != 0)
    {
	/* First child exits. */
	exit(0);
    }

    /* 2nd child continues. */
    QCI_is_daemon_proc = TRUE;
    chdir("/");
    umask(0);

    /*
     * HACK: There is no real way to tell which file descriptors are open to
     * close them. We can't cycle through all valid file descriptors (this may
     * be an exceedingly large number...), so we take Stevens' advice and close
     * the first 64 descriptors, most of which probably are not open.
     *
     * At some point I want to put some tests in autoconf for systems where
     * this can be done more correctly and define a closeall() function to
     * bury the resultant #ifdefs.
     */
    for(i=0; i < 64; i++)
	(void) close(i);

    openlog(prog_name, syslog_options, facility);
} /* QCI_daemonize */

pid_t xfork()
{
    pid_t pid;

    if ((pid = fork()) == -1)
	QCI_fatal_error("virtual memory exhausted in xfork()");

    return pid;
} /* xfork() */


/*
 * Function: xmalloc
 * Description: Safe malloc; exits on failure
 */
void* xmalloc(size_t size)
{
    void* p;

    p = malloc(size);
    if (!p)
	QCI_fatal_error("virtual memory exhausted in xmalloc(%d)", size);

    return p;
} /* xmalloc */

/*
 * Function: xcalloc
 * Description: Safe calloc; exits on failure
 */
void* xcalloc(size_t nmemb, size_t size)
{
    void* p;

    p = calloc(nmemb, size);
    if (!p)
	QCI_fatal_error( "virtual memory exhausted in xcalloc(%d, %d)",
			 nmemb, size);

    return p;
} /* xcalloc */

/*
 * Function: xrealloc
 * Description: Safe realloc; exits on failure
 */
void* xrealloc(void* ptr, size_t newsize)
{
    void* nptr;

    nptr = realloc(ptr, newsize);
    if (!nptr)
	QCI_fatal_error("could not realloc %d bytes", newsize);

    return nptr;
} /* xrealloc */

/*
 * Allocates space for and duplicates the contents of a buffer.
 */
void* xmemdup(const void* src, size_t n)
{
    void* dest = xmalloc(n);

    return memcpy(dest, src, n);
}

/* Allocates and zeros memory for a structure. Wraps calloc. */
void* QCI_scalloc(size_t n)
{
    void* p;

    p = calloc(1, n);
    if (!p)
	QCI_fatal_error("virtual memory exhausted in scalloc(%d)", n);

    return p;    
} /* QCI_scalloc */

/*
 * Dynamically allocate a multi-dimensional array such that it can be
 * deallocated using free.
 * Based heavily on public domain code from Paul Schlyter.
 *
 * *e_size*	: Size of each element.
 * *dims*	: number of dimensions (1 ... MDA_MAXDIMS)
 * *...*	: number of elements in each dimension (integers)
 *
 * Uses checked allocation (xmalloc, etc).
 * Memory is cleared using calloc.
 */
void* QCI_mda_malloc( int e_size, int dims, ... )
{
    unsigned int dim[QCI_MDA_MAXDIMS], dim_accum[QCI_MDA_MAXDIMS];
    va_list ap;
    int i, j;
    long int total_size;
    void** buf;
    void* tmp1;
    char* tmp2;
    
    assert ((dims > 0)  &&  (dims > QCI_MDA_MAXDIMS));

    memset(dim, 0, sizeof(dim));
    memset(dim_accum, 0, sizeof(dim_accum));

    /* Read dimension numbers */
    va_start(ap, dims);
    dim[0] = (dim_accum[0] = va_arg(ap,int)); 
    for (i = 1; i < dims; i++)
    {
	dim[i] = va_arg(ap,int);
	dim_accum[i] = dim_accum[i-1] * dim[i];
    }
    va_end(ap);

    /* Compute total array size */
    total_size = e_size * dim_accum[dims - 1];

    /* Add space for pointers */
    for (i = 0; i < dims - 1; i++ )
	total_size += (sizeof(void *) * dim_accum[i]);

    /* allocate the memory */
    tmp1 = xcalloc((size_t) total_size, 1);
    buf = (void **) tmp1;
    
    for (i = 1; i < dims; i++)            /* Fill in pointers */
    {
            int size;
            int accd = dim_accum[i-1], d = dim[i];

            size =  i == (dims - 1) ? e_size : sizeof(void *);

            tmp2 = (char *) buf + sizeof(void *) * accd;
            for (j = 0; j < accd; j++)
            {
                  *buf++ = tmp2;
                  tmp2 += size * d;
            }
      }

      return tmp1;

}  /* QCI_mda_malloc */

/* Wraps getlogin and checks memory errors. */
char* xgetlogin()
{
    return xstrdup(getlogin());
} /* xgetlogin */
