/*
 * Hashtable (HASH)
 * Author: Alex Plotnick - QLUE Consulting, Inc.
 *	Copyright (C) 1998 QLUE Consulting, Inc.
 * $Id: hashtable.c,v 3.0 1998/05/21 17:13:47 adfh Exp $
 *
 * Hashtable routines.  We use a seperate chaining method for collision,
 * detection, plus a single node cache.
 */

/*
 * See the file "LICENSE.txt" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */

#define _POSIX_SOURCE 1

#include "qci_util_config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include "qci_util.h"
#include "qstring.h"
#include "hashtable.h"

/* prime number is better for hashing algorithm */
const int HASH_DEFAULT_INITIAL_SIZE = 31;

static int hash(const char* x, int buckets);

static Hash_Node lookup(Hash_Table ht, const char* key, int check_size);

static void insert(Hash_Table ht,
		   const char* key, const char* value,
		   int hashval);

/*
 * Function: HASH_new
 * Description: Return a new hash table of size `size'.
 */
Hash_Table HASH_new(int size)
{
    int i;
    Hash_Table ht;

    if (size < 1) size = HASH_DEFAULT_INITIAL_SIZE;
 
    ht = (Hash_Table) xmalloc(sizeof(*ht));

    /* Initialize the new hashtable */
    ht->size = size;
    ht->count = 0;
    ht->resize_threshold = size << 1;
    ht->nodes = (Hash_Node*) xcalloc(size, sizeof(Hash_Node));
    for (i = 0; i < ht->size; i++)
	ht->nodes[i] = NULL;
    ht->node_cache = NULL;
    ht->hashval_cache = 0;

    return ht;
} /* HASH_new */

/*
 * Function: HASH_clear
 * Description: Empty the hash table
 */
void HASH_clear(Hash_Table ht)
{
    Hash_Node node, next;
    int i;

    assert(ht != NULL);

    for (i = 0; i < ht->size; i++)
    {
	for (node = ht->nodes[i]; node; )
	{
	    next = node->next;

	    free(node->key);
	    free(node->value);
	    free(node);

	    node = next;
	}
	ht->nodes[i] = NULL;
    }

    ht->count = 0;
    ht->node_cache = 0;
    ht->hashval_cache = 0;
}

/*
 * Function: HASH_delete
 * Description: Free all storage associated with hashtable `ht'.
 */
void HASH_delete(Hash_Table ht)
{
    Hash_Node node, next;
    int i;

    assert (ht != NULL);

    for (i = 0; i < ht->size; i++)
    {
	for (node = ht->nodes[i]; node; )
	{
	    next = node->next;

	    free(node->key);
	    free(node->value);
	    free(node);

	    node = next;
	}
    }
    free(ht->nodes);
    free(ht);
} /* HASH_delete */

/*
 * Function: HASH_add
 * Description: Add a new node to the hashtable `ht'.  This function should
 * be used for initializing a large table.  No resize checks are done here
 * at all. NOTE: it is possible to insert duplicate keys by using this
 * function.  They will be removed by the next resize.  In the meantime, the
 * last value inserted should always be returned by a fetch.
 */
void HASH_add(Hash_Table ht, const char* key, const char* value)
{
    int hashval;

    assert(key != NULL);

    hashval = hash(key, ht->size);
    insert(ht, key, value, hashval);

} /* HASH_add */


/*
 * Function: HASH_set
 * Description: Set the value of node `key' to `value'.  If no such
 * node exists, insert a new one.  If the table is too crowded, it
 * is resized.
 */
void HASH_set(Hash_Table ht, const char* key, const char* value)
{
    Hash_Node node;

    if (!(node = lookup(ht, key, 1)))
    {
	/* Key not found - insert a new node */
	insert(ht, key, value, ht->hashval_cache);
    } else {
	/* Key already used - replace the old value with the new */
	free(node->value);
	node->value = value ? xstrdup(value) : NULL;
    }

} /* HASH_set */ 

/*
 * Function: HASH_remove
 * Description: Remove the node associated with `key'.
 */
void HASH_remove(Hash_Table ht, const char* key)
{
    Hash_Node node, foo;

    if ((node = lookup(ht, key, 0)) == NULL)
	return;

    /* The following code depends rather heavily on the fact that lookup
     * caches its results
     */
    foo = ht->nodes[ht->hashval_cache];
    if (foo == node)
    {
	ht->nodes[ht->hashval_cache] = node->next;
    } else {
	while (foo->next != node)
	    foo = foo->next;
	foo->next = node->next;
    }
    free(node->key);
    free(node->value);
    free(node);
    ht->node_cache = NULL;

    ht->count--;
} /* HASH_remove */

/*
 * Function: HASH_exists
 * Description: Return 1 if there is a node with key `key', and 0 otherwise.
 */
int HASH_exists(Hash_Table ht, const char* key)
{
    return lookup(ht, key, 1) != NULL;
}

/*
 * Function: HASH_fetch
 * Description: Return the value associated with `key'.  Returns NULL
 * if there is no such node.
 */
const char* HASH_fetch(Hash_Table ht, const char* key)
{
    Hash_Node node;

    node = lookup(ht, key, 1);
    return node ? node->value : NULL;
}

/*
 * Function: HASH_resize
 * Description: Force the hashtable to resize itself to an optimal number
 * of buckets.
 */
void HASH_resize(Hash_Table ht)
{
    Hash_Node *old_nodes, node, next, foo;
    int i, old_size;
    int hashval;

    assert(ht != NULL);

    old_size = ht->size;
    old_nodes = ht->nodes;

    ht->size = ht->count;
    if (ht->size < 1) ht->size = HASH_DEFAULT_INITIAL_SIZE;
    ht->resize_threshold = ht->size << 1;
    ht->nodes = (Hash_Node*) xcalloc(ht->size, sizeof(Hash_Node));
    for (i = 0; i < ht->size; i++)
	ht->nodes[i] = NULL;
    ht->count = 0;

    /* Scan through the array of old nodes, and rehash them as we go
     * into the new table */
    for (i = 0; i < old_size; i++)
    {
	/* Walk the chain */
	for (node = old_nodes[i]; node; )
	{
	    next = node->next;

	    /* Add the node to the new table. */
	    hashval = hash(node->key, ht->size);

	    /* Walk the new chain */
	    for (foo = ht->nodes[hashval]; foo; foo = foo->next)
	    {
		if (!strcmp(foo->key, node->key))
		{
		    /* Duplicate key.  Toss the current one. */
		    free(node->key);
		    free(node->value);
		    free(node);

		    goto END;
		}
	    }

	    /* This node is still around.  Stick it at the head of the
	     * new chain for this hashval */
	    node->next = ht->nodes[hashval];
	    ht->nodes[hashval] = node;
	    ht->count++;

	END:
	    node = next;
	}
    }

    /* Invalidate the cache */
    ht->node_cache = NULL;

    free(old_nodes);
} /* HASH_resize */

#define MAXDEPTH 7
/*
 * Function: HASH_info
 * Description: Strictly a debugging tool.  Prints some basic info about
 * a hashtable.
 */
void HASH_info(Hash_Table ht)
{
    Hash_Node node;
    int chainlen[MAXDEPTH+1];
    int i, tmpcount, total;

    assert(ht != NULL);

    printf("\nHashtable Info:\n");
    printf("\tsize of nodes[]: %4d\n", ht->size);
    printf("\tnumber of nodes: %4d\n", ht->count);
    printf("\tresize threshold: %4d\n", ht->resize_threshold);

    memset(chainlen, 0, sizeof(chainlen));

    total = 0;
    for (i = 0; i < ht->size; i++)
    {
	tmpcount = 0;
	for (node = ht->nodes[i]; node; node = node->next)
	    tmpcount++;
	if (tmpcount >= MAXDEPTH)
	    chainlen[MAXDEPTH]++;
	else
	    chainlen[tmpcount]++;
	total += tmpcount;
    }

    for (i = 0; i < MAXDEPTH; i++)
	printf("\tnumber of hash values with chain of length %2d: %4d\n",
	       i, chainlen[i] );
    printf("\tnumber of hash values with chain of length %d+: %4d\n",
	   MAXDEPTH, chainlen[MAXDEPTH] );
    printf("\t\t\t\ttotal number of nodes: %4d\n\n", total);
} /* HASH_info */


/*
 * Function: HASH_check_invariants 
 * Verify invariant conditions and abort if anything is wrong
 */
void HASH_check_invariants(Hash_Table ht)
{
    int count = 0;
    Hash_Node node;
    int found_cache = 0;
    int i;

    assert(ht != NULL);
    assert(ht->nodes != NULL);
    assert((ht->node_cache == NULL)|| (ht->hashval_cache < ht->size));
    assert(ht->resize_threshold > ht->size);

    /* Make sure that count actually matches the number of elements.
     * At the same time, look for the cached node to make sure it is
     * valid.
     */
    for (i = 0; i < ht->size; i++)
    {
	for (node = ht->nodes[i]; node; node = node->next)
	{
	    if (ht->node_cache == node) found_cache = 1;
	    count++;
	}
    }

    assert(count == ht->count);
    assert(found_cache || (ht->node_cache == NULL));
} /* HASH_check_invariants */


/*
 * Function: hash
 * Description: Hashing function.  From `hashpjw', the Dragon book, p436.
 */
static int hash(const char* x, int buckets)
{
    unsigned int h = 0;
    unsigned int g;

    while (*x != 0)
    {
	h = (h << 4) + *x++;
	if ((g = h & 0xf0000000) != 0)
	    h = (h ^ (g >> 24)) ^ g;
    }

    return (int) (h % buckets);
} /* hash */

/*
 * Function: lookup
 * Description: Return a pointer to the node given by `key', or NULL
 * if there is no such node.  The hashval computed is always cached,
 * and if we are successful, the node is also cached.
 */
static Hash_Node lookup(Hash_Table ht, const char* key, int check_size)
{
    Hash_Node node;
    int hashval;

    /* Sanity check */
    assert(ht != NULL);
    assert(key != NULL);

    /* Check to see if we need to resize */
    if (check_size && ht->count >= ht->resize_threshold)
	HASH_resize(ht);

    /* Cache check */
    if ((node = ht->node_cache) != NULL)
	if (!strcmp(key, node->key))
	{
	    return node;
	}

    hashval = hash(key, ht->size);
    ht->hashval_cache = hashval;
    for (node = ht->nodes[hashval]; node; node = node->next)
	if (!strcmp(key, node->key))
	{
	    /* Got it... cache the node */
	    ht->node_cache = node;
	    return node;
	}

    return NULL;
} /* lookup */

/*
 * Function: insert
 * Description: Insert a new node into hashtable `ht' at position `hashval'.
 */
static void insert(Hash_Table ht,
		   const char* key, const char* value,
		   int hashval)
{
    Hash_Node node;

    /* Sanity check */
    assert(ht != NULL);
    assert(key != NULL);

    node = (Hash_Node) xmalloc(sizeof(*node));
    node->key = xstrdup(key);
    node->value = value ? xstrdup(value) : NULL;
    node->next = ht->nodes[hashval];
    ht->nodes[hashval] = node;
    ht->count++;

    ht->node_cache = node;
    ht->hashval_cache = hashval;
} /* insert */
