/* Status class implementation: inline functions.
   Copyright (C) 2001 Roberto Bagnara <bagnara@cs.unipr.it>

This file is part of the Parma Polyhedra Library (PPL).

The PPL is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2 of the License, or (at your
option) any later version.

The PPL is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
USA.

For the most up-to-date information see the Parma Polyhedra Library
site: http://www.cs.unipr.it/ppl/ . */

namespace Parma_Polyhedra_Library {

inline
Status::Status(status_t mask)
  : flags(mask) {
}

inline
Status::Status()
  : flags(ZERO_DIM_UNIV) {
}

inline Status
operator &(const Status& x, const Status& y) {
  return Status(x.flags & y.flags);
}

inline Status
operator |(const Status& x, const Status& y) {
  return Status(x.flags | y.flags);
}

inline bool
Status::test_all(status_t mask) const {
  return (flags & mask) == mask;
}

inline bool
Status::test_any(status_t mask) const {
  return flags & mask;
}

inline void
Status::set(status_t mask) {
  flags |= mask;
}

inline void
Status::reset(status_t mask) {
  flags &= ~mask;
}

inline bool
Status::test_zero_dim_univ() const {
  return flags == ZERO_DIM_UNIV;
}

inline void
Status::reset_zero_dim_univ() {
  // This is a no-op if the current status is not zero-dim.
  if (flags == ZERO_DIM_UNIV)
    // In the zero-dim space, if it is not the universe it is empty.
    flags = EMPTY;
}

inline void
Status::set_zero_dim_univ() {
  // Zero-dim universe is incompatible with anything else.
  flags = ZERO_DIM_UNIV;
}

inline bool
Status::test_empty() const {
  return test_any(EMPTY);
}

inline void
Status::reset_empty() {
  reset(EMPTY);
}

inline void
Status::set_empty() {
  flags = EMPTY;
}

inline bool
Status::test_c_up_to_date() const {
  return test_any(C_UP_TO_DATE);
}

inline void
Status::reset_c_up_to_date() {
  reset(C_UP_TO_DATE);
}

inline void
Status::set_c_up_to_date() {
  set(C_UP_TO_DATE);
}

inline bool
Status::test_g_up_to_date() const {
  return test_any(G_UP_TO_DATE);
}

inline void
Status::reset_g_up_to_date() {
  reset(G_UP_TO_DATE);
}

inline void
Status::set_g_up_to_date() {
  set(G_UP_TO_DATE);
}

inline bool
Status::test_c_minimized() const {
  return test_any(C_MINIMIZED);
}

inline void
Status::reset_c_minimized() {
  reset(C_MINIMIZED);
}

inline void
Status::set_c_minimized() {
  set(C_MINIMIZED);
}

inline bool
Status::test_g_minimized() const {
  return test_any(G_MINIMIZED);
}

inline void
Status::reset_g_minimized() {
  reset(G_MINIMIZED);
}

inline void
Status::set_g_minimized() {
  set(G_MINIMIZED);
}

inline bool
Status::test_sat_c_up_to_date() const {
  return test_any(SAT_C_UP_TO_DATE);
}

inline void
Status::reset_sat_c_up_to_date() {
  reset(SAT_C_UP_TO_DATE);
}

inline void
Status::set_sat_c_up_to_date() {
  set(SAT_C_UP_TO_DATE);
}

inline bool
Status::test_sat_g_up_to_date() const {
  return test_any(SAT_G_UP_TO_DATE);
}

inline void
Status::reset_sat_g_up_to_date() {
  reset(SAT_G_UP_TO_DATE);
}

inline void
Status::set_sat_g_up_to_date() {
  set(SAT_G_UP_TO_DATE);
}

} // namespace Parma_Polyhedra_Library
