/* Integer class implementation: inline functions.
   Copyright (C) 2001 Roberto Bagnara <bagnara@cs.unipr.it>

This file is part of the Parma Polyhedra Library (PPL).

The PPL is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2 of the License, or (at your
option) any later version.

The PPL is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
USA.

For the most up-to-date information see the Parma Polyhedra Library
site: http://www.cs.unipr.it/ppl/ . */

#ifndef _Integer_inlines_hh
#define _Integer_inlines_hh 1

#include <cassert>

namespace Parma_Polyhedra_Library {

inline
Integer::Integer() {
  mpz_init(v);
}

inline
Integer::Integer(char x) {
  mpz_init_set_si(v, (long int) x);
}

inline
Integer::Integer(int x) {
  mpz_init_set_si(v, (long int) x);
}

inline
Integer::Integer(unsigned int x) {
  mpz_init_set_ui(v, (unsigned long int) x);
}

inline
Integer::Integer(long int x) {
  mpz_init_set_si(v, x);
}

inline
Integer::Integer(unsigned long int x) {
  mpz_init_set_ui(v, x);
}

inline
Integer::Integer(const char* x, int base) {
#ifndef NDEBUG
  int ret =
#endif
    mpz_init_set_str (v, x, base);
  assert(ret == 0);
}

inline
Integer::Integer(const Integer& y) {
  mpz_init_set(v, y.v);
}

inline Integer&
Integer::operator =(char y) {
  mpz_set_si(v, (long int) y);
  return *this;
}

inline Integer&
Integer::operator =(int y) {
  mpz_set_si(v, (long int) y);
  return *this;
}

inline Integer&
Integer::operator =(unsigned int y) {
  mpz_set_ui(v, (unsigned long int) y);
  return *this;
}

inline Integer&
Integer::operator =(long int y) {
  mpz_set_si(v, y);
  return *this;
}

inline Integer&
Integer::operator =(unsigned long int y) {
  mpz_set_ui(v, y);
  return *this;
}

inline Integer&
Integer::operator =(double y) {
  mpz_set_d(v, y);
  return *this;
}

inline Integer&
Integer::operator =(const Integer& y) {
  mpz_set(v, y.v);
  return *this;
}

inline
Integer::~Integer() {
  mpz_clear(v);
}

inline size_t
Integer::num_digits(int base) const {
  return mpz_sizeinbase(v, base);
}

inline void
Integer::print(char* s, int base) const {
  mpz_get_str(s, base, v);
}

// Comparisons.

inline bool
operator ==(const Integer& x, const Integer& y) {
  return mpz_cmp(x.v, y.v) == 0;
}

inline bool
operator !=(const Integer& x, const Integer& y) {
  return mpz_cmp(x.v, y.v) != 0;
}

inline bool
operator <(const Integer& x, const Integer& y) {
  return mpz_cmp(x.v, y.v) < 0;
}

inline bool
operator >(const Integer& x, const Integer& y) {
  return mpz_cmp(x.v, y.v) > 0;
}

inline bool
operator <=(const Integer& x, const Integer& y) {
  return mpz_cmp(x.v, y.v) <= 0;
}

inline bool
operator >=(const Integer& x, const Integer& y) {
  return mpz_cmp(x.v, y.v) >= 0;
}

inline bool
operator ==(const Integer& x, signed int y) {
  if (y == 0)
    return mpz_sgn(x.v) == 0;
  else
    return mpz_cmp_si(x.v, y) == 0;
}

inline bool
operator !=(const Integer& x, signed int y) {
  if (y == 0)
    return mpz_sgn(x.v) != 0;
  else
    return mpz_cmp_si(x.v, y) != 0;
}

inline bool
operator <(const Integer& x, signed int y) {
  if (y == 0)
    return mpz_sgn(x.v) < 0;
  else
    return mpz_cmp_si(x.v, y) < 0;
}

inline bool
operator >(const Integer& x, signed int y) {
  if (y == 0)
    return mpz_sgn(x.v) > 0;
  else
    return mpz_cmp_si(x.v, y) > 0;
}

inline bool
operator <=(const Integer& x, signed int y) {
  if (y == 0)
    return mpz_sgn(x.v) <= 0;
  else
    return mpz_cmp_si(x.v, y) <= 0;
}

inline bool
operator >=(const Integer& x, signed int y) {
  if (y == 0)
    return mpz_sgn(x.v) >= 0;
  else
    return mpz_cmp_si(x.v, y) >= 0;
}

// Conversions.

inline void
conv(signed int& x, const Integer& y) {
  x = mpz_get_si(y.v);
}

inline signed int
to_slong(const Integer& y) {
  return mpz_get_si(y.v);
}

inline void
conv(unsigned int& x, const Integer& y) {
  x = mpz_get_ui(y.v);
}

inline unsigned int
to_ulong(const Integer& y) {
  return mpz_get_ui(y.v);
}

inline void
conv(double& x, const Integer& y) {
  x = mpz_get_d(y.v);
}

inline double
to_double(const Integer& y) {
  return mpz_get_d(y.v);
}

inline void
conv(float& x, const Integer& y) {
  x = float(mpz_get_d(y.v));
}

inline float
to_float(const Integer& y) {
  return float(mpz_get_d(y.v));
}

// Arithmetic operators.

inline Integer
operator +(const Integer& x, const Integer& y) {
  Integer z;
  mpz_add(z.v, x.v, y.v);
  return z;
}

inline Integer
operator +(const Integer& x, long y) {
  Integer z;
  if (y < 0)
    mpz_sub_ui(z.v, x.v, -y);
  else
    mpz_add_ui(z.v, x.v, y);
  return z;
}

inline Integer
operator +(long x, const Integer& y) {
  Integer z;
  if (x < 0)
    mpz_sub_ui(z.v, y.v, -x);
  else
    mpz_add_ui(z.v, y.v, x);
  return z;
}

inline Integer
operator -(const Integer& x, const Integer& y) {
  Integer z;
  mpz_sub(z.v, x.v, y.v);
  return z;
}

inline Integer
operator -(const Integer& x, long y) {
  Integer z;
  if (y < 0)
    mpz_add_ui(z.v, x.v, -y);
  else
    mpz_sub_ui(z.v, x.v, y);
  return z;
}

inline Integer
operator -(long x, const Integer& y) {
  Integer z;
  if (x < 0) {
    mpz_add_ui(z.v, y.v, -x);
    mpz_neg(z.v, z.v);
  }
  else {
    mpz_neg(z.v, y.v);
    mpz_sub_ui(z.v, y.v, x);
  }
  return z;
}

inline Integer
operator -(const Integer& x) {
  Integer z;
  mpz_neg(z.v, x.v);
  return z;
}

inline Integer
operator *(const Integer& x, const Integer& y) {
  Integer z;
  mpz_mul(z.v, x.v, y.v);
  return z;
}

inline Integer
operator *(const Integer& x, long y) {
  Integer z;
  if (y < 0) {
    mpz_mul_ui(z.v, x.v, -y);
    mpz_neg(z.v, z.v);
  }
  else
    mpz_mul_ui(z.v, x.v, y);
  return z;
}

inline Integer
operator *(long x, const Integer& y) {
  Integer z;
  if (x < 0) {
    mpz_mul_ui(z.v, y.v, -x);
    mpz_neg(z.v, z.v);
  }
  else {
    mpz_mul_ui(z.v, y.v, x);
  }
  return z;
}

inline Integer
operator /(const Integer& x, const Integer& y) {
  Integer z;
  mpz_tdiv_q(z.v, x.v, y.v);
  return z;
}

inline Integer
operator /(const Integer& x, long y) {
  Integer z;
  if (y < 0) {
    mpz_tdiv_q_ui(z.v, x.v, -y);
    mpz_neg(z.v, z.v);
  }
  else
    mpz_tdiv_q_ui(z.v, x.v, y);
  return z;
}

inline Integer
operator /(long x, const Integer& y) {
  Integer z;
  if (x < 0) {
    mpz_tdiv_q_ui(z.v, y.v, -x);
    mpz_neg(z.v, z.v);
  }
  else {
    mpz_tdiv_q_ui(z.v, y.v, x);
  }
  return z;
}

inline Integer
abs(const Integer& x) {
  Integer z;
  mpz_abs(z.v, x.v);
  return z;
}

inline void
Integer::negate() {
  mpz_neg(v, v);
}

inline Integer
operator <<(const Integer& x, long y) {
  Integer z;
  if (y < 0)
    mpz_fdiv_q_2exp(z.v, x.v, -y);
  else
    mpz_mul_2exp(z.v, x.v, (unsigned long int) y);
  return z;
}

inline Integer
operator >>(const Integer& x, long y) {
  Integer z;
  if (y < 0)
    mpz_mul_2exp(z.v, x.v, (unsigned long int) y);
  else
    mpz_fdiv_q_2exp(z.v, x.v, -y);
  return z;
}

// Op= notation.

inline Integer&
operator +=(Integer& x, const Integer& y) {
  mpz_add(x.v, x.v, y.v);
  return x;
}

inline Integer&
operator +=(Integer& x, long y) {
  if (y < 0)
    mpz_sub_ui(x.v, x.v, -y);
  else
    mpz_add_ui(x.v, x.v, y);
  return x;
}

inline Integer&
operator -=(Integer& x, const Integer& y) {
  mpz_sub(x.v, x.v, y.v);
  return x;
}

inline Integer&
operator -=(Integer& x, long y) {
  if (y < 0)
    mpz_add_ui(x.v, x.v, -y);
  else
    mpz_sub_ui(x.v, x.v, y);
  return x;
}

inline Integer&
operator *=(Integer& x, const Integer& y) {
  mpz_mul(x.v, x.v, y.v);
  return x;
}

inline Integer&
operator *=(Integer& x, long y) {
  if (y < 0) {
    mpz_mul_ui(x.v, x.v, -y);
    mpz_neg(x.v, x.v);
  }
  else
    mpz_mul_ui(x.v, x.v, y);
  return x;
}

inline Integer&
operator /=(Integer& x, const Integer& y) {
  mpz_tdiv_q(x.v, x.v, y.v);
  return x;
}

inline Integer&
operator /=(Integer& x, long y) {
  if (y < 0) {
    mpz_tdiv_q_ui(x.v, x.v, -y);
    mpz_neg(x.v, x.v);
  }
  else
    mpz_tdiv_q_ui(x.v, x.v, y);
  return x;
}

inline Integer&
operator <<=(Integer& x, long y) {
  if (y < 0)
    mpz_fdiv_q_2exp(x.v, x.v, (unsigned long int) -y);
  else
    mpz_mul_2exp(x.v, x.v, (unsigned long int) y);
  return x;
}

inline Integer&
operator >>=(Integer& x, long y) {
  if (y < 0)
    mpz_mul_2exp(x.v, x.v, (unsigned long int) y);
  else
    mpz_fdiv_q_2exp(x.v, x.v, (unsigned long int) -y);
  return x;
}

// Increment/decrement.

inline Integer&
Integer::operator ++() {
  mpz_add_ui(v, v, 1);
  return *this;
}

inline Integer
Integer::operator ++(int) {
  Integer tmp = *this;
  mpz_add_ui(v, v, 1);
  return tmp;
}

inline Integer&
Integer::operator --() {
  mpz_sub_ui(v, v, 1);
  return *this;
}

inline Integer
Integer::operator --(int) {
  Integer tmp = *this;
  mpz_sub_ui(v, v, 1);
  return tmp;
}

// Greatest Common Divisor.

inline void
Integer::gcd_assign(const Integer& y) {
  assert(*this != 0 || y != 0);
  mpz_gcd(v, v, y.v);
}

inline void
Integer::gcd_assign(long y) {
  assert(*this != 0 || y != 0);
  if (y < 0)
    mpz_gcd_ui(v, v, -y);
  else
    mpz_gcd_ui(v, v, y);
}

inline void
Integer::gcd_assign(const Integer& y, const Integer& z) {
  assert(y != 0 || z != 0);
  mpz_gcd(v, y.v, z.v);
}

inline Integer
gcd(const Integer& x, const Integer& y) {
  assert(x != 0 || y != 0);
  Integer z;
  mpz_gcd(z.v, x.v, y.v);
  return z;
}

inline Integer
gcd(const Integer& x, long y) {
  assert(x != 0 || y != 0);
  Integer z;
  if (y < 0)
    mpz_gcd_ui(z.v, x.v, -y);
  else
    mpz_gcd_ui(z.v, x.v, y);
  return z;
}

// Exact division.

inline void
Integer::exact_div_assign(const Integer& y) {
  mpz_divexact(v, v, y.v);
}

inline void
Integer::exact_div_assign(const Integer& y, const Integer& z) {
  mpz_divexact(v, y.v, z.v);
}

inline Integer
exact_div(const Integer& y, const Integer& z) {
  Integer x;
  mpz_divexact(x.v, y.v, z.v);
  return x;
}

inline void
Integer::mul_assign(const Integer& y, const Integer& z) {
  mpz_mul(v, y.v, z.v);
}

inline void
Integer::sub_assign(const Integer& y, const Integer& z) {
  mpz_sub(v, y.v, z.v);
}

// Sign extraction.

inline int
sgn(const Integer& x) {
  return mpz_sgn(x.v);
}

inline int
cmp(const Integer& x, const Integer& y) {
  return mpz_cmp(x.v, y.v);
}

inline
Integer::RandomState::RandomState() {
  gmp_randinit(state, GMP_RAND_ALG_DEFAULT, 128);
}

inline
Integer::RandomState::~RandomState() {
  gmp_randclear(state);
}

inline gmp_randstate_t&
Integer::RandomState::operator ()() {
  return state;
}

// Return a pseudo-random number in the range [0, n-1].
inline Integer
random(const Integer& x) {
  Integer z;
  // FIXME: this cast is due to a bug in GMP.
  mpz_urandomm(z.v, Integer::random_state(), const_cast<Integer&>(x).v);
  return z;
}

inline void
Integer::factorial() {
  assert(mpz_fits_ulong_p(v));
  mpz_fac_ui(v, mpz_get_ui(v));
}

inline Integer
factorial(unsigned long n) {
  Integer z;
  mpz_fac_ui(z.v, n);
  return z;
}

inline Integer
factorial(long n) {
  unsigned long un = (n < 0 ? -n : n);
  return factorial(un);
}

inline Integer
factorial(const Integer& x) {
  assert(mpz_fits_ulong_p(x.v));
  return factorial(mpz_get_ui(x.v));
}

inline void
Integer::swap(Integer& y) {
  mpz_swap(v, y.v);
}

inline const Integer&
Integer::zero() {
  static Integer z = 0;
  return z;
}

inline const Integer&
Integer::one() {
  static Integer o = 1;
  return o;
}

} // namespace Parma_Polyhedra_Library

inline void
std::swap(Parma_Polyhedra_Library::Integer& x,
	  Parma_Polyhedra_Library::Integer& y) {
  x.swap(y);
}

#endif
