/* Integer class declaration.
   Copyright (C) 2001 Roberto Bagnara <bagnara@cs.unipr.it>

This file is part of the Parma Polyhedra Library (PPL).

The PPL is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2 of the License, or (at your
option) any later version.

The PPL is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
USA.

For the most up-to-date information see the Parma Polyhedra Library
site: http://www.cs.unipr.it/ppl/ . */

#ifndef _Integer_defs_hh
#define _Integer_defs_hh 1

extern "C" {
#include <gmp.h>
}

#include <iosfwd>

namespace Parma_Polyhedra_Library {

class Integer {
public:
  Integer();

  Integer(char x);
  Integer(int x);
  Integer(unsigned int x);
  Integer(long int x);
  Integer(unsigned long int x);
  Integer(const char* x, int base = 10);

  Integer(const Integer& y);

  Integer& operator =(char y);
  Integer& operator =(int y);
  Integer& operator =(unsigned int y);
  Integer& operator =(long int y);
  Integer& operator =(unsigned long int y);
  Integer& operator =(double y);

  Integer& operator =(const Integer& y);

  ~Integer();

  // Swaps *this with y.
  void swap(Integer& y);

  size_t num_digits(int base = 10) const;
  void print(char* s, int base = 10) const;

  // Comparisons.
  friend bool operator ==(const Integer& x, const Integer& y);
  friend bool operator !=(const Integer& x, const Integer& y);
  friend bool operator <(const Integer& x, const Integer& y);
  friend bool operator >(const Integer& x, const Integer& y);
  friend bool operator <=(const Integer& x, const Integer& y);
  friend bool operator >=(const Integer& x, const Integer& y);

  friend bool operator ==(const Integer& x, signed int y);
  friend bool operator !=(const Integer& x, signed int y);
  friend bool operator <(const Integer& x, signed int y);
  friend bool operator >(const Integer& x, signed int y);
  friend bool operator <=(const Integer& x, signed int y);
  friend bool operator >=(const Integer& x, signed int y);

  // Conversions.
  friend void conv(signed int& x, const Integer& y);
  friend signed int to_slong(const Integer& y);
  friend void conv(unsigned int& x, const Integer& y);
  friend unsigned int to_ulong(const Integer& y);
  friend void conv(double& x, const Integer& y);
  friend double to_double(const Integer& y);
  friend void conv(float& x, const Integer& y);
  friend float to_float(const Integer& y);

  // Arithmetic operators.
  friend Integer operator +(const Integer& x, const Integer& y);
  friend Integer operator +(const Integer& x, long y);
  friend Integer operator +(long x, const Integer& y);
  friend Integer operator -(const Integer& x, const Integer& y);
  friend Integer operator -(const Integer& x, long y);
  friend Integer operator -(long x, const Integer& y);
  friend Integer operator -(const Integer& x);
  friend Integer operator *(const Integer& x, const Integer& y);
  friend Integer operator *(const Integer& x, long y);
  friend Integer operator *(long x, const Integer& y);
  friend Integer operator /(const Integer& x, const Integer& y);
  friend Integer operator /(const Integer& x, long y);
  friend Integer operator /(long x, const Integer& y);

  friend Integer abs(const Integer& x);

  friend Integer operator <<(const Integer& x, long y);
  friend Integer operator >>(const Integer& x, long y);

  // Op= notation.
  friend Integer& operator +=(Integer& x, const Integer& y);
  friend Integer& operator +=(Integer& x, long y);
  friend Integer& operator -=(Integer& x, const Integer& y);
  friend Integer& operator -=(Integer& x, long y);
  friend Integer& operator *=(Integer& x, const Integer& y);
  friend Integer& operator *=(Integer& x, long y);
  friend Integer& operator /=(Integer& x, const Integer& y);
  friend Integer& operator /=(Integer& x, long y);

  friend Integer& operator <<=(Integer& x, long y);
  friend Integer& operator >>=(Integer& x, long y);

  // Increment/decrement.
  Integer& operator ++();
  Integer operator ++(int);
  Integer& operator --();
  Integer operator --(int);

  // Negation, change of sign.
  void negate();

  // Input/output.
  friend std::istream& operator >>(std::istream& s, Integer& x);
  friend std::ostream& operator <<(std::ostream& s, const Integer& x);

  // Greatest Common Divisor.
  void gcd_assign(const Integer& y);
  void gcd_assign(long y);
  void gcd_assign(const Integer& y, const Integer& z);

  friend Integer gcd(const Integer& x, const Integer& y);
  friend Integer gcd(const Integer& x, long y);

  // Exact division.
  void exact_div_assign(const Integer& y);
  void exact_div_assign(const Integer& y, const Integer& z);
  friend Integer exact_div(const Integer& x, const Integer& y);

  // Multiplication with assignment.
  void mul_assign(const Integer& y, const Integer& z);

  // Subtraction with assignment.
  void sub_assign(const Integer& y, const Integer& z);

  // Sign: returns +1, 0, or -1 if x > 0, x = 0, or x < 0, respectively.
  friend int sgn(const Integer& x);

  // Comparison: returns a positive value, 0, or a negative value
  // if x > y, x = y, or x < y, respectively.
  friend int cmp(const Integer& x, const Integer& y);

  // Return a pseudo-random number in the range [0, n-1].
  friend Integer random(const Integer& x);

  // Factorial.
  void factorial();
  friend Integer factorial(const Integer& x);
  friend Integer factorial(long x);
  friend Integer factorial(unsigned long x);

  // Constants.
  static const Integer& zero();
  static const Integer& one();

private:
  // A multi-precision integer.
  mpz_t v;

  // The state for random number generation.
  class RandomState {
  public:
    RandomState();
    ~RandomState();

    gmp_randstate_t& operator ()();

  private:
    gmp_randstate_t state;
  };

  static RandomState random_state;
};

} // namespace Parma_Polyhedra_Library

namespace std {
  //! Specialize std::swap.
  void swap(Parma_Polyhedra_Library::Integer& x,
            Parma_Polyhedra_Library::Integer& y);
}

#include "Integer.inlines.hh"

#endif
