/*
 * lwp.h -- prototypes and structures for lightweight processes
 * Copyright (C) 1991-3 Stephen Crane.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 * 
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * author: Stephen Crane, (jsc@doc.ic.ac.uk), Department of Computing,
 * Imperial College of Science, Technology and Medicine, 180 Queen's
 * Gate, London SW7 2BZ, England.
 */
#ifndef _LWP_H
#define _LWP_H

/* maximum thread priority: number of priority levels */
#define MAXTPRI	8
/* `liveness' counter: check signals every `n' visits to the scheduler */
/* note: the lower this value, the more responsive the system but the */
/* more inefficient the context switch time */
#define LCOUNT	-1

#ifndef __signal_h
#include <signal.h>
#endif
#define SIGNALS (sigmask(SIGIO)|sigmask(SIGALRM))

/* this is a link to the real arch header file */
#include "arch.h"

#if defined(OWN_CONTEXT_SWITCH)
#ifdef __cplusplus
extern "C" {
#endif
int savep (jmp_buf);
void restorep (jmp_buf);
#ifdef __cplusplus
}
typedef int jmp_buf[16];
#endif
#else
#ifndef _SETJMP
#include <setjmp.h>
#define savep(x)	setjmp(x)
#define restorep(x)	longjmp(x, 1)
#endif
#endif

#ifndef _sys_time_h
#include <sys/time.h>
#endif

/* process control block.  do *not* change the position of context */
struct pcb {
	jmp_buf	context;	/* processor context area */
	void	*sbtm;		/* bottom of stack attached to it */
	int     size;           /* size of stack */
	void	(*entry)();	/* entry point */
	unsigned dead:1;	/* whether the process can be rescheduled */
	int	pri;		/* which scheduling queue we're on */
	struct timeval	dt;
	int	argc;		/* initial arguments */
	char	**argv;
	void	*envp;		/* user structure pointer */
	struct pcb	*next;
};

/* queue */
struct lpq {
	struct pcb *head, *tail;
};

/* semaphore */
struct sem {
	int count;
	struct lpq q;
};

/* signal struct */
struct siginfo {
	struct siginfo *next;
	int des;
	void (*han) (void *, int);
	void *ctx;
};

typedef double stkalign_t;

#ifdef __cplusplus
extern "C" {
#endif

/* semaphore operations */
struct sem *creats (int);
void signals (struct sem *);
void waits (struct sem *);

/* queue operations */
void toq (struct lpq *, struct pcb *);
struct pcb *hoq (struct lpq *);

/* process operations */
struct pcb *creatp (int, void (*)(), int, int, char **, void *);
void suicidep (void);
void destroyp (struct pcb *);
void readyp (struct pcb *);
void yieldp (void);
void *getenvp (struct pcb *);
void setenvp (struct pcb *, void *);
int prisetp (int);
void reschedp (void);

/* timer operations */
void delayp (long);
void onalarm (void);

/* initialisation */
struct pcb *initlp (int);
void initp (struct pcb *, void *);
void wrapp (void);

/* signal stuff */
int sigioset (int, void (*)(void *, int), void *);
int sigioclr (int);
#ifdef __cplusplus
}
#endif	/* __cplusplus */

#endif	/* _LWP_H */
