/* intercepts.c: This file is currently the only source file for
   the logwrites filter library.

   Written by Adam J. Richter
   Copyright 1996 Yggdrasil Computing, Inc.

   Logwrites and its documentation, including this file, may be freely
   copied under the terms and conditions of version 2 of the GNU General
   Public License, as published by the Free Software Foundation
   (Cambridge, Massachusetts, United States of America).

*/

/*
  New format of output:

	command cwd_realpath target_realpath target ...other args...
*/

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <fcntl.h>
#include <stdarg.h>
#include <unistd.h>
#include <stdlib.h>
#include <syscall.h>
#include <stdio.h>
#include <errno.h>

static int initialized = 0;
static int recursive = 0;

static char cwd[MAXPATHLEN + 1] = "(working directory unknown)";
static char *install_log = NULL;
static char do_backups = 0;
static char log_reads = 0;

/* Record current working directory and values of environment
   variables INSTALL_LOG and DO_BACKUPS. */
static void
initialize (void) {
  int len;

  if (initialized) return;

  install_log = getenv ("INSTALL_LOG");
  do_backups = getenv ("DO_BACKUPS") ? 1 : 0;
  log_reads = getenv ("LOG_READS") ? 1 : 0;

  initialized = 1;		/* getcwd recursively calls _xstat, so we
				   must pretend we are already initialized
				   before calling getcwd(). */
  strcpy(cwd,"?");
  if (getcwd(cwd,sizeof(cwd)) == NULL) {
    strcpy (cwd, "?");
    /* fail silently */
  }
}

/* log_event:   Record an event to the log file.
   The format of the evenents is:

   event cwd abs_path_to_target_file target_file_specified ...other args...

   log_event ("myevent", "myfile", "format of more args %s", "arg")


   myevent /current/dir /absolute/path/to/myfile myfile format of more args arg

*/

static void
log_event( char *cmd, const char *target_file, char *format, ... ) {
  va_list arg_ptr;
  FILE *log_file;
  char real_target_file[MAXPATHLEN+1];

  if (!initialized) initialize();
  if (recursive || !install_log) return;
  recursive = 1;
  if ((log_file = fopen (install_log, "a")) == NULL) {
    recursive = 0;
    return;
  }

  realpath(target_file, real_target_file);
  fprintf (log_file, "%s %s %s %s", cmd, cwd, real_target_file, target_file);

  if (format != NULL) {
    fprintf (log_file, " ");
    va_start(arg_ptr,format);
    vfprintf (log_file, format, arg_ptr);
    va_end(arg_ptr);
  }

  fprintf (log_file, "\n");

  fclose(log_file);
  recursive = 0;
}

static int
gen_backup_filename (const char *filename, char *backup_filename /* OUT */) {
  struct stat statbuf;
  int i;
  char *ptr;
  sprintf (backup_filename, "%s.old", filename);
  if (lstat (backup_filename, &statbuf) < 0) return;
  ptr = backup_filename + strlen(backup_filename);
  *(ptr++) = '.';
  for (i = 1;i != 0;i++) {
    sprintf (ptr, "%d", i);
    if (lstat (backup_filename, &statbuf) < 0) return 1;
  }
  backup_filename[0] = '\0';
  return 0;
}


static int
backup_file (const char *filename, char *backup_filename,
	     int regular_file_only) {
  struct stat statbuf;

  if (!initialized) initialize();
  if (!do_backups) return 0;
  if (recursive) return 0;
  if (lstat (filename, &statbuf) >= 0 &&
      (S_ISREG(statbuf.st_mode) || !regular_file_only) &&
      gen_backup_filename(filename, backup_filename)) {
    syscall(SYS_rename, filename, backup_filename);
    return 1;
  }
  return 0;
}


int
link (const char *oldpath, const char *newpath) {
  int result;

  if ((result = syscall(SYS_link, oldpath, newpath)) >= 0) {
    log_event("link", newpath, "%s", oldpath);
  }
  return result;
}

int
rename (const char *oldpath, const char *newpath) {
  int result;
  int backed_up;
  char backup_filename[MAXPATHLEN+1];

  backed_up = backup_file(newpath, backup_filename, 0);
  if ((result = syscall(SYS_rename, oldpath, newpath)) >= 0) {
    if (backed_up) {
      log_event("backup", backup_filename, "%s", newpath);
    }
    log_event("rename", newpath, "%s", oldpath);
  } else {
    if (backed_up) {
      syscall(SYS_rename, backup_filename, newpath);
    }
  }
  return result;
}

int
access (const char *pathname, int mode) {
  if (!initialized) initialize();
  if (log_reads) {
      log_event("access", pathname, "0%o", mode);
  }
  syscall(SYS_access, pathname, mode);
}

int
_xstat(int version, const char * path, struct stat * statbuf)
{
  if (!initialized) initialize();
  if (log_reads) {
      log_event ("stat", path, NULL);
  }
  switch(version)
  {
  case 1:
    return prev_stat (path, statbuf);
  default:
    errno = EINVAL;
    return -1;
  }
}

int
_lxstat(int version, const char * path, struct stat * statbuf)
{
  if (!initialized) initialize();
  if (log_reads) {
      log_event ("lstat", path, NULL);
  }
  switch(version)
  {
  case 1:
    return prev_lstat (path, statbuf);
  default:
    errno = EINVAL;
    return -1;
  }
}

int
symlink (const char *oldpath, const char *newpath) {
  int result;

  if ((result = syscall(SYS_symlink, oldpath, newpath)) >= 0) {
    log_event("symlink", newpath, "%s", oldpath);
  }
  return result;
}

int
unlink (const char *filename) {
  int result;
  char backup_filename[MAXPATHLEN+1];
  int backed_up;

  backed_up = backup_file(filename, backup_filename, 0);
  if ((result = syscall(SYS_unlink, filename)) >= 0) {
    if (backed_up) {
      log_event("backup", backup_filename, "%s", filename);
    }
    log_event("unlink", filename, NULL);
  } else {
    if (backed_up) {
      syscall(SYS_rename, backup_filename, filename);
    }
  }
  return result;
}

int
truncate (const char *filename, off_t length) {
  int result;
  char backup_filename[MAXPATHLEN+1];
  int backed_up;

  if (length == 0) {
    backed_up = backup_file(filename, backup_filename, 0);
  } else {
    backed_up = 0;
  }
  if ((result = syscall(SYS_unlink, filename)) >= 0) {
    if (backed_up) {
      log_event("backup", backup_filename, "%s", filename);
    }
    if (length == 0) {
      log_event("truncate", filename, "%d", length);
    }
  } else {
    if (backed_up) {
      syscall(SYS_rename, backup_filename, filename);
    }
  }
  return result;
}

int
prev_mknod (const char *filename, mode_t mode, dev_t dev) {
  int result;

  if ((result = syscall(SYS_mknod, mode, dev)) >= 0) {
    log_event("mknod", filename, "0%o 0x%x", mode, dev);
  }
  return result;
}

int
mkdir (const char *pathname, mode_t mode) {
  int result;

  if ((result = syscall(SYS_mkdir, pathname, mode)) >= 0) {
    log_event("mkdir", pathname, "0%o", mode);
  }
  return result;
}

/* XXX: Also intercept rmdir? */

int
open( const char *filename, int flags, ... ) {
    int result;
    mode_t mode;
    va_list arg_ptr;
    char backup_filename[MAXPATHLEN+1];
    int backed_up;
    int record_event;
    char *event;

    va_start(arg_ptr,flags);
    mode = va_arg(arg_ptr, mode_t);
    va_end(arg_ptr);

    if (!initialized) initialize();

    backed_up = 0;
    switch (flags & O_ACCMODE) {
    case O_WRONLY:
      event = "write";
      if ((flags & O_CREAT) && (flags & O_TRUNC)) {
	backed_up = backup_file(filename, backup_filename, 1);
      }
      break;
    case O_RDONLY:
      event = "read";
      break;
    case O_RDWR:
      event = "update";
      break;
    default:
      event = "unkown-open-mode";
      break;
    }
    if ((flags & O_ACCMODE) == O_WRONLY && (flags & O_CREAT) &&
	(flags & O_TRUNC)) {
      /* Only log write-only opens that will create the file if it doesn't
	 exist.  Otherwise, it is probably not an installation.  */
      backed_up = backup_file(filename, backup_filename, 1);
      record_event = 1;
    } else {
      backed_up = 0;
      record_event = log_reads;
    }

    result = syscall(SYS_open, filename, flags, mode);

    if (backed_up) {
      if (result < 0) {
	syscall(SYS_rename, backup_filename, filename);
      } else {
	log_event("backup", filename, "%s", backup_filename);
      }
    }
    if (record_event) {
      log_event(event, filename, "0x%x 0%o", flags, mode);
    }
    return result;
}

