/* Copyright is licensed under GNU LGPL.                (I.J. Wang, 2003)
*/

#ifndef WXTIME_H__
#define WXTIME_H__
#define WXTIME_VERSION 8

#include "wxret.h"
#include "wxstr.h"

/*
  WxTime is the class for time interval.
*/
class WxTime {
  public:
    typedef long long int SecsType;
    typedef      long int NanoType;
    static const char class_name[];

    WX_DECLARE_FAULT;

    WxTime() throw()
	     : _secs(Default_Secs), _nano(Default_Nano) {};

    WxTime(SecsType sec, NanoType nano) throw(std::exception,Fault);

    WxTime(const WxTime &c) throw()
	     : _secs(c._secs), _nano(c._nano) {};  // self-copy ok

    ~WxTime() throw() 
	      {}; 

    bool is_default(void) const throw()
	      { return (_secs==Default_Secs)&&(_nano==Default_Nano); };

    void reset(void) throw()
	      { _secs=Default_Secs; _nano=Default_Nano; };
    void reset(const WxTime &t) throw()
	      { _secs=t._secs; _nano=t._nano; };
    WxRet reset(SecsType sec, NanoType nano) throw(std::exception);

    void assign(const WxTime &rhs) throw()
	      { _secs=rhs._secs; _nano=rhs._nano; };

    WxRet assign(SecsType sec, NanoType nsec) throw(std::exception);

    WxTime abs(void) const throw(std::exception,Fault);

    WxRet add(SecsType sec, NanoType nsec) throw(std::exception);

    WxRet add(const WxTime& tm) throw(std::exception)
	      { return add(tm._secs,tm._nano); };

    //
    // Note: nsec is allowed to the maximum value of long int.
    //
    WxRet sub(SecsType sec, NanoType nsec) throw(std::exception);

    WxRet sub(const WxTime& tm) throw(std::exception)
	      { return sub(tm._secs,tm._nano); };

    const WxTime& operator =(const WxTime &rhs) throw()
	      { _secs=rhs._secs; _nano=rhs._nano; return(*this); };

    WxTime operator -() const throw(std::exception,Fault);

    WxTime operator +(const WxTime &rhs) const throw(std::exception,Fault);

    //
    // [Syn] t1-t2
    //
    WxTime operator -(const WxTime &rhs) const throw(std::exception,Fault);

    const WxTime & operator +=(const WxTime &rhs) throw(std::exception,Fault);

    const WxTime & operator -=(const WxTime &rhs) throw(std::exception,Fault);

    SecsType secs(void) const throw()
	      { return _secs; };

    //
    // Note: Let tm be an object of WxTime. The following equation holds.
    //       tm == WxTime(tm.secs(),tm.nano())
    //
    NanoType nano(void) const throw() 
	      { return _nano; };

    //
    // [Syn] Time comparisons
    //
    bool operator ==(const WxTime &rhs) const throw() 
	      { return (_nano==rhs._nano)&&(_secs==rhs._secs); };
    bool operator !=(const WxTime &rhs) const throw() 
	      { return (_nano!=rhs._nano)||(_secs!=rhs._secs); };
    bool operator >(const WxTime &rhs) const throw();
    bool operator >=(const WxTime &rhs) const throw();
    bool operator <(const WxTime &rhs) const throw();
    bool operator <=(const WxTime &rhs) const throw();

    //
    // [Internal] Convert the std::time_t t based on 1970-1-1 0800 to the seconds 
    //            since 1970-1-1 00:00.00 and store into secs
    //
    // Note: This function is asserted not to get error.
    //       (for the time being, sizeof(SecsType)>time_t)
    //
    inline static SecsType wx__time_t2secs(std::time_t t) throw()
              { return(static_cast<WxTime::SecsType>(t)+WX__TIME_T_OFFSET); };

    //
    // [Internal] Convert the seconds secs since 1970-1-1 00:00.00 to the time_t
    //            from 1970-1-1 08:00.0
    //
    // [Ret] 0= ok
    //      -1= conversion failed
    //
    inline static int wx__secs2time_t(WxTime::SecsType secs,std::time_t& t) throw()
              {
               const WxTime::SecsType tmp=secs-WX__TIME_T_OFFSET;
               if(tmp>=secs) {
                 return(-1);
               }
               t=static_cast<std::time_t>(tmp);
               if(tmp!=static_cast<WxTime::SecsType>(t)) {
                 return(-1);
               }
               return(0);
             };

  protected:
    //
    // [Internal] Construct object from the time_t based on 1970-1-1 0800
    //
    inline WxTime(std::time_t t) throw()
              { _secs=wx__time_t2secs(t); _nano=0; };
              
  private:
    //
    // Adding this number to time_t obtains the ellapsed seconds since 1970-1-1 
    // zero o'clock
    //
    static const SecsType WX__TIME_T_OFFSET=28800;

    //
    // [Syn] Normalize sec/nsec
    //       sec/nsec is normalized iff
    //
    //       1. (sec>0) &&(nsec>=0)&&(nsec<Giga)
    //       2. (sec<0) &&(nsec<=0)&&(nsec>-Giga)
    //       3. (sec==0)&&(nsec>-Giga)&&(nsec<Giga)
    // 
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    // 
    // [Ret] OK    
    //       WXM_ERANGE parameter normalization failure
    // 
    // [Property]
    //       nsec>0 --> sec>=0
    //       nsec<0 --> sec<=0
    //
    static WxRet _normalize(SecsType& sec, NanoType& nsec) throw(std::exception);
    static const SecsType Default_Secs=0;
    static const NanoType Default_Nano=0;
    static const NanoType Giga=WX_CONST_GIGA;    // 1x10^9

    SecsType _secs;
    NanoType _nano;

};

namespace Wx {
 WxRet sleep_till(const WxTime &wtm) throw(std::exception);

 //
 // Note: Platform may limit the interval (2038.1.18)
 //
 WxTime now(void) throw(std::exception);

 WxRet set_systime(const WxTime& t) throw(std::exception);

 WxStr what_is(const WxTime& tm) throw(std::exception);
};

#endif
