/* Copyright is licensed under GNU LGPL.                (I.J. Wang, 2003)
*/

#include "wxtime.h"
#include "wxmutex.h"
#include "wxnum.h"
#include <limits>
#include <time.h>   // for POSIX4 real time clock
#include <ctime>

//#undef _POSIX_TIMERS

#ifndef _POSIX_TIMERS
#include <sys/time.h>
//
// Mutex dedicated for sycronizing system time access (gettimeofday)
//
static WxMutex wx__cmtx;
#endif

class Wx__Assert_Check {
  public:
    Wx__Assert_Check() {
       if(sizeof(WxTime::SecsType)<=sizeof(std::time_t)) {
         WX_ABORT();
       }
    };
} static const wx__assert_check__;

const char WxTime::class_name[]="WxTime";

WxRet WxTime::_normalize(SecsType& sec, NanoType& nsec) throw(std::exception)
try {
 std::ldiv_t fr=std::ldiv(nsec,Giga);   // NanoType must be type long int

 SecsType ts=sec+static_cast<SecsType>(fr.quot);

 // Check for ts overflow
 if(fr.quot>=0) {
   if(ts<sec) {
     WX_RETURN(WXM_ERANGE);   // result not representable (sign inversion)
   }
 } else {
   if(ts>=sec) {
     WX_RETURN(WXM_ERANGE);   // result not representable (sign inversion)
   }
 }

 // make sec/nano the same sign
 if(ts>0) {
   if(fr.rem<0) {
     sec=ts-1;
     nsec=fr.rem+Giga;
     return(OK);
   }
   // FALLTHROUGH
 } else if(ts<0) {
   if(fr.rem>0) {
     sec=ts+1;
     nsec=fr.rem-Giga;
     return(OK);
   }
   // FALLTHROUGH
 }
 sec=ts;
 nsec=fr.rem;
 return(OK);
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
  WX_THROW(Wx_general_error());
};

WxTime::WxTime(SecsType sec, NanoType nsec) throw(std::exception,Fault)
try {
 WxRet r=_normalize(sec, nsec); 
 if(r!=OK) {
   WX_THROW( Fault(r) );
 }
 _secs=sec;
 _nano=nsec;
}
catch(const WxTime::Fault& e) {
  throw;
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
  WX_THROW(Wx_general_error());
};

WxRet WxTime::reset(SecsType sec, NanoType nano) throw(std::exception)
try {
 WxRet r=_normalize(sec, nano); 
 if(r!=OK) {
   WX_RETURN(r);
 }
 _secs=sec;
 _nano=nano;
 return(OK);
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
  WX_THROW(Wx_general_error());
};

WxRet WxTime::assign(SecsType sec, NanoType nsec) throw(std::exception)
try {
 WxRet r=_normalize(sec, nsec); 
 if(r!=OK) {
   WX_RETURN(r);
 }
 _secs=sec;
 _nano=nsec;
 return(OK);
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
  WX_THROW(Wx_general_error());
};

WxTime WxTime::abs(void) const throw(std::exception,Fault)
try {
 WxTime v;
 if(_secs>0) {
   v._secs=_secs;
   v._nano=_nano;
 } else if(_secs<0) {
   v._secs=-_secs;
   v._nano=-_nano;
   if(v._secs<0) {
     WX_THROW( Fault(WXM_MATHNEG) ); // inversion failed
   }
 } else {
   v._secs=0;
   if(_nano>=0) {
     v._nano=_nano;
   } else {
     v._nano=-_nano;
   }
 }
 return(v);
}
catch(const WxTime::Fault&) {
  throw;
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
  WX_THROW(Wx_general_error());
};

WxRet WxTime::add(SecsType sec, NanoType nsec) throw(std::exception)
try {
 WxTime v=*this;

 v._secs+=sec;
 v._nano+=nsec;

 if(_secs>0) {
   if(v._secs<=sec) {
     WX_RETURN(WXM_ERANGE);
   }
 } else {
   if(v._secs>sec) {
     WX_RETURN(WXM_ERANGE);
   }
 }

 WxRet r=_normalize(v._secs,v._nano);
 if(r!=OK) {
   WX_RETURN(r);
 }
 _secs=v._secs;
 _nano=v._nano;
 return(OK);
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
  WX_THROW(Wx_general_error());
};

WxRet WxTime::sub(SecsType sec, NanoType nsec) throw(std::exception)
try {
 WxTime v=*this;

 v._secs-=sec;
 v._nano-=nsec;

 if(sec>0) {
   if(_secs<=v._secs) {
     WX_RETURN(WXM_ERANGE);
   }
 } else {
   if(_secs>v._secs) {
     WX_RETURN(WXM_ERANGE);
   }
 }

 WxRet r=_normalize(v._secs,v._nano);
 if(r!=OK) {
   WX_RETURN(r);
 }
 _secs=v._secs;
 _nano=v._nano;
 return(OK);
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
  WX_THROW(Wx_general_error());
};

WxTime WxTime::operator -() const throw(std::exception,Fault)
try {
 WxTime tm(-_secs,-_nano);
 if((_secs!=0)&&(_secs==tm._secs)) { // might not be portable
   WX_THROW( Fault(WXM_MATHNEG) );      // negation failed (-LLONG_MAX-1)
 }
 return(tm);
}
catch(const WxTime::Fault&) {
  throw;
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
  WX_THROW(Wx_general_error());
};

WxTime WxTime::operator +(const WxTime &rhs) const throw(std::exception,Fault)
try {
 WxTime v=*this;

 v._secs+=rhs._secs;
 v._nano+=rhs._nano;

 if(_secs>0) {
   if(v._secs<=rhs._secs) {
     WX_THROW( Fault(WXM_ERANGE) );
   }
 } else {
   if(v._secs>rhs._secs) {
     WX_THROW( Fault(WXM_ERANGE) );
   }
 }

 WxRet r=_normalize(v._secs,v._nano);
 if(r!=OK) {
   WX_THROW( Fault(r) );
 }
 return(v);
}
catch(const WxTime::Fault&) {
  throw;
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
  WX_THROW(Wx_general_error());
};

WxTime WxTime::operator -(const WxTime &rhs) const throw(std::exception,Fault)
try {
 WxTime v=*this;

 v._secs-=rhs._secs;
 v._nano-=rhs._nano;

 if(rhs._secs>0) {
   if(_secs<=v._secs) {
     WX_THROW( Fault(WXM_ERANGE) );
   }
 } else {
   if(_secs>v._secs) {
     WX_THROW( Fault(WXM_ERANGE) );
   }
 }

 WxRet r=_normalize(v._secs,v._nano);
 if(r!=OK) {
   WX_THROW( Fault(r) );
 }
 return(v);
}
catch(const WxTime::Fault&) {
  throw;
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
  WX_THROW(Wx_general_error());
};

const WxTime & WxTime::operator +=(const WxTime &rhs) throw(std::exception,Fault)
try {
 *this=(*this)+WxTime(rhs);
 return(*this);
}
catch(const WxTime::Fault&) {
 throw;
}
catch(const Wx_bad_alloc&) {
 throw;
}
catch(const std::bad_alloc&) {
 WX_THROW( Wx_bad_alloc() );
}
catch(const Wx_general_error&) {
  throw;
}
catch(...) {
  WX_THROW(Wx_general_error());
};

const WxTime & WxTime::operator -=(const WxTime &rhs) throw(std::exception,Fault)
try {
 *this=(*this)-WxTime(rhs);
 return(*this);
}
catch(const WxTime::Fault&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(const Wx_general_error&) {
  throw;
}
catch(...) {
  WX_THROW(Wx_general_error());
};

bool WxTime::operator >(const WxTime &rhs) const throw()
{
 if(_secs>rhs._secs) {
   return true;
 } else if(_secs<rhs._secs) {
   return false;
 } else {}

 if(_nano>rhs._nano) {
   return true;
 } else {
   return false;
 }
}

bool WxTime::operator >=(const WxTime &rhs) const throw()
{
 if(_secs>rhs._secs) {
   return true;
 } else if(_secs<rhs._secs) {
   return false;
 } else {}

 if(_nano>=rhs._nano) {
   return true;
 } else {
   return false;
 }
}

bool WxTime::operator <(const WxTime &rhs) const throw()
{
 if(_secs<rhs._secs) {
   return true;
 } else if(_secs>rhs._secs) {
   return false;
 } else {}

 if(_nano<rhs._nano) {
   return true;
 } else {
   return false;
 }
}

bool WxTime::operator <=(const WxTime &rhs) const throw()
{
 if(_secs<rhs._secs) {
   return true;
 } else if(_secs>rhs._secs) {
   return false;
 } else {}

 if(_nano<=rhs._nano) {
   return true;
 } else {
   return false;
 }
}

WxStr Wx::what_is(const WxTime& tm) throw(std::exception)
try {
 const long long tm_secs=tm.secs();
 const long tm_nano=tm.nano();

 if((tm_nano<=-WX_CONST_GIGA)||(tm_nano>=WX_CONST_GIGA)) {
   WX_THROW(Wx_general_error()); // object assertion failed
 }

 WxRet r;
 WxStr str1;

 if(tm_nano==0) {
   if((r=Wx::append_number(str1,tm_secs, 10))!=OK) {
     WX_THROW(Wx_general_error()); // only WXM_EINVAL(invalid radix)
   }
   str1+=".0";
   return str1;
 }

 WxStr str2;

 //
 // Make str1 the representation of _secs
 //      str2 the representation of _nano
 //
 if(tm_nano<0) {
   if(tm_secs<0) {
     if((r=Wx::append_number(str1,tm_secs, 10))!=OK) {
       WX_THROW(Wx_general_error()); // only WXM_EINVAL(invalid radix)
     }
   } else {
     if(tm_secs!=0) {
       WX_THROW(Wx_general_error()); // assertion(_secs==0) failed
     }
     str1="-0";
   }
   if((r=Wx::append_number(str2, -tm_nano, 10))!=OK) {
     WX_THROW(Wx_general_error()); // only WXM_EINVAL(invalid radix)
   }
 } else {
   if((r=Wx::append_number(str1,tm_secs, 10))!=OK) {
     WX_THROW(Wx_general_error()); // only WXM_EINVAL(invalid radix)
   }
   if((r=Wx::append_number(str2, tm_nano, 10))!=OK) {
     WX_THROW(Wx_general_error()); // only WXM_EINVAL(invalid radix)
   }
 }
 str1+='.';

 // assert( str2[0]!=0 )
 int lz=9-(int)str2.size();
 if(lz<0) {
   WX_THROW(Wx_general_error());  // str2.size() should be 1-9
 }
 for( ; lz>0; --lz) {
   str1+='0';   // prepend leading zero's
 }

 //
 // locate the last non-'0' character in str2
 //
 for(lz=str2.size()-1; (lz>=0)&&(str2[lz]=='0'); --lz) {};

 r=str1.append(str2,0,lz+1);
 if(r!=OK) {
   WX_THROW(Wx_general_error());
 }
 return(str1);
}
catch(const WxStr::Fault& e) {
 WX_THROW( Wx_general_error() );
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(const Wx_general_error&) {
  throw;
}
catch(...) {
  WX_THROW(Wx_general_error());
};

WxTime Wx::now(void) throw(std::exception)
try {
#ifdef _POSIX_TIMERS
 struct timespec tm_now;
 if(::clock_gettime(CLOCK_REALTIME,&tm_now)!=0) {
   WX_THROW( Wx_bad_errno(errno) ); // man. only EINVAL which should not happen here.
 }

 //
 // timespec conversion should not generate exception Fault
 // note: WxTime ctor is assumed not to cause exception
 //
 return WxTime(WxTime::wx__time_t2secs(tm_now.tv_sec),tm_now.tv_nsec);
#else
 struct ::timeval tm_now;
 WxRet r;
 if((r=wx__cmtx.lock())!=OK) {
   WX_THROW(Wx_general_error());
 }
 //
 // guess this function is not thread-safe, so a mutex supplied.
 //
 if(::gettimeofday(&tm_now,0)!=0) {
   //
   // man indicates all errors are associated with time zone, which is not 
   // used int time(), so it is not supposed to happen here.
   //
   wx__cmtx.unlock();    // not check the error, for result is the same.
   WX_THROW(Wx_general_error());
 }
 if((r=wx__cmtx.unlock())!=OK) {
   WX_THROW(Wx_general_error());
 }

 //
 // timespec conversion should not generate exception Fault
 // note: WxTime ctor is assumed not to cause exception
 //
 return WxTime(wx__time_t2secs(tm_now.tv_sec),tm_now.tv_usec*1000);
#endif
}
catch(const WxTime::Fault& e) {
  WX_THROW( Wx_general_error() );  // should not get this
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(const Wx_general_error&) {
  throw;
}
catch(...) {
  WX_THROW(Wx_general_error());
};

WxRet Wx::set_systime(const WxTime& wxtm) throw(std::exception)
try {
#ifdef _POSIX_TIMERS
 struct timespec tm_now;
 if(WxTime::wx__secs2time_t(wxtm.secs(),tm_now.tv_sec)!=0) {
   WX_RETURN(WXM_ERANGE);        // conversion failure
 }
 tm_now.tv_nsec=wxtm.nano();

 if(::clock_settime(CLOCK_REALTIME,&tm_now)!=0) {
   const int v=errno;
   switch(v) {
     case EPERM: WX_RETURN(WXM_EPERM);
     case EINVAL: WX_RETURN(WXM_ERANGE); // guess this errmsg
     case EFAULT: // FALLTHROUGH
     default:    WX_THROW( Wx_bad_errno(v) );  // should not happen
   };
 }
 return(OK);
#else
 WxRet r;
 struct timeval tm_now;

 if(WxTime::wx__secs2time_t(wxtm.secs(),tm_now.tv_sec)!=0) {
   WX_RETURN(WXM_ERANGE);        // conversion failure
 }

 if((r=wx__cmtx.lock())!=OK) {
   WX_THROW(Wx_general_error());
 }
 tm_now.tv_usec=static_cast<long>(wxtm.nano()/1000);

 // guess this function is not thread-safe, so a mutex supplied
 const int en=(::settimeofday(&tm_now,0)==0)? 0:errno;
 if((r=wx__cmtx.unlock())!=OK) {
   WX_THROW(Wx_general_error());
 }
 switch(en) {
   case 0: return(OK);
   case EPERM: WX_RETURN(WXM_EPERM);
   case EINVAL: // FALLTHROUGH
   case EFAULT: // FALLTHROUGH
   default:    WX_THROW( Wx_bad_errno(en) );  // should not happen
 };
#endif
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(const Wx_general_error&) {
  throw;
}
catch(...) {
  WX_THROW(Wx_general_error());
};

/*
WxRet WxTime::set_systime(long long sec, long nsec) throw(std::exception,Fault)
try {
 return set_systime(WxTime(sec,nsec));
}
catch(const WxTime::Fault& e) {
  if(e==WXM_ERANGE) {
    WX_RETURN(WXM_ERANGE);
  }
  WX_THROW(Wx_general_error());    // unexcepted errmsg
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(const Wx_general_error&) {
  throw;
}
catch(...) {
  WX_THROW(Wx_general_error());
};
*/
//----------------------------------------------------------------------------
//
// [Internal] Class that just sleeps
//            This class is dedicated for Wx::sleep_till.
//
// [Exception] Wx_general_error
//             Wx_bad_errno
//             Wx_bad_alloc
//
// [Exception] WxRet
//             WXM_ERANGE
//
class Wx__Sleep {
  public:
    Wx__Sleep(const WxTime& wtm) throw(std::exception)
             try {
               // ::pthread_cond_timedwait() interpret time_t from 1970-1-1,0800.
               if(WxTime::wx__secs2time_t(wtm.secs(),_ts.tv_sec)!=0) {
		  WX_THROW( WxRet(WXM_ERANGE) );
               }
               
               _ts.tv_nsec=wtm.nano();

               ::pthread_mutex_init(&_mtx,NULL);   // manual: always ok
               const int v=::pthread_mutex_lock(&_mtx);
               if(v!=0) {
                 ::pthread_mutex_destroy(&_mtx);
                 WX_THROW( Wx_bad_errno(v) );
               }

               // Note: No exception for the following codes.
               ::pthread_cond_init(&_cond,NULL);   // manual: always ok

               struct ::_pthread_cleanup_buffer cubuf;
               ::_pthread_cleanup_push(&cubuf,
               reinterpret_cast<void(*)(void*)>(_cleanup_func), this);

               while(::pthread_cond_timedwait(&_cond,&_mtx,&_ts)!=ETIMEDOUT) {};

               ::_pthread_cleanup_pop(&cubuf,true); 
              }
              catch(const WxRet&) {
                throw;
              }
              catch(const std::bad_alloc&) {
                 WX_THROW( Wx_bad_alloc() );
              }
              catch(...) {
                 WX_THROW(Wx_general_error());
              };

    ~Wx__Sleep() throw() { /* deliberately do nothing */ };
  private:
    static void _cleanup_func(Wx__Sleep* optr) throw();

    pthread_cond_t _cond;
    struct timespec _ts;
    pthread_mutex_t _mtx;

    Wx__Sleep();  // not to use
};

void Wx__Sleep::_cleanup_func(Wx__Sleep* optr) throw()
{
  ::pthread_mutex_unlock(&optr->_mtx);
  ::pthread_mutex_destroy(&optr->_mtx);
  ::pthread_cond_destroy(&optr->_cond);
};

WxRet Wx::sleep_till(const WxTime &wtm) throw(std::exception)
try {
 Wx__Sleep delay_obj(wtm);
 return(OK);
}
catch(const WxRet& e) {
  if(e==WXM_ERANGE) {
    WX_RETURN(WXM_ERANGE);
  }
  WX_THROW(Wx_general_error());
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_general_error&) {
  throw;
}
catch(...) {
  WX_THROW(Wx_general_error());
};
