/* Copyright is licensed under GNU LGPL.                (I.J. Wang, 2003)

   WxThread is the base class to handle a thread.

   Derived classes must reimplement the pure virtual function tmain(), which runs
   the created thread.

   Note: Add -lpthread -lrt to g++ linker options

   tba+ use as array

  [Avoided C Use] pthread_attr_t, pthread_t,
        pthread_create,pthread_setcancelstate,thread_self,thread_equal
        pthread_attr_init, pthread_attr_destroy
        pthread_attr_setdetachstate
        pthread_init,pthread_create
        pthread_cancel,pthread_exit,pthread_testcancel,sched_yield,pthread_yield

  Note: Some of the error messages are based on Linux Programmer's Manual.
*/

#ifndef WXTHREAD_H__
#define WXTHREAD_H__

#define WXTHREAD_VERSION 8

#include "wxmutex.h"
#include "wxcond.h"
#include "wxstr.h"

/*
   Thread Identifier class
*/
class WxThreadID {
  public:
    // 
    // [Not Cancelable]
    // 
    // [Syn] Construct default object
    //
    WxThreadID() throw() 
              : _tid(Default_ThreadID) {};
    
    ~WxThreadID() throw() 
              {};

    //
    // [Not Cancelable]
    //
    // [Syn] Is object the same state as default constructed one
    //
    bool is_default(void) const throw()
              { return _tid==Default_ThreadID; };

    //
    // [Syn] Reset object
    //
    void reset(void) throw()
              { _tid=Default_ThreadID; };

    //
    // [Syn] Assign object
    //
    const WxThreadID& operator =(const WxThreadID& tid) throw()
              { _tid=tid._tid; return(*this); };

    //
    // [Syn] Compare thread identifiers for the same running thread
    //
    // [Ret] true= id1 and id2 refer to the same running thread
    //       false= otherwise
    //
    // Note: id1 and id2 must refer to running thread(s), otherwise
    //       result is meaningless.
    //
    inline bool operator==(WxThreadID id) const throw()
              { return ::pthread_equal(id._tid,_tid); };
    inline bool operator!=(WxThreadID id) const throw()
              { return !operator==(id); };

    unsigned int wx_id_value(void) const throw()
              { return _tid; };
  private:
    static const ::pthread_t Default_ThreadID;
    inline WxThreadID(::pthread_t id) throw() : _tid(id) {};
    ::pthread_t _tid;

    friend class Wx_Thread;  // Wx_Thread need to use WxThreadID(::pthread_t)
};

namespace Wx {
 //
 // [Syn] Get the descriptive string of this object.
 //
 // [Exception] Wx_bad_alloc
 //             Wx_general_error
 //
 // [Ret] The descriptive string of this object.
 //
 extern WxStr what_is(const WxThreadID& tid) throw(std::exception);
};

/*
   Wx_Thread is the base class to run virtual member function 'tmain(...)' in the
   created thread.

   Basic Usage:

     1.Declares an object of the class which inherits WxThread. The pure virtual 
       function tmain() must be reimplemented and runs the thread.

     2.Call begin(), then tmain() is scheduled to run in the created thread.

   Note: Several rules found to use the thread class correctly. See comments in
         ~Wx_Thread(), reset, tmain_enter, tmain, tmain_leave.

*/
class Wx_Thread {
  public:
    WX_DECLARE_FAULT;  // declaraion of Fault

    //
    // Thread state enumeration
    //
    enum ThreadState{
           Ready,       // Object is construtced in Ready state, no thread.
           Running,     // Thread is created to run tmain()
           Terminated   // Thread terminated
    };

    //----------------------------------------------------------------------
    // Thread function tmain() should not use these public members  except
    // static ones.
    //---------------------------------------------------------------------
	 
    //
    // [Not Cancelable]
    //
    // [Syn] Construct the default thread object.
    //       The object is constructed with ThreadState Ready.
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    //
    Wx_Thread() throw(std::exception);

    //
    // [Not Cancelable]
    //
    // [Syn] Destroy object
    // 
    // Note: Be sure cancellation points are executed in a given fixed time interval
    //       in tmain(). Otherwise, destructor or reset() may wait forever.
    //
    // Note: All destructors of the derving class should stop the running thread
    //       before other calls, or the running tmain() may access the destroyed
    //       data members. (this is the problem of the construction of thread class)
    //
    //       For example, the first lines in the destructor of all derived classes
    //       should be equivalent to the following:
    //
    //          cancel();             // send cancel signal
    //          wait_stopped();       // wait until terminated state if run.
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    //
    virtual ~Wx_Thread() throw(std::exception);
 
    //
    // [Not Cancelable]
    //
    // [Syn] Is object the same state as default constructed one
    //
    bool is_default(void) const throw(std::exception);

    // 
    // [Cancel Point]
    // [Syn] Reset the object to the state as default constructed.
    //       (running thread will get a cancel signal)
    //
    // Note: Be sure cancellation points are executed in a given fixed time interval
    //       in tmain(). Otherwise, destructor or reset() may wait forever.
    // Note: tmain() may not call this function
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    //
    // [Ret] OK  object is reset to the state as default constructed.
    //               (running thread has been cancelled if any)
    //
    // [Refer] constructor Wx_Thread()
    //
    WxRet reset(void) throw(std::exception);
	 
    //
    // [Not Cancelable]
    //
    // [Syn] Get the thread state
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    //
    // [Ret] enum ThreadState
    //       Ready   ...... thread has never begun
    //       Running ...... thread is running
    //       Terminated ... thread terminated
    //
    // [Refer] ThreadState thread state
    //
    ThreadState thread_state(void) const throw(std::exception);

    //
    // [Not Cancelable]
    //
    // [Syn] Creates a thread to run tmain().
    //       The thread is created with cancellation enabled.
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    //
    // [Ret] OK   tmain() is scheduled to run the thread (may not now)
    //       WXM_EEXIST thread is running or terminated (reset to reuse).
    //       WXM_EAGAIN not enough system resource for the new thread
    //
    WxRet begin(void) throw(std::exception);

    //
    // [Not Cancelable]
    //
    // [Syn] Send signal to cancel (stop) the running tmain()
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    //
    // [Ret] OK   cancel signal is sent (thread may not terminate yet).
    //       WXM_ENOENT no running thread  (not in Running state)
    //       WXM_EPERM tmain() can not call this function
    //
    // Note: The exit code for canceled thread is WXM_THRDCAN
    //
    // [Reffer] cancel_point
    //
    WxRet cancel(void) throw(std::exception);
	 
    //
    // [Not Cancelable]
    //
    // [Syn] Get the tmain() exit code
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    //
    // [Ret] The exit WxRet of tmain(). The returned value only valid if a thread
    //       was created and terminated. WXM_THRDCAN is returned if the thread was 
    //       canceled. Default object gets default errmsg.
    // 
    WxRet exit_code(void) const throw(std::exception);
	 
    //
    // [Cancel Point]
    //
    // [Syn] Wait until tmain() stopped (terminated state)
    //
    // Note: tmain() may not call this function
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    //
    // [Ret] OK  tmain() has been run and stopped.
    //       WXM_EPERM called from tmain()
    //       WXM_ENOENT object is in Ready state (thread not run)
    // 
    WxRet wait_stopped(void) const throw(std::exception);
    WxRet wait_nready(void) const throw(std::exception);
	 
    //
    // [Syn] Get the thread ID that runs tmain()
    //
    // [Ret] ID of the thread that runs tmain(). The returned ID is
    //       invalid if there is no running thread (tmain() not begun)
    //
    inline WxThreadID tmain_id(void) const throw()
              { return WxThreadID(_thrd_id); };

    //
    // [Syn] Get id of the calling thread
    //
    // [Ret] Id of the calling thread
    //
    inline static WxThreadID thread_self(void) throw()
              { return WxThreadID(::pthread_self()); };

    //
    // [Not Cancelable]
    //
    // [Static] Yield execution for other threads to resume.
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    //
    // [Ret] OK execution yielded (may or may not happen)
    //       WxErrMsg(errno) ...... do no care for now
    //
    static WxRet yield(void) throw(std::exception);
	 
    static bool set_cancelable(bool en) throw(std::exception);

    //
    // [Cancel Point] 
    //
    // [Static] Declare explicitly that cancellation may take place in this
    //          function, nothing else is otherwise done.
    //
    // [Reffer] set_cancelable(...)
    //
    inline static void cancel_point(void) throw()
              { ::pthread_testcancel(); };

    //
    // [Not Cancelable]
    //
    // [Static] Get the total number of running threads.
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    //
    // [Ret] number the total number of running threads. 
    //       (created by this class)
    //
    static size_t total_threads(void) throw(std::exception);

  protected:
    //------------------------------------------
    // This protected interface is for tmain().
    //------------------------------------------

    //
    // [Not Cancelable]
    //
    // [Syn] Exit the thread
    //
    // Note: Objects constructed in thread might not be destroyed.
    //       For example, auto_ptr<...> objects would not be destroyed by
    //       this exit function. See tmain_leave(),WxPtr. 
    //
    // [Ret] Never Return
    //
    void exit(WxRet) throw(std::exception) __attribute__ ((__noreturn__));

    //
    // [Not Cancelable]
    //
    // [Syn] Abort the thread
    //
    // Note: NOT IMPLEMENTED, current function is the same as std::terminate()
    // Note: Objects constructed in thread may not be destroyed. See tmain_leave().
    // 
    // [Ret] Never Return
    //
    void abort(void) const throw() __attribute__ ((__noreturn__));

    //
    // [Not Cancelable]
    //
    // [Syn] The created thread invokes this virtual function before entering
    //       tmain(). Cancelability is desabled in this function.
    //
    // Note: Do not access any member of this base class, deadlock may result.
    //
    // Note: This virtual function is actually provided for symmetry, not
    //       particularly special in functionality.
    //       It is nice to put the resource allocation codes here, and the
    //       release code in tmain_leave().
    //
    virtual void tmain_enter(void) throw() {};

    //
    // [Syn] The pure virtual function to run the thread function.
    //
    // Note: Be sure cancellation points are executed in a given fixed time 
    //       interval in tmain(). Otherwise, destructor may wait forever.
    // Note: Do not use the public non-static member functions in the thread to
    //       operate the thread itself.
    // Note: Thread using contains several pitfalls.
    //       It could be a bug without caution to declare objects of classes 
    //       that require destructor to release allocated resources, inside the
    //       function or called from inside it.
    //       For example, WxByteTape,WxByteFlow and their inheritance, WxMutex and 
    //       WxCond... are such classes. And notably, std::string and other STL classes.
    //       To use such objects locally(auto) inside the thread function, use
    //       operator new and WxPtr to handle it. The same rule apply also to
    //       class that contains such class as data member.
    //
    virtual WxRet tmain(void) throw()=0;

    //
    // [Not Cancelable]
    //
    // [Syn] This virtual function is called immediately after the thread leaves 
    //       tmain() via return,exit or cancel.
    //       Derived class reimplements this function to do the cleanup.
    //       Cancelability is desabled in this function.
    // Note: Normally the resources allocated in the thread (tmain) are
    //       released in this function. 
    //       Be careful not to repeat the cleanup from the destructor.
    //
    // Note: Do not access any member of this base class, deadlock may result.
    //
    virtual void tmain_leave(void) throw() {};
          
  private:
    // [Not Cancelable]
    static void _once_func(void) throw();
    static pthread_once_t _once_var;

    //
    // [Not Cancelable]
    //
    // [Syn] The created thread function. tmain() is invoked from inside.
    //
    // [Ret] NULL
    //
    static void *_begin_func(Wx_Thread *obj_ptr) throw();

    //
    // [Not Cancelable]
    //
    // [Syn] Cleanup function of the thread (exit/cancel). 
    //       tmain_leave() is invoked from inside.
    //
    static void _cleanup_func(Wx_Thread *obj_ptr) throw();

    //
    // Static members for counting the threads.
    //
    static size_t _tt_thrds;  // number of total running threads WxThread created
    static WxMutex _tt_mtx;   // mutex for _tt_thrds

    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    //
    static void inc_thread_count(void) throw(std::exception);
    static void dec_thread_count(void) throw(std::exception);
  private:
    mutable WxCond _thrd_stat_cond;// condition that _thrd_stat set value
    ThreadState _thrd_stat;
    ::pthread_attr_t _thrd_attr;
    ::pthread_t _thrd_id;
    WxRet _exit_obj;
    bool _in_destroy;              // destructor set this flag true
    mutable WxMutex _mtx;          // mutex for all the private data members
 private:
    Wx_Thread(const Wx_Thread&);   // not to used
    const Wx_Thread& operator =(const Wx_Thread&);   // not to used

    void* set_terminate(void*);    // not defined yet
    void* set_unexpected(void*);   // not defined yet
};

/*
*/
class WxThread : public Wx_Thread {
  public:
    WX_DECLARE_FAULT;  // declaraion of Fault

    WxThread() throw(std::exception);
    ~WxThread() throw(std::exception);
  private:
       WxThread(const WxThread&);   // not to use
       const WxThread& operator =(const WxThread&);   // not to use
};

/*
  [Thread-Safe]
  [Internal] Class for disabling cancelability of the calling thread
*/
class Wx__NoCancel {
  public:
    //
    // [Syn] Construct object. The cancelablity of the calling thread is disabled.
    // 
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    // 
    Wx__NoCancel() throw(std::exception)
              { _pre_stt=WxThread::set_cancelable(false); };

    //
    // [Syn] Destructor. Recover the previous cancelablity of the calling thread.
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    // 
    ~Wx__NoCancel() throw(std::exception)
              { WxThread::set_cancelable(_pre_stt); };

  private:
     bool _pre_stt;
};

//
// Define a Wx__NoCancel object
//
#define WX_NOCANCEL Wx__NoCancel wx_auto_obj_nocancel

#endif
