/* Copyright is licensed under GNU LGPL.                (I.J. Wang, 2003)
*/

#include "wxthread.h"
#include "wxnum.h"

//
// Because no document found for the default value of ::pthread_t, the 
// following lines uses whatever value it is initialized. (may not work)
//
static ::pthread_t wx__default_thread_id__;
const ::pthread_t WxThreadID::Default_ThreadID=wx__default_thread_id__;

WxStr Wx::what_is(const WxThreadID& tid) throw(std::exception)
try {
 WxStr v; 
 // describe WxThreadID in hexidecimal number 
 Wx::append_number(v,tid.wx_id_value(),16);
 return(v); 
}
catch(const WxStr::Fault& e) {
 WX_THROW( Wx_general_error() );
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const Wx_general_error&) {
  throw;
}
catch(...) {
  WX_THROW(Wx_general_error());
};

//-----------------------------------------------------------------------
// These codes count the number of running threads created by Wx_Thread
//-----------------------------------------------------------------------
size_t Wx_Thread::_tt_thrds=0;  // number of total running threads WxThread created 
WxMutex Wx_Thread::_tt_mtx;     // mutex for _tt_thrds

// not-cancelable
void Wx_Thread::inc_thread_count(void) throw(std::exception)
try {
 WxRet r;
 if((r=_tt_mtx.lock())!=OK) {
   WX_THROW(Wx_general_error());
 }
 ++_tt_thrds;
 if((r=_tt_mtx.unlock())!=OK) {
   WX_THROW(Wx_general_error());
 }
 return;
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
  WX_THROW(Wx_general_error());
};

// not-cancelable
void Wx_Thread::dec_thread_count(void) throw(std::exception)
try {
 WxRet r;
 if((r=_tt_mtx.lock())!=OK) {
   WX_THROW(Wx_general_error());
 }
 --_tt_thrds;
 if((r=_tt_mtx.unlock())!=OK) {
   WX_THROW(Wx_general_error());
 }
 return;
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
  WX_THROW(Wx_general_error());
};

//------------------------------------------------------------------------------
bool Wx_Thread::set_cancelable(bool en) throw(std::exception)
try {
 int oldstate;
 int v=::pthread_setcancelstate( en? PTHREAD_CANCEL_ENABLE: PTHREAD_CANCEL_DISABLE
                          , &oldstate);
 if(v!=0) {
   WX_THROW( Wx_bad_errno(v) ); // man. says only parameter error. which should not happen.
 }
 return (oldstate==PTHREAD_CANCEL_ENABLE);
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
  WX_THROW(Wx_general_error());
};

//--------------------------------------------------------------------
static pthread_t main_thrd_id=::pthread_self(); // not used yet

::pthread_once_t Wx_Thread::_once_var=PTHREAD_ONCE_INIT;

void Wx_Thread::_once_func(void) throw()
{
 // intended for feature test code, nothing yet.
};

//
// [Spc] Wx_Thread relies on pthread. Default thread attributes are:
//
//   detatchstate= PTHREAD_CREATE_DETATCHED
//   schedpolicy= SCHED_OTHER (regular, non-realtime)
//   schedparam= 0
//   inheritsched= PTHREAD_EXPLICIT_SCHED
//   scope= PTHREAD_SCOPE_SYSTEM
//
Wx_Thread::Wx_Thread() throw(std::exception)
try : _thrd_stat_cond(), _thrd_stat(Ready), _thrd_id(WxThreadID::Default_ThreadID), _exit_obj(), _in_destroy(false), _mtx()
{
 WxLock aa(_mtx);
 int v;

  if((v=::pthread_once(&_once_var, _once_func))!=0) {
     WX_THROW( Wx_bad_errno(v) );             // always succeed (man)
  }

  if((v=::pthread_attr_init(&_thrd_attr))!=0) {
    //
    // manu says only ENOMEM, linux man just says errno.
    // 
    if(v==ENOMEM) {
      WX_THROW( Wx_bad_alloc() );
    }
    WX_THROW( Wx_bad_errno(v) );
  }

  if((v=::pthread_attr_setdetachstate(&_thrd_attr,PTHREAD_CREATE_DETACHED))!=0) {
    // man. indicates only EINVAL
    WX_THROW( Wx_bad_errno(v) );
  }
}
catch(const WxLock::Fault&) {
 WX_THROW(Wx_general_error());
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

Wx_Thread::~Wx_Thread() throw(std::exception)
try {
  WxLock aa(_mtx);

  //
  // Destructors should not be cancelable for the least reason that resources 
  // had been allocated. Cancelation cannot be safe and correct.
  //
  Wx__NoCancel noc;

  if(_in_destroy==true) {
    WX_THROW(Wx_general_error());    // destroy twice
  }
  _in_destroy=true;

  int v;

  //
  // Cancel the running thread if it is running, and wait 
  // until the thread terminated.
  //
  if(_thrd_stat==Running) {
    //
    // Cancel and wait unitil _thrd_stat==Terminated
    //
    // Note: Loop forever if the threead is not cancellable.
    //
    while((v=::pthread_cancel(_thrd_id))!=0) {
      // ref. note5
      if(v==EAGAIN) {
        Wx__Base::yield();
        continue;
      }
      WX_THROW( Wx_bad_errno(v) );    // ESRCH
    }
    while(_thrd_stat!=Terminated) {   // ref note4
      if(_thrd_stat_cond.wait(aa)!=WXM_EINTR) {
        WX_THROW(Wx_general_error()); // WXM_EFAULT
      }
    }
   } else if(_thrd_stat==Ready) {
    //
    // Thread tmain() have never been run.
    //
  } else {
    // tmain() already in Terminated state.
  }
 
  //
  // Cleanup members that need cleanup
  //
  if((v=::pthread_attr_destroy(&_thrd_attr))!=0) {
    // manu says only EINVAL could happen
    WX_THROW( Wx_bad_errno(v) );
  }
  return;
}
catch(const WxLock::Fault&) {
 WX_THROW(Wx_general_error());
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

bool Wx_Thread::is_default(void) const throw(std::exception)
try {
 WxLock aa(_mtx);
 //
 // _thrd_stat==Ready ensures the default state, including
 //   
 //   1. _thrd_id==WxThreadID::Default_ThreadID
 //   2. _exit_obj.is_default()
 //   3. _in_destroy=false
 //   4. _thrd_attr is default
 //
 return(_thrd_stat==Ready);
}
catch(const WxLock::Fault&) {
 WX_THROW(Wx_general_error());
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet Wx_Thread::reset(void) throw(std::exception)
try {
  WxLock aa(_mtx);

  if(_in_destroy==true) {
    //
    // Object is being destroyed in the unlocking interval. ref. note4
    //
    if(_thrd_stat==Ready) {
      WX_ABORT();            // assertion failed
    }
    if(::pthread_equal(_thrd_id,::pthread_self())) {
      // thread can not reset itself
      WX_THROW( Wx_general_error() );
    }
    while(_thrd_stat!=Terminated) {
      if(_thrd_stat_cond.wait(aa)!=WXM_EINTR) {
        WX_THROW(Wx_general_error()); // WXM_EFAULT
      }
    }
    return(OK);
  }
  int v;

  //
  // If the thread is running, cancel it.
  //
  if(_thrd_stat==Running) {

    if(::pthread_equal(_thrd_id,::pthread_self())) {
      // thread can not reset itself
      WX_THROW( Wx_general_error() );
    }
    while((v=::pthread_cancel(_thrd_id))!=0) {
      // ref. note5
      if(v==EAGAIN) {
        Wx__Base::yield();
        continue;
      }
      WX_THROW( Wx_bad_errno(v) );
    }
    while(_thrd_stat!=Terminated) {
      // cancel_point
      if(_thrd_stat_cond.wait(aa)!=WXM_EINTR) {
        WX_THROW(Wx_general_error());   // WXM_EFAULT
      }
    }
  } else {
    // Ready or Terminated
  }
 
  //
  // Reset thread attr object
  //
  if((v=::pthread_attr_destroy(&_thrd_attr))!=0) {
    // manu says only EINVAL could happen
    // but ignore the error to continue as constructing a default object.
    // FALLTHROUGH
  }
  if((v=::pthread_attr_init(&_thrd_attr))!=0) {
    //
    // manu says only ENOMEM, linux man just says errno.
    // 
    ::pthread_attr_destroy(&_thrd_attr); // make this member invalid.
    if(v==ENOMEM) {
      WX_THROW( Wx_bad_alloc() );
    }
    WX_THROW( Wx_bad_errno(v) );
  }
  if((v=::pthread_attr_setdetachstate(&_thrd_attr,PTHREAD_CREATE_DETACHED))!=0) {
    // man. indicates only EINVAL
    ::pthread_attr_destroy(&_thrd_attr);   // make this member invalid
    WX_THROW( Wx_bad_errno(v) );
  }

  _thrd_stat=Ready;
  _thrd_id=WxThreadID::Default_ThreadID;
  _exit_obj.reset();
  _thrd_stat_cond.broadcast();
  return(OK);
}
catch(const WxLock::Fault&) {
 WX_THROW(Wx_general_error());
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

Wx_Thread::ThreadState Wx_Thread::thread_state(void) const throw (std::exception)
try {
 WxMutex::lock(&_mtx);
 ThreadState retv=_thrd_stat;
 WxMutex::unlock(&_mtx);
 return(retv);
}
catch(const WxMutex::Fault&) {
  WX_THROW(Wx_general_error());
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

//
// [Syn] That the thread tmain() exit/cancel invokes this cleanup function
//
void Wx_Thread::_cleanup_func(Wx_Thread * obj_ptr) throw()
try {
  WxLock aa(obj_ptr->_mtx);

  if(obj_ptr->_thrd_id!=::pthread_self()) {  // assertion check
    WX_ABORT();
  }

  //
  // tmain_leave() is called with _mtx locked and cancelability disabled.
  // No need to recover, as going to end.
  //
  if(::pthread_setcancelstate(PTHREAD_CANCEL_DISABLE,NULL)!=0) {
    WX_ABORT();  // see note2, the entire program may suffer if got this wrong.
  }

  //
  // Set the thread state to Terminated and boradcast, ref. note3
  //  (destructor and reset() may wait on this)
  //
  obj_ptr->_thrd_stat=Terminated;
  obj_ptr->_thrd_stat_cond.broadcast();
 
  //
  // tmain_leave() is called with _mtx locked and cancelability disabled.
  // No need to recover, as going to end.
  //
  obj_ptr->tmain_leave();                 
 
  //
  // Decrease the number of total running threads
  //
  dec_thread_count();

  return;
}
catch(const WxLock::Fault&) {
  WX_ABORT(); // so we can trace the line
}
catch(const Wx_general_error&) {
  WX_ABORT(); // so we can trace the line
}
catch(const Wx_bad_errno&) {
  WX_ABORT(); // so we can trace the line
}
catch(const Wx_bad_alloc&) {
  WX_ABORT(); // so we can trace the line
}
catch(const std::bad_alloc&) {
  WX_ABORT(); // so we can trace the line
}
catch(const std::exception&) {
  WX_ABORT(); // so we can trace the line
}
catch(...) {
  WX_ABORT(); // so we can trace the line
};

void * Wx_Thread::_begin_func(Wx_Thread *obj_ptr) throw()
try {
  //
  // tmain_enter() is called with _mtx locked and cancelability disabled.
  //
  {
   // By not using WxLock, tmain_leave() is sure called iff tmain_enter() called.  
   Wx__NoCancel noc;
   WxRet r=obj_ptr->_mtx.lock();
   if(r!=OK) {
     obj_ptr->_thrd_stat=Terminated;
     obj_ptr->_exit_obj=r;
     obj_ptr->_thrd_stat_cond.broadcast();
     WX_HERE(obj_ptr->_exit_obj); // set it however, chance is on my side
     return(NULL);     // this failure can be handled, so return.
   }
   if(obj_ptr->_thrd_id!=::pthread_self()) {  // assertion check
     WX_ABORT();
   }
   obj_ptr->tmain_enter();
   r=obj_ptr->_mtx.unlock();
   if(r!=OK) {
     obj_ptr->_thrd_stat=Terminated;
     obj_ptr->_exit_obj=r;
     obj_ptr->_thrd_stat_cond.broadcast();
     WX_HERE(obj_ptr->_exit_obj);
     obj_ptr->tmain_leave();
     return(NULL);     // this failure can be handled, so return.
   }
  }
 
  //
  // Setup cleanup function and call tmain()
  //
  struct ::_pthread_cleanup_buffer pcpbuf;
  ::_pthread_cleanup_push(&pcpbuf,
                      reinterpret_cast<void(*)(void*)>(_cleanup_func), obj_ptr);
  WxRet r=obj_ptr->tmain();
  ::_pthread_cleanup_pop(&pcpbuf,false);  

  //
  // tmain() return gets here. ref. _cleanup_func(...)
  //
  // The following codes are the same as _cleanup_func except 
  // the tmain() exit code setup. (cannot call _cleanup_func, for the mutex sake)
  //
  WxLock aa(obj_ptr->_mtx);
 
  obj_ptr->_exit_obj=r;        // set tmain() returned WxRet
  
  //
  // tmain_leave() is called with _mtx locked and cancelability disabled.
  // No need to recover, as going to end.
  //
  if(::pthread_setcancelstate(PTHREAD_CANCEL_DISABLE,NULL)!=0) {
    // ref. note2
    WX_ABORT();
  }

  //
  // Set the thread state to Terminated and boradcast, ref. note3
  //  (destructor and reset() may wait on this)
  //
  obj_ptr->_thrd_stat=Terminated;
  obj_ptr->_thrd_stat_cond.broadcast();

  //
  // tmain_leave() is called with _mtx locked and cancelability disabled.
  // No need to recover, as going to end.
  //
  obj_ptr->tmain_leave();

  //
  // Decrease the number of total running threads
  //
  dec_thread_count();

  return(NULL);  // return value not used
}
catch(const WxLock::Fault&) {
  WX_ABORT(); // so we can trace the line
}
catch(const Wx_general_error&) {
  WX_ABORT(); // so we can trace the line
}
catch(const Wx_bad_errno&) {
  WX_ABORT(); // so we can trace the line
}
catch(const Wx_bad_alloc&) {
  WX_ABORT(); // so we can trace the line
}
catch(const std::bad_alloc&) {
  WX_ABORT(); // so we can trace the line
}
catch(const std::exception&) {
  WX_ABORT(); // so we can trace the line
}
catch(...) {
  WX_ABORT(); // so we can trace the line
};

WxRet Wx_Thread::begin(void) throw(std::exception)
try {
  WxLock aa(_mtx);
  int v;

  if(_in_destroy==true) {
    //
    // Object is being destroyed in the unlocking interval. ref note4
    //
    if(_thrd_stat==Ready) {
      WX_ABORT();            // assertion failed
    }
    WX_RETURN(WXM_EBADF);
  }
  if(_thrd_stat!=Ready) {
    WX_RETURN(WXM_EBADF);
  }

  //
  // ANSI C allows any pointer converts to void* and back (not transitive)
  //
  // Note: Thread is created with cancellation enabled and detached.
  //
  if((v=::pthread_create(&_thrd_id,&_thrd_attr, 
 		 reinterpret_cast<void *(*)(void *)>(&_begin_func),
 		 static_cast<void *>(this)))!=0) {
     if(v==EAGAIN) {
       WX_RETURN(WXM_EAGAIN);
     }
     WX_THROW( Wx_bad_errno(v) );
  }

  // 
  // Check that _thrd_id is not the value identical to the default used
  // Refer to definition of WxThreadID::Default_ThreadID.
  //
  if(_thrd_id==WxThreadID::Default_ThreadID) {
    while(::pthread_cancel(_thrd_id)!=0) {
      // ref. note5
      if(v==EAGAIN) {
        Wx__Base::yield();
        continue;
      }
      WX_THROW( Wx_bad_errno(v) );    // ESRCH
    }
    WX_THROW(Wx_general_error());
  }

  try {
    inc_thread_count();

    //
    // Pre-set the thread exit code as by cancellation.
    // Other types of tmain() termination will overwrite this.
    //
    _exit_obj=WXM_THRDCAN;
    WX_HERE(_exit_obj);
    _thrd_stat=Running;
    _thrd_stat_cond.broadcast();
  }
  catch(...) {
    while((v=::pthread_cancel(_thrd_id))!=0) {
      // ref. note5
      if(v==EAGAIN) {
        Wx__Base::yield();
        continue;
      }
      WX_THROW( Wx_bad_errno(v) );    // ESRCH
    }
    throw;
  };
  return(OK);
}
catch(const WxLock::Fault&) {
 WX_THROW(Wx_general_error());
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet Wx_Thread::cancel(void) throw(std::exception)
try {
  WxLock aa(_mtx);

  if(_in_destroy==true) {
    //
    // Object is being destroyed in the unlocking interval.
    // This function just meant to send cancel signal.
    //
    if(_thrd_stat==Ready) {
      WX_ABORT();            // assertion failed
    }
    if(::pthread_equal(_thrd_id,::pthread_self())) {
      WX_RETURN(WXM_EPERM);  // thread can not cancel itself
    }
    return(OK); 
  }
  if(_thrd_stat!=Running) {
    WX_RETURN(WXM_ENOENT);
  }
  if(::pthread_equal(_thrd_id,::pthread_self())) {
    WX_RETURN(WXM_EPERM); // thread cannot cancel itself (use exit(...), or return)
  }
  int v;
  while((v=::pthread_cancel(_thrd_id))!=0) {
    // ref. note5
    if(v==EAGAIN) {
      Wx__Base::yield();
      continue;
    }
    WX_THROW( Wx_bad_errno(v) );    // ESRCH
  }
  _thrd_stat_cond.broadcast();
  return(OK);
}
catch(const WxLock::Fault&) {
  WX_THROW(Wx_general_error());
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet Wx_Thread::exit_code(void) const throw(std::exception)
try {
 WxLock aa(_mtx);
 return(_exit_obj);
}
catch(const WxLock::Fault&) {
  WX_THROW(Wx_general_error());
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet Wx_Thread::wait_stopped(void) const throw(std::exception)
try {
  WxLock aa(_mtx);

  if(_in_destroy==true) {
    //
    // Object is being destroyed in the unlocking interval.
    //
    if(_thrd_stat==Ready) {
      WX_ABORT();            // assertion failed
    }
    if(::pthread_equal(_thrd_id,::pthread_self())) {
      WX_RETURN(WXM_EPERM);  // thread cannot wait_stopped itself
    }
    while(_thrd_stat!=Terminated) {
      if(_thrd_stat_cond.wait(aa)!=WXM_EINTR) {
        WX_THROW(Wx_general_error()); // WXM_EFAULT
      }
    }
    return(OK);
  }
  if(_thrd_stat==Ready) {
    WX_RETURN(WXM_ENOENT);
  }
  if(_thrd_stat==Running) {
    if(::pthread_equal(_thrd_id,::pthread_self())) {
      WX_RETURN(WXM_EPERM);  // thread cannot wait_stopped itself
    }
  }
  while(_thrd_stat!=Terminated) {
    // cancel_point
    if(_thrd_stat_cond.wait(aa)!=WXM_EINTR) {
      WX_THROW(Wx_general_error());
    }
    if(_in_destroy==true) {
      //
      // Object is being destroyed in the unlocking interval.
      // just check the assertion
      //
      if(_thrd_stat==Ready) {
        WX_ABORT();            // assertion failed
      }
    }
  }
  return(OK);
}
catch(const WxLock::Fault&) {
  WX_THROW(Wx_general_error());
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet Wx_Thread::wait_nready(void) const throw(std::exception)
try {
  WxLock aa(_mtx);

  if(::pthread_equal(_thrd_id,::pthread_self())) {
    WX_RETURN(WXM_EPERM);  // thread cannot wait_nready itself
  }

  while(_thrd_stat==Ready) {
    // cancel_point
    if(_thrd_stat_cond.wait(aa)!=WXM_EINTR) {
      WX_THROW(Wx_general_error());
    }
  }
  return(OK);
}
catch(const WxLock::Fault&) {
  WX_THROW(Wx_general_error());
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet Wx_Thread::yield(void) throw(std::exception)
try {
 const int v=Wx__Base::yield();
 if(v==0) {
   return(OK);
 }
 WX_RETURN(WxErrMsg(v));
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

size_t Wx_Thread::total_threads(void) throw(std::exception)
try {
 if(_tt_mtx.lock()!=OK) {
   WX_THROW(Wx_general_error());
 }
 const size_t num=_tt_thrds;
 if(_tt_mtx.unlock()!=OK) {
   WX_THROW(Wx_general_error());
 }
 return(num);
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

void Wx_Thread::exit(WxRet exit_ret) throw(std::exception)
try {
  //
  // Note: Gap exists between unlock _mtx and pthread_exit.
  //       Have to use WxLock.
  //
  WxLock aa(_mtx);
 
  if(_thrd_stat!=Running) {
    // assertion: thread tmain() always sees Running state   
    WX_ABORT();
  }
  _exit_obj=exit_ret;

  //
  // pthread_exit() does not unwinding the stack and the parameter is not used.
  // 
  ::pthread_exit(0);        
}
catch(const WxLock::Fault&) {
  WX_THROW(Wx_general_error());
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

void Wx_Thread::abort(void) const throw()
try {
  Wx__Base::abort();
}
catch(const WxLock::Fault&) {
  WX_ABORT();
}
catch(const Wx_general_error&) {
  WX_ABORT();
}
catch(const Wx_bad_errno&) {
  WX_ABORT();
}
catch(const Wx_bad_alloc&) {
  WX_ABORT();
}
catch(const std::bad_alloc&) {
  WX_ABORT();
}
catch(...) {
  WX_ABORT();
};

//---------------------------------------------------------------------------------

WxThread::WxThread() throw(std::exception)
try : Wx_Thread() {
}
catch(...) {
  throw;
};

WxThread::~WxThread() throw(std::exception)
try {
}
catch(...) {
  throw;
};
//---------------------------------------------------------------------------------

/*
 note2: WxThread assert the function call never fail:
        :::pthread_setcancelstate( PTHREAD_CANCEL_ENABLE: PTHREAD_CANCEL_DISABLE
                         , &oldstate)
        it might be better let this function always succeed.
 note3: Only while the thread terminates (by return/exit/cancel) moves _thrd_stat 
        from Running to Terminated.
        Codes are in  _cleanup_func(...) if tmain() canceled.
                    _begin_func(...) if tmain() returned
 note4: Object is being destroyed in the unlocking interval. Function called might
        return OK or pretend OK, as next call will get mutex exception.
        _in_destroy can be seen true only while the destructor canceled the running thread.
 note5:
       ::pthread_cancel found to return EAGAIN (on RH Linux9.0)
       while man. indicates only fail on ESRCH.
*/
