/* Copyright is licensed under GNU LGPL.                (I.J. Wang, 2003)
*/
#ifndef WXTERMINAL_H__
#define WXTERMINAL_H__
#define WXTERMINAL_VERSION 8
#define WXTERMIOS_VERSION 8

#include "wxchrfile.h"
#include <termios.h>

/*
   WxTermios defines the control attribute of WxTerminal (struct termios)

   Note: Use of this class:
         1. Read the terminal attributes before modifing them.
         2. Modifies only those bits required
         3. Setting the terminal attributes should follow a read back of the terminal
            attribute to check if the settings was accepted. (Refer to setattr())

  Note: To be safe and portable, use drain() to wait for all output data transmitted
        before writting the attributes to the terminal.

        Refer to POSIX Programmer's Guide,Donald Lewine,p161

  [Avoided C Use] struct termios
                  cfsetispeed, cfsetospeed, cfgetispeed, cfgetospeed
*/
class WxTermios {
  public:
    static const char class_name[];
    WX_DECLARE_FAULT;

    WxTermios() throw()
             { std::memset(&_tio,0,sizeof(_tio)); };

    WxTermios(const WxTermios& tio) throw(std::exception);

    ~WxTermios() throw()
              {};

    bool is_default(void) const throw(std::exception);

    void reset(void) throw()
             { std::memset(&_tio,0,sizeof(_tio)); };

    void assign(const WxTermios& tio) throw(std::exception);

    const WxTermios& operator =(const WxTermios &tio) throw(std::exception)
              { assign(tio); return(*this); };

    speed_t getispeed(void) const throw(std::exception);

    speed_t getospeed(void) const throw(std::exception);

    WxRet setispeed(speed_t speed) throw(std::exception);

    WxRet setospeed(speed_t speed) throw(std::exception);
    
    inline tcflag_t iflag(void) const throw()
              { return _tio.c_iflag; };
    inline void iflag(tcflag_t v) throw()
              { _tio.c_iflag=v; };
    inline void iflag_or(tcflag_t v) throw()
              { _tio.c_iflag|=v; };
    inline void iflag_and(tcflag_t v) throw()
              { _tio.c_iflag&=v; };

    inline tcflag_t oflag(void) const throw()
              { return _tio.c_oflag; };
    inline void oflag(tcflag_t v) throw()
              { _tio.c_oflag=v; };
    inline void oflag_or(tcflag_t v) throw()
              { _tio.c_oflag|=v; };
    inline void oflag_and(tcflag_t v) throw()
              { _tio.c_oflag&=v; };

    inline tcflag_t cflag(void) const throw()
              { return _tio.c_cflag; };
    inline void cflag(tcflag_t v) throw()
              { _tio.c_cflag=v; };
    inline void cflag_or(tcflag_t v) throw()
              { _tio.c_cflag|=v; };
    inline void cflag_and(tcflag_t v) throw()
              { _tio.c_cflag&=v; };

    inline tcflag_t lflag(void) const throw()
              { return _tio.c_lflag; };
    inline void lflag(tcflag_t v) throw()
              { _tio.c_lflag=v; };
    inline void lflag_or(tcflag_t v) throw()
              { _tio.c_lflag|=v; };
    inline void lflag_and(tcflag_t v) throw()
              { _tio.c_lflag&=v; };

    inline cc_t cc_VSTOP(void) const throw()
              { return _tio.c_cc[VSTOP]; };
    inline void cc_VSTOP(cc_t v) throw()
              { _tio.c_cc[VSTOP]=v; };

    inline cc_t cc_VSTART(void) const throw()
              { return _tio.c_cc[VSTART]; };
    inline void cc_VSTART(cc_t v) throw()
              { _tio.c_cc[VSTART]=v; };

    inline cc_t cc_VSUSP(void) const throw()
              { return _tio.c_cc[VSUSP]; };
    inline void cc_VSUSP(cc_t v) throw()
              { _tio.c_cc[VSUSP]=v; };

    inline cc_t cc_VQUIT(void) const throw()
              { return _tio.c_cc[VQUIT]; };
    inline void cc_VQUIT(cc_t v) throw()
              { _tio.c_cc[VQUIT]=v; };

    inline cc_t cc_VKILL(void) const throw()
              { return _tio.c_cc[VKILL]; };
    inline void cc_VKILL(cc_t v) throw()
              { _tio.c_cc[VKILL]=v; };

    inline cc_t cc_VERASE(void) const throw()
              { return _tio.c_cc[VERASE]; };
    inline void cc_VERASE(cc_t v) throw()
              { _tio.c_cc[VERASE]=v; };

    inline cc_t cc_VEOL(void) const throw()
              { return _tio.c_cc[VEOL]; };
    inline void cc_VEOL(cc_t v) throw()
              { _tio.c_cc[VEOL]=v; };

    inline cc_t cc_VEOF(void) const throw()
              { return _tio.c_cc[VEOF]; };
    inline void cc_VEOF(cc_t v) throw()
              { _tio.c_cc[VEOF]=v; };

    inline cc_t cc_VINTR(void) const throw()
              { return _tio.c_cc[VINTR]; };
    inline void cc_VINTR(cc_t v) throw()
              { _tio.c_cc[VINTR]=v; };

    inline cc_t cc_VTIME(void) const throw()
              { return _tio.c_cc[VTIME]; };
    inline void cc_VTIME(cc_t v) throw()
              { _tio.c_cc[VTIME]=v; };

    inline cc_t cc_VMIN(void) const throw()
              { return _tio.c_cc[VMIN]; };
    inline void cc_VMIN(cc_t v) throw()
              { _tio.c_cc[VMIN]=v; };

    bool operator ==(const WxTermios& rhs) const throw()
              { return std::memcmp(&_tio,&rhs._tio,sizeof(_tio))==0; };
    bool operator !=(const WxTermios& rhs) const throw()
              { return std::memcmp(&_tio,&rhs._tio,sizeof(_tio))!=0; };
  private:
    struct termios _tio;

    friend class WxTerminal;  // WxTerminal need to access _tio directly
};

/*
  Note: Some of the error messages are based on Linux Programmer's Manual.

  [Avoided C Use] tcsendbreak, tcdrain, tcflow, tcflush
                  tcgetattr, tcsetattr
*/
class WxTerminal : public WxChrFile {
  public:
    static const char class_name[];
    WX_DECLARE_FAULT;

    WxTerminal() throw()
              : WxChrFile() {};

    WxTerminal(const WxTerminal &src) throw(std::exception,WxRet);

    virtual ~WxTerminal() throw(std::exception);

    // [Derived]
    //bool is_default(void) const throw() 

    // [Derived]
    // WxFileHandle fh(void) const throw()

    // [Derived]
    // WxRet stat(WxFileStat& filestat) const throw(std::exception)

    // [Derived]
    // virtual WxRet reset(void) throw(std::exception)

    // [Derived]
    //WxRet virtual read(void *buf, size_t count, size_t *n_read) throw(std::exception)

    // [Derived]
    //WxRet virtual max_read(void* buf,size_t count,size_t* n_read) throw(std::exception)

    // [Derived]
    //WxRet virtual write(const void *buf, size_t count, size_t *n_written) throw(std::exception)

    virtual WxRet drain(void) throw(std::exception);

    // [Derived]
    //virtual WxRet open(const WxPathName& pathname,int f) throw(std::exception);

    virtual WxRet assignf(WxFileHandle fh) throw(std::exception);

    virtual WxRet assignf(const WxTerminal& src) throw(std::exception);

    virtual WxRet sendbreak(int duration) throw(std::exception);

    virtual WxRet flush(int qsel) throw(std::exception);

    virtual WxRet flow(int action) throw(std::exception);
    
    WxRet getattr(WxTermios& tio) const throw(std::exception);

    virtual WxRet setattr(const WxTermios& tio,int option) throw(std::exception);

    WxRet devname(WxPathName& pathname) const throw(std::exception);

    // [Derived]
    // WxRet wx_set_fh(WxFileHandle fh) throw(std::exception)
  protected:
    //
    // [Syn] Construct object from the open fd
    //
    // Note: This is not a direct copy of fd but a duplicate by dup()
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    // 
    // [Exception] Fault
    //       WXM_EINVAL fd is not an open fd
    //       WXM_EBADF  fd is not valid for the operation
    //       WXM_EMFILE too many open fd
    //
    // Note: errmsg is converted from the errno ::dup returned
    //
    // [Refer] ::dup(int)
    //
    WxTerminal(WxFileHandle fh) throw(std::exception,WxRet)
              : WxChrFile(fh) {};

    // The following are public derived functions, and disabled and hidden.
    virtual WxRet assignf(const WxChrFile& src) throw(std::exception)
              { WX_RETURN(WXM_VFDIS); };

    // [Derived]
    // WxRet wx_open(const WxPathName& pathname,int f) throw(std::exception)
    // WxRet wx_open(const WxPathName& pathname, int f, mode_t m) throw(std::exception)
    // WxRet wx_assign(const WxByteFlow& byteflow) throw(std::exception)
    // WxRet wx_fcntl(int cmd,int *ret_f) const throw(std::exception)

    // WxRet wx_fcntl(int cmd, long arg, int *ret_f) const throw(std::exception)
    //          { return WxByteFlow::fcntl(cmd,arg,ret_f); }; 

    // WxRet wx_fcntl(int cmd, struct flock *lock,int *ret_f) const throw(std::exception)
    //          { return WxByteFlow::fcntl(cmd,lock,ret_f); };
  private:

    // Hide
};

#endif
