/* Copyright is licensed under GNU LGPL.                (I.J. Wang, 2003)
*/

#include "wxsockfile.h"
#include <cstring>
#include <netinet/in.h> 

/*
 note1: The codes assume that the open socket on PF_??? access address of
        AF_??? exactly.
*/
const char WxSockFile::class_name[]="WxSockFile";

WxRet WxSockFile::wx_set_fh(WxFileHandle fh, int family) throw(std::exception)
try {
 const WxRet r=WxByteFlow::wx_set_fh(fh);
 if(r!=OK) {
   WX_RETURN(r);
 }
 _family=family;
 return(OK);
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxSockFile::WxSockFile(const WxSockFile& src) throw(std::exception,WxRet)
try : WxByteFlow(src), _family(src._family) {
}
catch(const WxByteFlow::Fault& e) {
  WX_THROW( Fault(e) );
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxSockFile::~WxSockFile() throw(std::exception)
try {
}
catch(const WxByteFlow::Fault& e) {
  WX_THROW( Fault(e) );
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet WxSockFile::drain(void) throw(std::exception)
try {
  WX_RETURN( WxByteFlow::drain() );
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet WxSockFile::open(int domain, int type, int proto) throw(std::exception)
try {
 if(WxByteFlow::is_default()==false) {
   WX_RETURN(WXM_NDEFAULT);
 }
 int s=::socket(domain,type,proto);
 if(s<0) {
   WX_RETURN(WxErrMsg(errno));
 }
 try {
   WxFileHandle fh(s);
   if(wx_set_fh(fh,domain)!=OK) {
     WX_THROW(Wx_general_error());
   }
 }
 catch(...) {
   ::close(s);
   throw;
 }
 return(OK);
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet WxSockFile::assignf(WxFileHandle fh) throw(std::exception)
try {
  // stat fh
  if(fh.is_default()==false) {
    WxFileStat stt;
    const WxRet r=fh.stat(stt);
    if(r!=OK) {
      WX_RETURN(r);
    } 
    if(stt.is_sock()==false) {
      WX_RETURN(WXM_EBADF);
    }
  }
  WX_RETURN( WxByteFlow::wx_assignf(fh) );
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet WxSockFile::assignf(const WxSockFile& src) throw(std::exception)
try {
 const WxRet r=WxByteFlow::wx_assignf(src);
 if(r!=OK) {
   WX_RETURN(r);
 }
 _family=src._family;
 return(OK);
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet WxSockFile::shutdown(int how) throw(std::exception)
try {
 if(::shutdown(WxByteFlow::fh().fd(),how)==0) {
   return(OK);
 }
 WX_RETURN(WxErrMsg(errno));
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet WxSockFile::bind(const Wx_SockAddr& addr) throw(std::exception)
try {
 if(WxByteFlow::is_default()) {
   WX_RETURN(WXM_EBADF);       // prevent WXM_EAFNOSUPPORT reported first
 }
 if(addr.const_saddr_ptr()->sa_family!=_family) {
   WX_RETURN(WXM_EAFNOSUPPORT);
 }
 if(::bind(WxByteFlow::fh().fd(),addr.const_saddr_ptr(),
                           *addr.const_slen_ptr())!=0) {
   WX_RETURN(WxErrMsg(errno));
 }
 return(OK);
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet WxSockFile::listen(int backlog) throw(std::exception)
try {
 if(::listen(WxByteFlow::fh().fd(),backlog)==0) {
   return(OK);
 }
 WX_RETURN(WxErrMsg(errno));
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet WxSockFile::accept(WxSockFile &skt, Wx_SockAddr& addr) throw(std::exception)
try {
 if(WxByteFlow::is_default()) {
   WX_RETURN(WXM_EBADF);       // prevent WXM_EAFNOSUPPORT reported first
 }
 if(skt.is_default()==false) {
   WX_RETURN(WXM_NDEFAULT);
 }
 if(addr.const_saddr_ptr()->sa_family!=_family) {
   WX_RETURN(WXM_EAFNOSUPPORT);
 }
 //
 // Direct access of addr. The ::accept is thought to return the exact address
 // family of the socket protocol family (value). see note1.
 //
 *addr.slen_ptr()=addr.saddr_size();
 const int s=::accept(WxByteFlow::fh().fd(),addr.saddr_ptr(), addr.slen_ptr());
 if(s<0) {
   // make sure that addr is default whild failed
   *addr.slen_ptr()=Wx_SockAddr::DefaultSockLen;
   WX_RETURN(WxErrMsg(errno));
 }

 try {
   if(addr.const_saddr_ptr()->sa_family!=_family) {
     WX_THROW(Wx_general_error());  // assertion failure (note1)
   }
   WxFileHandle fh(s);
   if(skt.wx_set_fh(fh,_family)!=OK) {
     WX_THROW(Wx_general_error());
   }
 }
 catch(...) {
   // make sure that addr is default whild failed
   *addr.slen_ptr()=Wx_SockAddr::DefaultSockLen;
   ::close(s);
   throw;
 }
 return(OK);
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet WxSockFile::connect(const Wx_SockAddr& peer_addr) throw(std::exception)
try {
 if(WxByteFlow::is_default()) {
   WX_RETURN(WXM_EBADF);       // prevent WXM_EAFNOSUPPORT reported first
 }
 if(peer_addr.const_saddr_ptr()->sa_family!=_family) {
   WX_RETURN(WXM_EAFNOSUPPORT);
 }
 if(::connect(WxByteFlow::fh().fd(),peer_addr.const_saddr_ptr(), 
                              *peer_addr.const_slen_ptr())!=0) {
   WX_RETURN(WxErrMsg(errno));
 }
 return(OK);
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet WxSockFile::get_addr(Wx_SockAddr& addr) const throw(std::exception)
try {
 if(WxByteFlow::is_default()) {
   WX_RETURN(WXM_EBADF);       // prevent WXM_EAFNOSUPPORT reported first
 }
 if(addr.const_saddr_ptr()->sa_family!=_family) {
   WX_RETURN(WXM_EAFNOSUPPORT);
 }
 if(::getsockname(WxByteFlow::fh().fd(),addr.saddr_ptr(),
                                   addr.slen_ptr())!=0) {
   WX_RETURN(WxErrMsg(errno));
 }
 if(addr.const_saddr_ptr()->sa_family!=_family) {
   WX_THROW(Wx_general_error());  // assertion failure (note1)
 }
 return(OK);
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet WxSockFile::get_peer_addr(Wx_SockAddr& addr) const throw(std::exception)
try {
 if(WxByteFlow::is_default()) {
   WX_RETURN(WXM_EBADF);       // prevent WXM_EAFNOSUPPORT reported first
 }
 if(addr.const_saddr_ptr()->sa_family!=_family) {
   WX_RETURN(WXM_EAFNOSUPPORT);
 }
 if(::getpeername(WxByteFlow::fh().fd(),addr.saddr_ptr(),
                                   addr.slen_ptr())!=0) {
   WX_RETURN(WxErrMsg(errno));
 }
 if(addr.const_saddr_ptr()->sa_family!=_family) {
   WX_THROW(Wx_general_error());  // assertion failure (note1)
 }
 return(OK);
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};
