/* Copyright is licensed under GNU LGPL.                (I.J. Wang, 2003)

   WxSockAddrLocal deals with the local address (AF_LOCAL)

  [Avoided C Use] struct sockaddr/ socklen_t/inet_ntop
                  inet_ntop

  Note: Some of the error messages are based on Linux Programmer's Manual.
*/

#ifndef WXSOCKADDRLOCAL_H__
#define WXSOCKADDRLOCAL_H__
#define WXSOCKADDRLOCAL_VERSION 8

#include "wxret.h"
#include "wxpathname.h"
#include "wx_sockaddr.h"
#include <sys/un.h>

//
// Class for local socket address (family=AF_LOCAL)
//
// Note that object may by modified via the base class
//
class WxSockAddrLocal : public Wx_SockAddr {
  public:
    static const char class_name[];
    WX_DECLARE_FAULT;

    //
    // _slen(DefaultSockLen) defines the default object
    //
    WxSockAddrLocal() throw()
              : _slen(DefaultSockLen) { _saddr.sun_family=AF_LOCAL; };

    WxSockAddrLocal(const WxSockAddrLocal &src) throw(std::exception);
    
    WxSockAddrLocal(const WxPathName& path) throw(std::exception,WxRet);
    
    virtual ~WxSockAddrLocal() throw()
              {};

    bool is_default(void) const throw()
              { return _slen==DefaultSockLen; };

    int family(void) const throw()
              { return AF_LOCAL; };

    void reset(void) throw()
              { _saddr.sun_family=AF_LOCAL; _slen=DefaultSockLen; };

    void assign(const WxSockAddrLocal& src) throw(std::exception);

    WxRet assign(const WxPathName& path) throw(std::exception)
              { return _assign(path); };

    const WxSockAddrLocal& operator=(const WxSockAddrLocal& rhs) throw(std::exception)
              { assign(rhs); return(*this); };

    WxRet get_notation(WxStr& addr) const throw(std::exception);

    bool operator==(const WxSockAddrLocal &rhs) const throw(std::exception);
    bool operator!=(const WxSockAddrLocal &rhs) const throw(std::exception)
              { return !operator==(rhs); };

  private:
    virtual struct sockaddr* saddr_ptr(void) throw()
              { return (struct sockaddr*)(&_saddr); };
    virtual  socklen_t* slen_ptr(void) throw()
              { return &_slen; };
    virtual size_t saddr_size(void) const throw()
              { return sizeof(_saddr); };
    virtual const struct sockaddr* const_saddr_ptr(void) const throw()
              { return (const struct sockaddr*)(&_saddr); };
    virtual const socklen_t* const_slen_ptr(void) const throw()
              { return &_slen; };

    //
    // [Internal] Assign object from the pathname path
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //
    // [Ret] OK    
    //       WXM_ENAMETOOLONG path too long
    //
    WxRet _assign(const WxPathName& path) throw(std::exception);
  private:
    struct sockaddr_un _saddr;
    socklen_t _slen;         // [DefaultSockLen, sizeof(_saddr.sunfamily)...]
};

#endif
