/* Copyright is licensed under GNU LGPL.                (I.J. Wang, 2003)

   WxSockAddrINet deals with the IPv4 address (AF_INET)

  [Avoided C Use] struct sockaddr/ socklen_t/inet_ntop
                  sockaddr_in

  Note: Some of the error messages are based on Linux Programmer's Manual.
*/

#ifndef WXSOCKADDRINET_H__
#define WXSOCKADDRINET_H__
#define WXSOCKADDRINET_VERSION 8

#include "wxret.h"
#include "wxpathname.h"
#include "wx_sockaddr.h"
#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>

//
// Class for IPv4 address (family=AF_INET)
//
class WxSockAddrINet : public Wx_SockAddr {
  public:
    static const char class_name[];
    WX_DECLARE_FAULT;

    WxSockAddrINet() throw()
              : _slen(DefaultSockLen) { _saddr.sin_family=AF_INET; };

    WxSockAddrINet(const WxSockAddrINet &src) throw(std::exception);
    
    WxSockAddrINet(ulong ip, ushort port) throw(std::exception);

    WxSockAddrINet(const WxStr& addr, ushort port) throw(std::exception,WxRet);

    virtual ~WxSockAddrINet() throw() 
              {};

    bool is_default(void) const throw()
              { return _slen==DefaultSockLen; };

    int family(void) const throw()
              { return AF_INET; };

    void reset(void) throw()
              { _saddr.sin_family=AF_INET; _slen=DefaultSockLen; };

    void assign(const WxSockAddrINet &src) throw(std::exception)
              { _assign(src); };

    void assign(ulong ip, ushort port) throw(std::exception)
              { _assign(ip,port); };

    WxRet assign(const WxStr& addr, ushort port) throw(std::exception)
              { return _assign(addr.std_string(),port); };

    const WxSockAddrINet &operator=(const WxSockAddrINet &rhs) throw(std::exception)
              { _assign(rhs); return(*this); };

    WxRet get_notation(WxStr& addr) const throw(std::exception);

    unsigned short port(void) const throw()
              { return _slen==DefaultSockLen?0: ::ntohs(_saddr.sin_port); };

    ulong ip(void) const throw()
              { return _slen==DefaultSockLen? 
                      INADDR_ANY: ::ntohl(_saddr.sin_addr.s_addr); };
 
    bool operator==(const WxSockAddrINet &rhs) const throw(std::exception);
    bool operator!=(const WxSockAddrINet &rhs) const throw(std::exception)
              { return !operator==(rhs); };

  private:
    virtual struct sockaddr* saddr_ptr(void) throw()
              { return (struct sockaddr*)(&_saddr); };
    virtual  socklen_t* slen_ptr(void) throw()
              { return &_slen; };
    virtual size_t saddr_size(void) const throw()
              { return sizeof(_saddr); };
    virtual const struct sockaddr* const_saddr_ptr(void) const throw()
              { return (const struct sockaddr*)(&_saddr); };
    virtual const socklen_t* const_slen_ptr(void) const throw()
              { return &_slen; };

    //
    // [Internal] Assigned object by copy from src
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //
    void _assign(const WxSockAddrINet &src) throw(std::exception);

    // 
    // [Internal] Assign object to IPv4 address (family=AF_INET)
    //
    //        port= IPv4 Port number
    //            = 0, no port (ephemeral port with bind())
    //        ip  = IP (numerical) address.
    //            = INADDR_ANY
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    //
    void _assign(ulong ip, ushort port) throw(std::exception);

    //
    // [Internal] Assign object to IPv4 address (family=AF_INET)
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    //
    // [Ret] OK    
    //       WXM_EADDRNOTAVAIL addr not recognizable
    //       WXM_AFNOSUPPORT   AF not supported
    //
    // Note: current version support dot-decimal notation
    //
    WxRet _assign(const std::string& addr, ushort port) throw(std::exception);

  private:
    struct sockaddr_in _saddr;
    socklen_t _slen;  
};

#endif
