/* Copyright is licensed under GNU LGPL.                (I.J. Wang, 2003)

 This file is part of wxret.h and is included there. 
 Definition code is in wxret.cpp

 [Implement]
      WxErrMsg has exactly one member const char*, utilizing the fact that the
      address of static allocated c-string is unique in a process. The pointer 
      is itself an id.
 */
#ifndef WXR_ERRMSG_H__
#define WXR_ERRMSG_H__
#define WXR_ERRMSG_VERSION 8

#include <cstring>

class WxErrMsg;
//------------------------------------//
//  These are WxLib defined errmsg's  //
//------------------------------------//

//
// Constants map to C-library errno's by prefixing errno names with "WXM_".
//
// Applications should use these constant objects.
//
extern const WxErrMsg OK;  		// Success

extern const WxErrMsg WXM_EPERM;  	// Operation not permitted 
extern const WxErrMsg WXM_ENOENT; 	// No such file or directory 
extern const WxErrMsg WXM_ESRCH;  	// No such process 
extern const WxErrMsg WXM_EINTR;  	// Interrupted system call 
extern const WxErrMsg WXM_EIO;    	// Input/output error 
extern const WxErrMsg WXM_ENXIO;  	// No such device or address 
extern const WxErrMsg WXM_E2BIG;  	// Arg list too long 
extern const WxErrMsg WXM_ENOEXEC;	// Exec format error 
extern const WxErrMsg WXM_EBADF;  	// Bad file number 
extern const WxErrMsg WXM_ECHILD; 	// No child processes 
extern const WxErrMsg WXM_EAGAIN; 	// Try again 
extern const WxErrMsg WXM_ENOMEM; 	// Out of memory 
extern const WxErrMsg WXM_EACCES; 	// Permission denied 
extern const WxErrMsg WXM_EFAULT; 	// Bad address 
extern const WxErrMsg WXM_ENOTBLK;	// Block device required 
extern const WxErrMsg WXM_EBUSY;  	// Device or resource busy 
extern const WxErrMsg WXM_EEXIST; 	// File exists 
extern const WxErrMsg WXM_EXDEV;  	// Cross-device link 
extern const WxErrMsg WXM_ENODEV; 	// No such device 
extern const WxErrMsg WXM_ENOTDIR;	// Not a directory 
extern const WxErrMsg WXM_EISDIR; 	// Is a directory 
extern const WxErrMsg WXM_EINVAL; 	// Invalid argument 
extern const WxErrMsg WXM_ENFILE; 	// File table overflow 
extern const WxErrMsg WXM_EMFILE; 	// Too many open files 
extern const WxErrMsg WXM_ENOTTY; 	// Not a typewriter 
extern const WxErrMsg WXM_ETXTBSY;	// Text file busy 
extern const WxErrMsg WXM_EFBIG;  	// File too large 
extern const WxErrMsg WXM_ENOSPC; 	// No space left on device 
extern const WxErrMsg WXM_ESPIPE; 	// Illegal seek 
extern const WxErrMsg WXM_EROFS;  	// Read-only file system 
extern const WxErrMsg WXM_EMLINK; 	// Too many links 
extern const WxErrMsg WXM_EPIPE;  	// Broken pipe 
extern const WxErrMsg WXM_EDOM;   	// Math argument out of domain of func 
extern const WxErrMsg WXM_ERANGE; 	// Math result not representable 
extern const WxErrMsg WXM_EDEADLK;	// Resource deadlock would occur 
extern const WxErrMsg WXM_ENAMETOOLONG;// File name too long 
extern const WxErrMsg WXM_ENOLCK; 	// No record locks available 
extern const WxErrMsg WXM_ENOSYS; 	// Function not implemented 
extern const WxErrMsg WXM_ENOTEMPTY;	// Directory not empty 
extern const WxErrMsg WXM_ELOOP;  	// Too many symbolic links encountered
extern const WxErrMsg WXM_EWOULDBLOCK;  // Operation would block 
extern const WxErrMsg WXM_E41;          // Unknown error 41
extern const WxErrMsg WXM_ENOMSG;	// No message of desired type 
extern const WxErrMsg WXM_EIDRM;  	// Identifier removed 
extern const WxErrMsg WXM_ECHRNG; 	// Channel number out of range 
extern const WxErrMsg WXM_EL2NSYNC;	// Level 2 not synchronized 
extern const WxErrMsg WXM_EL3HLT; 	// Level 3 halted 
extern const WxErrMsg WXM_EL3RST; 	// Level 3 reset 
extern const WxErrMsg WXM_ELNRNG; 	// Link number out of range 
extern const WxErrMsg WXM_EUNATCH;	// Protocol driver not attached 
extern const WxErrMsg WXM_ENOCSI; 	// No CSI structure available 
extern const WxErrMsg WXM_EL2HLT; 	// Level 2 halted 
extern const WxErrMsg WXM_EBADE;  	// Invalid exchange 
extern const WxErrMsg WXM_EBADR;  	// Invalid request descriptor 
extern const WxErrMsg WXM_EXFULL; 	// Exchange full 
extern const WxErrMsg WXM_ENOANO; 	// No anode 
extern const WxErrMsg WXM_EBADRQC;	// Invalid request code 
extern const WxErrMsg WXM_EBADSLT;	// Invalid slot 
extern const WxErrMsg WXM_EDEADLOCK;	// EDEADLK
extern const WxErrMsg WXM_E58;          // Unknown error 58
extern const WxErrMsg WXM_EBFONT; 	// Bad font file format 
extern const WxErrMsg WXM_ENOSTR; 	// Device not a stream 
extern const WxErrMsg WXM_ENODATA;	// No data available 
extern const WxErrMsg WXM_ETIME;  	// Timer expired 
extern const WxErrMsg WXM_ENOSR;  	// Out of streams resources 
extern const WxErrMsg WXM_ENONET; 	// Machine is not on the network 
extern const WxErrMsg WXM_ENOPKG; 	// Package not installed 
extern const WxErrMsg WXM_EREMOTE;	// Object is remote 
extern const WxErrMsg WXM_ENOLINK;	// Link has been severed 
extern const WxErrMsg WXM_EADV;   	// Advertise error 
extern const WxErrMsg WXM_ESRMNT; 	// Srmount error 
extern const WxErrMsg WXM_ECOMM;  	// Communication error on send 
extern const WxErrMsg WXM_EPROTO;       // Protocol error 
extern const WxErrMsg WXM_EMULTIHOP;	// Multihop attempted 
extern const WxErrMsg WXM_EDOTDOT;	// RFS specific error 
extern const WxErrMsg WXM_EBADMSG;	// Not a data message 
extern const WxErrMsg WXM_EOVERFLOW;	// Value too large for defined data type 
extern const WxErrMsg WXM_ENOTUNIQ; 	// Name not unique on network 
extern const WxErrMsg WXM_EBADFD; 	// File descriptor in bad state 
extern const WxErrMsg WXM_EREMCHG;	// Remote address changed 
extern const WxErrMsg WXM_ELIBACC;	// Can not access a needed shared library 
extern const WxErrMsg WXM_ELIBBAD;	// Accessing a corrupted shared library 
extern const WxErrMsg WXM_ELIBSCN;	// .lib section in a.out corrupted 
extern const WxErrMsg WXM_ELIBMAX;	// Attempting to link in too many shared libraries 
extern const WxErrMsg WXM_ELIBEXEC;	// Cannot exec a shared library directly 
extern const WxErrMsg WXM_EILSEQ;  	// Illegal byte sequence 
extern const WxErrMsg WXM_ERESTART;	// Interrupted system call should be restarted 
extern const WxErrMsg WXM_ESTRPIPE;	// Streams pipe error 
extern const WxErrMsg WXM_EUSERS;     // Too many users 
extern const WxErrMsg WXM_ENOTSOCK;   // Socket operation on non-socket 
extern const WxErrMsg WXM_EDESTADDRREQ;// Destination address required 
extern const WxErrMsg WXM_EMSGSIZE;   // Message too long 
extern const WxErrMsg WXM_EPROTOTYPE; // Protocol wrong type for socket 
extern const WxErrMsg WXM_ENOPROTOOPT;// Protocol not available 
extern const WxErrMsg WXM_EPROTONOSUPPORT;// Protocol not supported 
extern const WxErrMsg WXM_ESOCKTNOSUPPORT;// Socket type not supported 
extern const WxErrMsg WXM_EOPNOTSUPP;	// Operation not supported on transport endpoint 
extern const WxErrMsg WXM_EPFNOSUPPORT;// Protocol family not supported 
extern const WxErrMsg WXM_EAFNOSUPPORT;// Address family not supported by protocol 
extern const WxErrMsg WXM_EADDRINUSE;	// Address already in use 
extern const WxErrMsg WXM_EADDRNOTAVAIL;// Cannot assign requested address 
extern const WxErrMsg WXM_ENETDOWN;	// Network is down 
extern const WxErrMsg WXM_ENETUNREACH;// Network is unreachable 
extern const WxErrMsg WXM_ENETRESET;	// Network dropped connection because of reset 
extern const WxErrMsg WXM_ECONNABORTED;// Software caused connection abort 
extern const WxErrMsg WXM_ECONNRESET;  // Connection reset by peer 
extern const WxErrMsg WXM_ENOBUFS;     // No buffer space available 
extern const WxErrMsg WXM_EISCONN;     // Transport endpoint is already connected 
extern const WxErrMsg WXM_ENOTCONN;    // Transport endpoint is not connected 
extern const WxErrMsg WXM_ESHUTDOWN;   // Cannot send after transport endpoint shutdown 
extern const WxErrMsg WXM_ETOOMANYREFS;// Too many references: cannot splice 
extern const WxErrMsg WXM_ETIMEDOUT;   // Connection timed out 
extern const WxErrMsg WXM_ECONNREFUSED;// Connection refused 
extern const WxErrMsg WXM_EHOSTDOWN;   // Host is down 
extern const WxErrMsg WXM_EHOSTUNREACH;// No route to host 
extern const WxErrMsg WXM_EALREADY;    // Operation already in progress 
extern const WxErrMsg WXM_EINPROGRESS; // Operation now in progress 
extern const WxErrMsg WXM_ESTALE;      // Stale NFS file handle 
extern const WxErrMsg WXM_EUCLEAN;     // Structure needs cleaning 
extern const WxErrMsg WXM_ENOTNAM;     // Not a XENIX named type file 
extern const WxErrMsg WXM_ENAVAIL;     // No XENIX semaphores available 
extern const WxErrMsg WXM_EISNAM;      // Is a named type file 
extern const WxErrMsg WXM_EREMOTEIO;   // Remote I/O error 
extern const WxErrMsg WXM_EDQUOT;      // Quota exceeded 
extern const WxErrMsg WXM_ENOMEDIUM;   // No medium found 
extern const WxErrMsg WXM_EMEDIUMTYPE; // Wrong medium type 
extern const WxErrMsg WXM_E125;        // Operation canceled
extern const WxErrMsg WXM_E126;        // Unknown error 126
extern const WxErrMsg WXM_E127;        // Unknown error 127

//----------------------//
//  WxLib added errmsg  //
//----------------------//
extern const WxErrMsg WXM_THRDCAN;     // Thread canceled
extern const WxErrMsg WXM_NDEFAULT;    // Object not default
extern const WxErrMsg WXM_DIVZERO;     // Divisor is zero
extern const WxErrMsg WXM_MATHNEG;     // Math negatation failure
extern const WxErrMsg WXM_VFDIS;       // Virtual function disabled

/*
 [Thread-Safe]

 Note: All member functions do not throw exception but those involve 
       errno conversions (WxErrNo(int),reset(int).assign(int),operator=(int)).

 Note: WxErrMsg is not valid between processs, not valid for storage.
*/
class WxErrMsg {
  public:
    enum { MaxErrNo=127 };   // errno is supposed in [0-MaxErrNo]

    //
    // Note: default object is not identical to OK object.
    //
    inline WxErrMsg() throw()
              : _emsg(0) {};

    inline WxErrMsg(const WxErrMsg &src) throw()
              : _emsg(src._emsg) {};

    //
    // Note: Application could use C-library errno by this conversion.
    //       The drawback is a bit slower.
    // Note: explicit to prevent implicit conversion because this constructor
    //       may throw exception, mix-up with errno...
    //
    inline explicit WxErrMsg(int en) throw(std::exception)
              { _set_emsg(en); };

    inline bool is_default(void) const throw() 
              { return _emsg==0; };

    inline void reset(void) throw()
              { _emsg=0; };
    inline void reset(const WxErrMsg& ec) throw()
              { _emsg=ec._emsg; };

    inline void assign(const WxErrMsg& ec) throw()
              { _emsg=ec._emsg; };
    int assign(int en) throw(std::exception);

    inline const WxErrMsg& operator =(const WxErrMsg& rhs) throw()
              { _emsg=rhs._emsg; return *this; };

    inline const WxErrMsg& operator =(int en) throw(std::exception)
              { _set_emsg(en); return *this; };

    inline bool operator==(const WxErrMsg& rhs) const throw()
              { return _emsg==rhs._emsg; };
    inline bool operator!=(const WxErrMsg& rhs) const throw()
              { return _emsg!=rhs._emsg; };
 
    int c_errno(void) const throw(std::exception);

    //
    // [Syn] Get the enrolled string of this object.
    //
    // [Ret] pointer to The enrolled descriptive string
    //
    const char* c_errstr(void) const throw()
              { return (_emsg==0)? "Default error string":_emsg; };

    inline static WxErrMsg enroll(const char* estr) throw() 
              {
                //
                // Checking for EFAULT/EFBIG save usage errors,
                // Left to be defined.
                // enroll(0) returns default object
                //
                return WxErrMsg(estr); 
              };
  private:
    inline explicit WxErrMsg(const char* estr) throw()
              : _emsg(estr) {};

    //
    // [Exception] Wx_bad_alloc
    //             Wx_general_error
    //             Wx_bad_errno     argument not in the range [0-MaxErrNo]
    //     
    void _set_emsg(int en) throw(std::exception);
    static const char* const wx_errno_tab[];
    const char* _emsg;
};

#endif
