/* Copyright is licensed under GNU LGPL.                (I.J. Wang, 2003)
*/

#ifndef WXPTR_H__
#define WXPTR_H__
#define WXPTR_VERSION 8

#include "wxret.h"
#include <pthread.h>

/*
   WxPtr is a template for cancellation-safe pointers.
   This file contains definitions of WxPtr without cpp file.

   Note: Thread cancellation does not call destructors, as the usual pattern of C++.
         (Putting these codes in the constructor/destructor is a good idea but
          not without its drawbacks and less flexible.)

   Note: This class is intended locally constructed (auto)
         The alternative is to handle pointers or objects in WxThread::tmain_leave
  
   Note: This template does not handle array, i.g. WxPtr<T> obj(new T[]).
*/
template<class T>
class WxPtr {
  public:
    typedef T DataType;

    //
    // [Imp] Memory block is allocated in the the local stack.
    //
    explicit WxPtr(DataType* p) throw(std::exception)
             try : _ptr(p)
              {
               ::_pthread_cleanup_push(&_buf,
	          reinterpret_cast<void(*)(void*)>(_delete_func),
	          static_cast<void *>(this));
              }
              catch(const std::bad_alloc&) {
                WX_THROW( Wx_bad_alloc() );
              }
              catch(...) {
                WX_THROW( Wx_general_error() );
              };

    //
    // [Imp] Pops out the memory block in the local stack
    //   
    ~WxPtr()
              {
                ::_pthread_cleanup_pop(&_buf,_ptr!=0);
                _ptr=0; 
              };

    void reset(DataType* p)
              {
               if(p==_ptr) {
                 return; 
               }
               if(_ptr!=0) {
                  delete _ptr;
               }
                _ptr=p; 
              };

    DataType& operator*() const throw() { return *_ptr; };
    DataType* operator->() const throw() { return _ptr; };

    DataType* get(void) const throw() { return _ptr; };

    //
    // Note: Memory allocated in stack is not released until object destroyed.
    //
    DataType* release(void) throw()
              {
               DataType* retv=_ptr;
               _ptr=0;
               return(retv);
              };
  private:
    static void _delete_func(WxPtr* p)
              {
               if(p->_ptr!=0) {              // what would this -> refers to?
                 delete p->_ptr; p->_ptr=0;
               }
              };
    DataType* _ptr;
    struct _pthread_cleanup_buffer _buf;
  private:
    WxPtr(const WxPtr&);                  // not to use
    const WxPtr &operator=(const WxPtr&); // not to use

    bool operator==(const WxPtr& r) const throw(); // not to use
};

#endif
