/* Copyright is licensed under GNU LGPL.                (I.J. Wang, 2003)
*/

#include "wxpathname.h"
#include <cctype>

const char WxPathName::class_name[]="WxPathName";

//
// Use POSIX filename
//
//#define WX__USE_POSIX_FILENAME

//
// [Syn] Check for valid pathname
//
// [Exception] Wx_general_error
//             Wx_bad_alloc
//
// [Ret] OK               pathname is valid
//       WXM_EINVAL       invalid character or double delimiter in pathname
//       WXM_ENAMETOOLONG pathname too long
//       WXM_EFAULT       pathname pointer is zero
//
// PATHNAME::= EMPTY
//         ::= [DELIMITER]. FILENAME ,{DELIMITER+FILENAME}+ ,[DELIMITER]
// FILENAME::= {FILENAMECHAR}+
// FILENAMECHAR::= any character except DELIMITER and \0
// DELIMITER::= '/'
//
WxRet WxPathName::_chk_pathname(const char* pathname, size_t plen) const throw(std::exception)
try {
 if(pathname==0) {
   WX_RETURN(WXM_EFAULT);
 }
 if(plen==0) {
   return(OK);
 }
 if(plen>PATH_MAX) {
    WX_RETURN(WXM_ENAMETOOLONG); // pathname too long
 }

 //
 // Scan for invalid pathname elements
 //
 for(size_t i=0; i<plen; ++i) {
   const char ch=pathname[i];
   if(ch==0) {
     WX_RETURN(WXM_EINVAL);   // pathname contains \0
   }
#ifdef WX__USE_POSIX_FILENAME
   if(std::isalnum(ch)||(ch=='.')||(ch=='-')==(ch=='_')) {
     continue;
   }
#endif
   if(ch==Delimiter) {
     if((i+1<plen)&&(pathname[i+1]==Delimiter)) {
       WX_RETURN(WXM_EINVAL);   // double delimiter
     }
   } else {
#ifdef WX__USE_POSIX_FILENAME
       WX_RETURN(WXM_EINVAL);   // pathname contains invalid character
#endif
   }
 }
 return(OK);
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

//
// [Syn] Check for valid pathname
//
// [Exception] Wx_general_error
//             Wx_bad_alloc
//
// [Ret] OK    pathname is valid. 
//             Length of the pathname is stored to size_ptr, if it is not zero
//       WXM_EINVAL       invalid character or double delimiter in pathname
//       WXM_ENAMETOOLONG pathname too long
//       WXM_EFAULT       pathname pointer is zero
//
WxRet WxPathName::_chk_pathname(const char* pathname,size_t* size_ptr) const throw(std::exception)
try {
 if(pathname==0) {
   WX_RETURN(WXM_EFAULT);
 }
 //
 // Scan for invalid pathname elements
 //
 size_t pz=0;
 for(pz=0; pz<PATH_MAX; ++pz) {
   const char ch=pathname[pz];
   if(ch==0) {
     break;
   }
#ifdef WX__USE_POSIX_FILENAME
   if(std::isalnum(ch)||(ch=='.')||(ch=='-')==(ch=='_')) {
     continue;
   }
#endif
   if(ch==Delimiter) {
     if((pz+1<PATH_MAX)&&(pathname[pz+1]==Delimiter)) {
       WX_RETURN(WXM_EINVAL);   // double delimiter
     }
   } else {
#ifdef WX__USE_POSIX_FILENAME
       WX_RETURN(WXM_EINVAL);   // pathname contains invalid character
#endif
   }
 }
 if(pz>=PATH_MAX) {
   if(pathname[pz]==0) {
     if(size_ptr!=0) {
       *size_ptr=pz;
     }
     return(OK);
   }
   WX_RETURN(WXM_ENAMETOOLONG); // pathname too long
 }
 if(size_ptr!=0) {
   *size_ptr=pz;
 }
 return(OK);
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

//
// [Syn] Assign _faddr to contain plen characters pointed by pathname
//       The trailing delimiter is removed except pathname contains only the
//       delimiter.
//
// [Exception] Wx_general_error
//             Wx_bad_alloc
//
// [Ret] OK               pathname is valid
//       WXM_ENAMETOOLONG pathname too long
//       WXM_EFAULT       pathname pointer is zero
//
WxRet WxPathName::_assign(const char* pathname,size_t plen) throw(std::exception)
try {
 if(pathname==0) {
   WX_RETURN(WXM_EFAULT);
 }
 if(plen<=0) {
   _faddr.reset();
   return(OK);
 }

 WxRet r;
 //
 // Not assign the trailing delimiter except the pathname contains exactly the
 // only delimiter.
 //
 if(pathname[plen-1]==Delimiter) {
    if(plen==1) {
      _faddr=pathname[0];
    } else {
      if((r=_faddr.assign(pathname,plen-1))!=OK) {
        WX_THROW( Wx_general_error() );  // no possible reason to fail
      }
    }
    return(OK);
 }
 if((r=_faddr.assign(pathname,plen))!=OK) {
   WX_THROW( Wx_general_error() );  // no possible reason to fail
 }
 return(OK);
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

//----------------------------------------------------------------------
WxPathName::WxPathName(const WxPathName& pathname) throw(std::exception)
try : _faddr(pathname._faddr) {
}
catch(const WxStr::Fault& e) {
 WX_THROW( Wx_general_error() );
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxPathName::WxPathName(const WxStr& pathname) throw(std::exception,Fault)
try : _faddr() {
 WxRet r=_chk_pathname(pathname.data(),pathname.size());
 if(r!=OK) {
   WX_THROW( Fault(r) );
 }
 if((r=_assign(pathname.data(),pathname.size()))!=OK) {
   WX_THROW( Fault(r) );
 }
 return;
}
catch(const WxStr::Fault&) {
 WX_THROW( Wx_general_error() );
}
catch(const WxPathName::Fault&) {
  throw;
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxPathName::WxPathName(const std::string& pathname) throw(std::exception,Fault)
try : _faddr() {
 WxRet r=_chk_pathname(pathname.data(),pathname.size());
 if(r!=OK) {
   WX_THROW( Fault(r) );
 }
 if((r=_assign(pathname.data(),pathname.size()))!=OK) {
   WX_THROW( Fault(r) );
 }
 return;
}
catch(const WxPathName::Fault&) {
  throw;
}
catch(const WxStr::Fault&) {
 WX_THROW( Wx_general_error() );
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxPathName::WxPathName(const char* pathname) throw(std::exception,Fault)
try : _faddr() {
 size_t psize;
 WxRet r=_chk_pathname(pathname,&psize);
 if(r!=OK) {
   WX_THROW( Fault(r) );
 }
 if(psize<=0) {
   return;
 }
 if((r=_assign(pathname,psize))!=OK) {
   WX_THROW( Fault(r) );
 }
 return;
}
catch(const WxPathName::Fault&) {
  throw;
}
catch(const WxStr::Fault&) {
 WX_THROW( Wx_general_error() );
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxStr WxPathName::filename(void) const throw(std::exception)
try {
 if(_faddr.is_default()) {
   return WxStr();   // default object
 }
 if((_faddr.size()==1)&&(_faddr[0]==Delimiter)) {
   return WxStr();   // default object
 }
 size_t ia;
 if(_faddr.rfind(ia,_faddr.size()-1,Delimiter)!=OK) {
   return _faddr;    // no delimiter found
 }
 ++ia;
 return WxStr(_faddr,ia,_faddr.size()-ia);
}
catch(const WxStr::Fault& e) {
 WX_THROW( Wx_general_error() );
}
catch(const Wx_general_error&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

void WxPathName::reset(void) throw(std::exception)
try {
  return _faddr.reset();
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet WxPathName::assign(const WxStr& pathname) throw(std::exception)
try {
 WxRet r=_chk_pathname(pathname.data(),pathname.size());
 if(r!=OK) {
   WX_RETURN(r);
 }
 WX_RETURN( _assign(pathname.data(), pathname.size()) );
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet WxPathName::assign(const std::string& pathname) throw(std::exception)
try {
 WxRet r=_chk_pathname(pathname.data(),pathname.size());
 if(r!=OK) {
   WX_RETURN(r);
 }
 WX_RETURN( _assign(pathname.data(), pathname.size()) );
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet WxPathName::assign(const char* pathname) throw(std::exception)
try {
 size_t psize;
 const WxRet r=_chk_pathname(pathname,&psize);
 if(r!=OK) {
   WX_RETURN(r);
 }
 WX_RETURN( _assign(pathname, psize) );
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet WxPathName::assign(const char* pathname,size_t len) throw(std::exception)
try {
 const WxRet r=_chk_pathname(pathname,len);
 if(r!=OK) {
   WX_RETURN(r);
 }
 WX_RETURN( _assign(pathname, len) );
 return(OK);
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

void WxPathName::assign(const WxPathName& pathname) throw(std::exception)
try {
 _faddr=pathname._faddr;
 return;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

const WxPathName& WxPathName::operator =(const WxPathName& rhs) throw(std::exception)
try {
  _faddr=rhs._faddr;
  return *this;
}
catch(const WxStr::Fault& e) {
 WX_THROW( Wx_general_error() );
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

const WxPathName& WxPathName::operator =(const WxStr& rhs) throw(std::exception,Fault)
try {
 const WxRet r=assign(rhs);
 if(r!=OK) {
   WX_THROW( Fault(r) );
 }
 return *this;
}
catch(const WxPathName::Fault&) {
  throw;
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

const WxPathName& WxPathName::operator =(const std::string& rhs) throw(std::exception,Fault)
try {
 const WxRet r=assign(rhs);
 if(r!=OK) {
   WX_THROW( Fault(r) );
 }
 return *this;
}
catch(const WxPathName::Fault&) {
  throw;
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

const WxPathName& WxPathName::operator =(const char* rhs) throw(std::exception,Fault)
try {
 const WxRet r=assign(rhs);
 if(r!=OK) {
   WX_THROW( Fault(r) );
 }
 return *this;
}
catch(const WxPathName::Fault&) {
  throw;
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

bool WxPathName::operator ==(const WxPathName& rhs) const throw(std::exception)
try {
  return _faddr==rhs._faddr;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

//-----------------------------------------------------------------
namespace Wx {

WxRet unlink(const WxPathName& pathname) throw(std::exception)
try {
 if(::unlink(pathname.pathname().c_str())!=0) {
   WX_RETURN(WxErrMsg(errno));
 }
 return(OK);
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet access(const WxPathName& pathname,int mode) throw(std::exception)
try {
 if(::access(pathname.pathname().c_str(),mode)!=0) {
   WX_RETURN(WxErrMsg(errno));
 };
 return(OK);
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet chdir(const WxPathName& path) throw(std::exception)
try {
 if(::chdir(path.pathname().c_str())!=0) {
   WX_RETURN(WxErrMsg(errno));
 };
 return(OK);
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet getcwd(WxPathName& path) throw(std::exception)
try {
 for(size_t bsize=256; bsize<=PATH_MAX+256; bsize+=256) {
   std::auto_ptr<char>buf(new char[bsize]);
   if(::getcwd(buf.get(),bsize)==buf.get()) {
     path.assign(buf.get());
     return(OK);
   }
   const int v=errno;
   if(v==ERANGE) {
     continue;
   }
   switch(v) {
     case EACCES: // FALLTHROUGH
     case ENOENT:
          WX_RETURN(WxErrMsg(v));
   };
   WX_THROW( Wx_bad_errno(v) );   // not expected errno
 }
 WX_THROW(Wx_general_error());  // name too long
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet chown(const WxPathName& path,uid_t owner,gid_t group) throw(std::exception)
try {
 if(::chown(path.pathname().c_str(),owner,group)!=0) {
   WX_RETURN(WxErrMsg(errno));
 };
 return(OK);
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet chmod(const WxPathName& path,mode_t mode) throw(std::exception)
try {
 if(::chmod(path.pathname().c_str(),mode)!=0) {
   WX_RETURN(WxErrMsg(errno));
 };
 return(OK);
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

};
