/* Copyright is licensed under GNU LGPL.                (I.J. Wang, 2003)
*/

#ifndef WXONCE_H__
#define WXONCE_H__
#define WXONCE_VERSION 8

#include "wxret.h"     // include WxLib base definitions
#include <pthread.h>

/*
  WxOnce defines the object that ensures the only member function once(...)
  calls the given paramter function at most once.

*/
class WxOnce {
  public:
    //
    // [Syn] Construct default object
    //
    inline WxOnce() throw() 
	  :_ovar(DefaultValue) {};
    
    //
    // [Syn] Construct object and call the given parameter function immedicately.
    //
    // Note: If object is declared auto, static modifier may be required.
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //
    inline WxOnce( void(*once_func)(void) ) throw(std::exception)
	      :_ovar(DefaultValue) { once(once_func); };

    inline ~WxOnce() throw() 
              {};

    //
    // [Syn] Is object the same as that of default constructed state.
    //
    // [Ret] true = object is the same as default constructed state.
    //       false= otherwise
    //
    bool is_default(void) const throw()
              { return _ovar==DefaultValue; };

    //
    // [Syn] Call the parameter function at most once.
    //       This function moves object away from the default state
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //
    inline void once( void(*once_func)(void) ) throw(std::exception)
              try {
                ::pthread_once(&_ovar, once_func); 
              }
              catch(const std::bad_alloc&) {
                WX_THROW( Wx_bad_alloc() );
              }
              catch(...) {
                WX_THROW(Wx_general_error());
              };
  private:
    static const ::pthread_once_t DefaultValue=PTHREAD_ONCE_INIT;
    ::pthread_once_t _ovar;

    WxOnce(const WxOnce&) throw(); // not to use
    const WxOnce& operator=(const WxOnce&) throw(); // not to use
    bool operator==(const WxOnce&) throw(); // not to use
};

#endif
