/* Copyright is licensed under GNU LGPL.                (I.J. Wang, 2003)
*/

#ifndef WXNUM_H__
#define WXNUM_H__
#define WXNUM_VERSION 8

#include "wxret.h"
#include <cstring>
#include <limits>
#include "wxstr.h"

namespace Wx {
 //
 // [Syn] Convert num to string in radix representation, append the result to str.
 //
 // Note: The digit greater than 9 is represented from 'A' to 'Z'. (10-35)
 //
 // [Exception] Wx_bad_alloc
 //             Wx_general_error
 //
 // [Ret] OK      (minus prefixes the result if num is negative)
 //       WXM_EINVAL  radix not in the range [2-36]
 //
 WxRet append_number(WxStr &str,   int num, int radix) throw(std::exception);
 WxRet append_number(WxStr &str,  uint num, int radix) throw(std::exception);
 WxRet append_number(WxStr &str,  long num, int radix) throw(std::exception);
 WxRet append_number(WxStr &str, ulong num, int radix) throw(std::exception);
 WxRet append_number(WxStr &str, long long num, int radix) throw(std::exception);
 WxRet append_number(WxStr &str, unsigned long long num, int radix) throw(std::exception);

};

namespace WxNum {
 //
 WX_DECLARE_FAULT;
 inline uint cvr_unsign(int n) throw(Fault)
    {
     if(n<0) {
       WX_THROW( Fault(WXM_EINVAL) );
     }
    };
 inline int cvr_sign(uint n) throw(Fault)
    {
     if(n>uint(std::numeric_limits<int>::max())) {
       WX_THROW( Fault(WXM_EDOM) );
     }
    };

 //
 // [Syn] Get the absolute value of n
 //
 // [Exception] Wx_bad_alloc
 //             Wx_general_error
 //
 // [Exception] Fault
 //       WXM_MATHNEG     math negation failure
 //
 // [Ret] The absolute value of n
 //
 int abs(int n) throw(std::exception,Fault);
 long int abs(long int n) throw(std::exception,Fault);
 long long int abs(long long int n) throw(std::exception,Fault);

 //
 // [Syn] compute division
 //
 // [Exception] Wx_bad_alloc
 //             Wx_general_error
 //
 // [Exception] Fault
 //       WXM_DIVZERO     divisor is zero
 //       WXM_MATHNEG     math negation failure
 //
 // [Ret] struct div_t,ldiv_t,lldiv_t of operation result
 //
 std::div_t div(int number, int denom) throw(std::exception,Fault);
 std::ldiv_t div(long int number, long int denom) throw(std::exception,Fault);
 std::lldiv_t div(long long int number, long long int denom) throw(std::exception,Fault);

 //
 // [Syn] Square root of whole number
 //
 // [Exception] Wx_bad_alloc
 //             Wx_general_error
 //
 // [Ret] Floor of the square root of n
 //
 uint sqrt(uint n) throw(std::exception);
 unsigned long long sqrt(unsigned long long n) throw(std::exception);

};
#endif
