/* Copyright is licensed under GNU LGPL.                (I.J. Wang, 2003)
*/

#include "wxnum.h"
#include <cstdlib>

namespace Wx {

WxRet append_number(WxStr &str, int num, int radix) throw(std::exception)
try {
 if((radix<2)||(radix>36)) {
   WX_RETURN(WXM_EINVAL);
 }

 if(std::numeric_limits<int>::digits>32) {
   WX_ABORT();
 }
 char dbuf[34];  // 32*('0'|'1') +'-' + 0
 std::div_t dt;
 int n;
 
 dt.quot=num;
 n=0;
 do {
   dt=std::div(dt.quot,radix);
   dbuf[n++]=(char)(dt.rem);
 } while(dt.quot!=0);

 if(num<0) {
   str+='-';
   for(--n; n>=0; --n) {
     const char ch=-dbuf[n];
     str+= (ch>9)?  (ch+'7'):(ch+'0');
   }
 } else {
   for(--n; n>=0; --n) {
     const char ch=dbuf[n];
     str+= (ch>9)?  (ch+'7'):(ch+'0');
   }
 }
 return(OK);
}
catch(const WxStr::Fault& e) {
  WX_THROW(Wx_general_error());
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
  WX_THROW(Wx_general_error());
};

WxRet append_number(WxStr &str, long num, int radix) throw(std::exception)
try {
 if((radix<2)||(radix>36)) {
   WX_RETURN(WXM_EINVAL);
 }

 if(std::numeric_limits<long>::digits>32) {
   WX_ABORT();
 }
 char dbuf[34];  // 32*('0'|'1') +'-' + 0
 std::ldiv_t dt;
 long n;

 dt.quot=num;
 n=0;
 do {
   dt=std::ldiv(dt.quot,radix);
   dbuf[n++]=(char)(dt.rem);
 } while(dt.quot!=0);

 if(num<0) {
   str+='-';
   for(--n; n>=0; --n) {
     const char ch=-dbuf[n];
     str+= (ch>9)?  (ch+'7'):(ch+'0');
   }
 } else {
   for(--n; n>=0; --n) {
     const char ch=dbuf[n];
     str+= (ch>9)?  (ch+'7'):(ch+'0');
   }
 }
 return(OK);
}
catch(const WxStr::Fault& e) {
  WX_THROW(Wx_general_error());
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
  WX_THROW(Wx_general_error());
};

WxRet append_number(WxStr &str, long long num, int radix) throw(std::exception)
try {
 if((radix<2)||(radix>36)) {
   WX_RETURN(WXM_EINVAL);
 }

 if(std::numeric_limits<long long>::digits>64) {
   WX_ABORT();
 }
 char dbuf[66];  // 64*('0'|'1') +'-' + 0
 std::lldiv_t dt;
 long long n;

 dt.quot=num;
 n=0;
 do {
   dt=std::lldiv(dt.quot,radix);
   dbuf[n++]=(char)(dt.rem);
 } while(dt.quot!=0);

 if(num<0) {
   str+='-';
   for(--n; n>=0; --n) {
     const char ch=-dbuf[n];
     str+= (ch>9)?  (ch+'7'):(ch+'0');
   }
 } else {
   for(--n; n>=0; --n) {
     const char ch=dbuf[n];
     str+= (ch>9)?  (ch+'7'):(ch+'0');
   }
 }
 return(OK);
}
catch(const WxStr::Fault& e) {
  WX_THROW(Wx_general_error());
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
  WX_THROW(Wx_general_error());
};

WxRet append_number(WxStr &str, uint num, int radix) throw(std::exception)
try {
 if((std::numeric_limits<uint>::digits-std::numeric_limits<int>::digits)!=1) {
   WX_ABORT();
 }
 if(num<=uint(std::numeric_limits<int>::max())) {
   return append_number(str,int(num),radix);
 }

 {
   uint q=num/radix;
   if(q>uint(std::numeric_limits<int>::max())) {
     WX_THROW(Wx_general_error());
   }
   WxRet r=append_number(str,int(q),radix);
   if(r!=OK) {
     WX_RETURN(r);
   }
 }

 {
  const char ch=num%radix;
  str+= (ch>9)?  (ch+'7'):(ch+'0');
 }
 return(OK);
}
catch(const WxStr::Fault& e) {
  WX_THROW(Wx_general_error());
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const Wx_general_error&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
  WX_THROW(Wx_general_error());
};

WxRet append_number(WxStr &str, ulong num, int radix) throw(std::exception)
try {
 if((std::numeric_limits<ulong>::digits-std::numeric_limits<long>::digits)!=1) {
   WX_ABORT();
 }
 if(num<=ulong(std::numeric_limits<long>::max())) {
   return append_number(str,long(num),radix);
 }

 {
   ulong q=num/radix;
   if(q>ulong(std::numeric_limits<long>::max())) {
     WX_THROW(Wx_general_error());
   }
   WxRet r=append_number(str,long(q),radix);
   if(r!=OK) {
     WX_RETURN(r);
   }
 }

 {
  const char ch=num%radix;
  str+= (ch>9)?  (ch+'7'):(ch+'0');
 }
 return(OK);
}
catch(const WxStr::Fault& e) {
  WX_THROW(Wx_general_error());
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const Wx_general_error&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
  WX_THROW(Wx_general_error());
};

WxRet append_number(WxStr &str, unsigned long long num, int radix) throw(std::exception)
try {
 if((std::numeric_limits<unsigned long long>::digits
    -std::numeric_limits<long long>::digits)!=1) {
   WX_ABORT();
 }
 if(num<=(unsigned long long)(std::numeric_limits<long long>::max())) {
   return append_number(str,(long long)(num),radix);
 }

 {
   unsigned long long q=num/radix;
   if(q>(unsigned long long)(std::numeric_limits<long long>::max())) {
     WX_THROW(Wx_general_error());
   }
   WxRet r=append_number(str,(long long)(q),radix);
   if(r!=OK) {
     WX_RETURN(r);
   }
 }

 {
  const char ch=num%radix;
  str+= (ch>9)?  (ch+'7'):(ch+'0');
 }
 return(OK);
}
catch(const WxStr::Fault& e) {
  WX_THROW(Wx_general_error());
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const Wx_general_error&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
  WX_THROW(Wx_general_error());
};

};  // end Wx
//---------------------------------------------------------------

inline static bool wx__testfail_neg(int n)
  { return n<=INT_MIN; };
inline static bool wx__testfail_neg(long int n)
  { return n<=LONG_MIN; };
inline static bool wx__testfail_neg(long long int n)
  { return n<=LONG_LONG_MIN; };

namespace WxNum {

int abs(int n) throw(std::exception,Fault)
try {
 if(wx__testfail_neg(n)) {
   WX_THROW( Fault(WXM_MATHNEG) );
 }
 return std::abs(n);
}
catch(const WxNum::Fault&) {
 throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
  WX_THROW(Wx_general_error());
};

long int abs(long int n) throw(std::exception,Fault)
try {
 if(wx__testfail_neg(n)) {
   WX_THROW( Fault(WXM_MATHNEG) );
 }
 return std::labs(n);
}
catch(const WxNum::Fault&) {
 throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
  WX_THROW(Wx_general_error());
};

long long int abs(long long int n) throw(std::exception,Fault)
try {
 if(wx__testfail_neg(n)) {
   WX_THROW( Fault(WXM_MATHNEG) );
 }
 return std::llabs(n);
}
catch(const WxNum::Fault&) {
 throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
  WX_THROW(Wx_general_error());
};

std::div_t div(int number, int denom) throw(std::exception,Fault)
try {
 if(denom>=0) {
   if(denom==0) {
     WX_THROW( Fault(WXM_DIVZERO) );
   }
 } else {
   if((denom==-1)&&(wx__testfail_neg(number))) {
     WX_THROW( Fault(WXM_MATHNEG) );
   }
 }
 return std::div(number,denom);
}
catch(const WxNum::Fault&) {
 throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
  WX_THROW(Wx_general_error());
};

std::ldiv_t div(long int number, long int denom) throw(std::exception,Fault)
try {
 if(denom>=0) {
   if(denom==0) {
     WX_THROW( Fault(WXM_DIVZERO) );
   }
 } else {
   if((denom==-1)&&(wx__testfail_neg(number))) {
     WX_THROW( Fault(WXM_MATHNEG) );
   }
 }
 return std::ldiv(number,denom);
}
catch(const WxNum::Fault&) {
 throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
  WX_THROW(Wx_general_error());
};

std::lldiv_t div(long long int number, long long int denom) throw(std::exception,Fault)
try {
 if(denom>=0) {
   if(denom==0) {
     WX_THROW( Fault(WXM_DIVZERO) );
   }
 } else {
   if((denom==-1)&&(wx__testfail_neg(number))) {
     WX_THROW( Fault(WXM_MATHNEG) );
   }
 }
 return std::lldiv(number,denom);
}
catch(const WxNum::Fault&) {
 throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
  WX_THROW(Wx_general_error());
};

uint sqrt(uint n) throw(std::exception)
try {
 if(n<=1) {
   return(n);
 }

 uint s1,s2=n>>1;
 do {
   s1=s2;
   s2=(s1+n/s1)>>1;
 } while(s2<s1);

 return(s1);
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
  WX_THROW(Wx_general_error());
};

unsigned long long sqrt(unsigned long long n) throw(std::exception)
try {
 if(n<=1) {
   return(n);
 }

 unsigned long long s1,s2=n>>1;
 do {
   s1=s2;
   s2=(s1+n/s1)>>1;
 } while(s2<s1);

 return(s1);
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
  WX_THROW(Wx_general_error());
};

};  // end WxNum
