/* Copyright is licensed under GNU LGPL.                (I.J. Wang, 2003)

   WxMutex provides serializing access of codes between threads.

  [Avoided C Use] pthread_mutex_t, struct _pthread_cleanup_buffer
                    pthread_mutexattr_t
         pthread_mutex_lock, pthread_mutex_unlock, pthread_mutex_init,
         pthread_mutex_trylock, pthread_mutex_destroy
         _pthread_cleanup_push, _pthread_cleanup_pop
         pthread_mutexattr_settype, pthread_mutexattr_destroy,

  Note: Some of the error messages are based on Linux Programmer's Manual.
*/
#ifndef WXMUTEX_H__
#define WXMUTEX_H__
#define WXMUTEX_VERSION 8

#include "wxret.h"
#include <pthread.h>

/*
  Note: Operation of WxMutex is often done by the proxy WxLock. 
        Refer to WxLock.
*/
class WxMutex {
  public:
    static const char class_name[];
    WX_DECLARE_FAULT;  // declaraion of Fault

    //
    // [Syn] Enumeration of the mutex type
    //
    enum MutexType { Fast=0,ErrorCheck,Recursive };

    WxMutex(MutexType t=Fast) throw(std::exception);

    ~WxMutex() throw(std::exception);

    WxRet lock(void) throw(std::exception);

    WxRet trylock(void) throw(std::exception);

    WxRet unlock(void) throw(std::exception);

    static void lock(WxMutex *mtx) throw(std::exception,Fault);

    static void trylock(WxMutex *mtx) throw(std::exception,Fault);

    static void unlock(WxMutex *mtx) throw(std::exception,Fault);

  private:
    //
    // Note: Dedicated for use by WxLock. see note1.
    // 
    // [Exception] Refer unlock(WxMutex*)
    // 
    inline static void s_unlock(WxMutex *mtx) throw(std::exception,Fault)
              { WxMutex::unlock(mtx); };

    long _sig;              // WxMutex uses signature for the waken up
                            // WxCond to check that this object is still valid.
                            // Subtlety of threaded codes need this.
    ::pthread_mutex_t _mtx; 
    friend class WxCond;    // WxCond needs to access _mtx directly
    friend class WxLock;    // WxLock needs to call s_unlock(...). see note1
  private:
    WxMutex(const WxMutex&);                   // no copy constructor
    const WxMutex& operator =(const WxMutex&); // no assign operator
    bool operator==(const WxMutex&) const;     // no comparison
};

/*
 [Thread-Safe]

 [Syn] WxLock lock the given WxMutex while constructed, unlock it while program leaves
       the scope or being cancelled. WxCond needs WxLock to perform wait functions.
 
 Note: Object should be declared auto and must not be accessed by more than one thread.

 [Example]
        WxMutex mtx;       // mutex defined anywhere accessible

        {
         WxLock aa(mtx);   // Construct WxLock object aa and lock mtx
                           // Codes follow in the curly braced scope are
           ...             // serialized for thread access.
        }
*/
class WxLock {
  public:
    static const char class_name[];
    WX_DECLARE_FAULT;  // declaraion of Fault

    //
    // [Not Cancelable]
    // [Syn] Construct object and lock the given mutex.
    //       On destruction or cancellation, the mutex is unlocked.
    //
    // [Exception] Wx_bad_errno
    //             Wx_general_error
    //             Wx_bad_alloc
    //
    // [Exception] Fault
    //       WXM_EFAULT mtx is zero
    //       WXM_EDEADLK already locked (ErrorCheck only)
    //       WXM_EINVAL mutex not properly initialized
    //
    // [Imp] Memory block is allocated in local stack (storing cleanup information).
    //
    explicit WxLock(WxMutex &mtx) throw(std::exception,Fault);

    //
    // [Not Cancelable]
    // [Syn] Destroy object and unlock the handling WxMutex if it is locked.
    //
    // [Exception] Wx_bad_errno
    //             Wx_general_error
    //             Wx_bad_alloc
    //
    ~WxLock() throw(std::exception);

    //
    // [Not Cancelable]
    // [Syn] Unlock the handling WxMutex. Object is no more usable for WxCond.
    //       This function is primarily for hierarchical locking method.
    //
    // Note: This function can be called at most once. There is no way to lock again.
    //       Memory block is left idle in the local stack until destroyed.
    //
    // [Exception] Wx_bad_errno
    //             Wx_general_error
    //             Wx_bad_alloc
    //
    // [Ret] OK    
    //       WXM_ENOENT mutex is unlocked
    //       WXM_EPERM  calling thread does not own the mutex (ErrorCheck only)
    //       WXM_EINVAL mutex not properly initialized
    // 
    // [Refer] unlock()
    // 
    WxRet unlock(void) throw(std::exception);

  private:
    WxMutex *_pmtx;
    struct _pthread_cleanup_buffer _buf;
    friend class WxCond;      // WxCond needs to access _pmtx directly
  private:
    WxLock();                               // not to use
    WxLock(const WxLock&);                  // not to use
    const WxLock &operator=(const WxLock&); // not to use
    bool operator==(const WxLock&) const;   // not to use
};

/*
   note1: I had problem letting the compiler to identify the static function
          'WxMutex::unlock', so use a different name, If a correct writting found,
          this should be removed.
*/
#endif
