/* Copyright is licensed under GNU LGPL.                (I.J. Wang, 2003)
*/

#include "wxmutex.h"

//
// [Internal Use] Wx__MutexAttr defines a class for use by WxMutex(constructor)
//                execlusively.
//
class Wx__MutexAttr {
  public:
    Wx__MutexAttr(WxMutex::MutexType k) throw(std::exception);
    ~Wx__MutexAttr() throw() { ::pthread_mutexattr_destroy(&_attr); };
    const ::pthread_mutexattr_t* attr(void) const throw() { return &_attr; };
  private:
    ::pthread_mutexattr_t _attr;
    Wx__MutexAttr();  // not to use
};

//
// .Class only used to declare const static object
//
Wx__MutexAttr::Wx__MutexAttr(WxMutex::MutexType k) throw(std::exception)
try {
 int v=::pthread_mutexattr_init(&_attr);
 if(v!=0) {
   WX_ABORT(v); // man. always return 0
 }
 switch(k) {
   case WxMutex::Fast:
        if((v=::pthread_mutexattr_settype(&_attr,PTHREAD_MUTEX_FAST_NP))!=0) {
          WX_THROW( Wx_bad_errno(v) ); // man. says only EINVAL
        }
        break;
   case WxMutex::ErrorCheck:
        if((v=::pthread_mutexattr_settype(&_attr,PTHREAD_MUTEX_ERRORCHECK_NP))!=0) {
          WX_THROW( Wx_bad_errno(v) ); // man. says only EINVAL
        }
        break;
   case WxMutex::Recursive:
        if((v=::pthread_mutexattr_settype(&_attr,PTHREAD_MUTEX_RECURSIVE_NP))!=0) {
          WX_THROW( Wx_bad_errno(v) ); // man. says only EINVAL
        }
        break;
   default:
        WX_THROW(Wx_general_error());
 }
}
catch(const Wx_bad_errno&) {
  ::pthread_mutexattr_destroy(&_attr);
  throw;
}
catch(const std::bad_alloc&) {
  ::pthread_mutexattr_destroy(&_attr);
  WX_THROW( Wx_bad_alloc() );
}
catch(const Wx_general_error&) {
  ::pthread_mutexattr_destroy(&_attr);
  throw;
}
catch(...) {
 ::pthread_mutexattr_destroy(&_attr);
 WX_THROW(Wx_general_error());
};
//----------------------------------------------------------------------------
//
// Define the Wx__MutexAttr objects for WxMutex in constructor.
//
static const Wx__MutexAttr MutexAttr_Fast(WxMutex::Fast);
static const Wx__MutexAttr MutexAttr_ErrorCheck(WxMutex::ErrorCheck);
static const Wx__MutexAttr MutexAttr_Recursive(WxMutex::Recursive);

const char WxMutex::class_name[]="WxMutex";

//
// Construct the mutex
//
WxMutex::WxMutex(MutexType t) throw(std::exception)
try : _sig(WX_NONE_ID)
{
 int v;
 switch(t) {
   case Fast: 
        v=::pthread_mutex_init(&_mtx,MutexAttr_Fast.attr());
        if(v!=0) {
          WX_THROW( Wx_bad_errno(v) );     // man. says always 0
        };
        break;
   case ErrorCheck:
        v=::pthread_mutex_init(&_mtx,MutexAttr_ErrorCheck.attr());
        if(v!=0) {
          WX_THROW( Wx_bad_errno(v) );     // man. says always 0
        }
        break;
   case Recursive:
        v=::pthread_mutex_init(&_mtx,MutexAttr_Recursive.attr());
        if(v!=0) {
          WX_THROW( Wx_bad_errno(v) );     // man. says always 0
        }
        break;
   default:
        WX_THROW(Wx_general_error());    // MutexType changed, but code not updated.
 }
 _sig=WXMUTEX_ID;
}
catch(const Wx_except&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
  WX_THROW(Wx_general_error());
};

WxMutex::~WxMutex() throw(std::exception)
try {
 if(_sig!=WXMUTEX_ID) {
   WX_THROW(Wx_general_error());
 }
 const int v=::pthread_mutex_destroy(&_mtx);
 _sig=WX_NONE_ID;
 if(v==0) {
   return;
 }
 if(v==EBUSY) {
   //
   // man. indicates only EBUSY
   //
   // Note: Further access of the object gets SIGSEGV signal or Wx_general_error
   //       Mutex resource may lost
   //
   WX_THROW( Wx_bad_errno(v) );
 }
 
 //
 // Note: Further access of the object gets SIGSEGV signal or Wx_general_error
 //       Mutex resource may lost
 //
 WX_THROW( Wx_bad_errno(v) );
}
catch(const Wx_except&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
  WX_THROW(Wx_general_error());
};

WxRet WxMutex::lock(void) throw(std::exception)
try {
 if(_sig!=WXMUTEX_ID) {
   WX_THROW(Wx_general_error());
 }
 switch(const int v=::pthread_mutex_lock(&_mtx)) {
   case 0:       return(OK);
   case EDEADLK: WX_RETURN(WXM_EDEADLK);
   case EINVAL:  WX_RETURN(WXM_EINVAL);
   default:      WX_THROW( Wx_bad_errno(v) );
 }
}
catch(const Wx_except&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
  WX_THROW(Wx_general_error());
};

WxRet WxMutex::trylock(void) throw(std::exception) 
try {
 if(_sig!=WXMUTEX_ID) {
   WX_THROW(Wx_general_error());
 }
 switch(const int v=::pthread_mutex_trylock(&_mtx)) {
   case 0:      return(OK);
   case EBUSY:  WX_RETURN(WXM_EBUSY);
   case EINVAL: WX_RETURN(WXM_EINVAL);
   default:     WX_THROW( Wx_bad_errno(v) );
 }
}
catch(const Wx_except&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
  WX_THROW(Wx_general_error());
};

WxRet WxMutex::unlock(void) throw(std::exception)
try {
 if(_sig!=WXMUTEX_ID) {
   WX_THROW(Wx_general_error());
 }

 //
 // On my Linux platform:
 //   1. unlock ErrorCheck kind of WxMutex in unlocked state gets EPERM
 //   2. unlock Fast kind of WxMutex in unlocked state gets 0
 // 
 switch(const int v=::pthread_mutex_unlock(&_mtx)) {
   case 0:      return(OK);
   case EPERM:  WX_RETURN(WXM_EPERM);
   case EINVAL: WX_RETURN(WXM_EINVAL);
   default:     WX_THROW( Wx_bad_errno(v) );
 }
}
catch(const Wx_except&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
  WX_THROW(Wx_general_error());
};

void WxMutex::lock(WxMutex *mtx) throw(std::exception,Fault)
try {
 if(mtx==0) {
   WX_THROW( Fault(WXM_EFAULT) );
 }
 WxRet r=mtx->lock();
 if(r!=OK) {
   WX_THROW( Fault(r) );
 }
}
catch(const Fault& e) {
  throw;
}
catch(const Wx_except&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
  WX_THROW(Wx_general_error());
};

void WxMutex::trylock(WxMutex *mtx) throw(std::exception,Fault)
try {
 if(mtx==0) {
   WX_THROW( Fault(WXM_EFAULT) );
 }
 WxRet r=mtx->trylock();
 if(r!=OK) {
   WX_THROW( Fault(r) );
 }
}
catch(const Fault& e) {
  throw;
}
catch(const Wx_except&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
  WX_THROW(Wx_general_error());
};

void WxMutex::unlock(WxMutex *mtx) throw(std::exception,Fault)
try {
 if(mtx==0) {
   WX_THROW( Fault(WXM_EFAULT) );
 }
 WxRet r=mtx->unlock();
 if(r!=OK) {
   WX_THROW( Fault(r) );
 }
}
catch(const Fault& e) {
  throw;
}
catch(const Wx_except&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
  WX_THROW(Wx_general_error());
};

//------------------------------------------------
const char WxLock::class_name[]="WxLock";
WxLock::WxLock(WxMutex &mtx) throw(std::exception,Fault)
try :_pmtx(0) 
{
 WxMutex::lock(&mtx);
 _pmtx=&mtx;
 ::_pthread_cleanup_push(&_buf,
       reinterpret_cast<void(*)(void*)>(WxMutex::s_unlock),  // do not know failure
       static_cast<void *>(_pmtx));
}
catch(const WxMutex::Fault& e) {
  throw WxLock::Fault(e);
}
catch(const Wx_except&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
  WX_THROW(Wx_general_error());
};

WxLock::~WxLock() throw(std::exception)
try {
 //
 // exceptions in the cleanup function seems helpless?
 //
 ::_pthread_cleanup_pop(&_buf,_pmtx!=0); 
 _pmtx=0; 
}
catch(const Wx_except&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
  WX_THROW(Wx_general_error());
};

WxRet WxLock::unlock(void) throw(std::exception)
try {
 if(_pmtx==0) {
   WX_RETURN(WXM_ENOENT);
 }
 WxRet r;
 if((r=_pmtx->unlock())!=OK) {
   WX_RETURN(r);
 }
 _pmtx=0;
 return(OK);
}
catch(const Wx_except&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
  WX_THROW(Wx_general_error());
};
