/* Copyright is licensed under GNU LGPL.                (I.J. Wang, 2003)

  WxDate deals with the calendar notion of year,month,day,hour,minute,second.

*/
#ifndef WXDATE_H__
#define WXDATE_H__

#define WXDATE_VERSION 8

#include "wxtime.h"

/*
  [Avoided C Use] struct tm,
            mktime(...), localtime_r(...);
*/
class WxDate {
  public:
    static const char class_name[];
    WX_DECLARE_FAULT;  // declaraion of Fault

    WxDate() throw()
       : _year(1970),_month(1),_mday(1),_hour(0),_min(0),_sec(0),
         _wday(4),_yday(0), _wxt(0,0) {};

    WxDate(const WxDate &src) throw()
              { _assign(src); };

    WxDate(const WxTime &tm) throw(std::exception,Fault);

    WxDate(int year,int month,int mday, int hour,int min, int sec) throw(std::exception,Fault);

    ~WxDate() throw() {};
 
    bool is_default(void) const throw() 
              { return _wxt.is_default(); };
    
    WxRet reset(void) throw(std::exception);

    int year(void) const throw() 
              { return _year; };

    //
    // [Syn] Get the month of a year of the object
    // 
    // [Ret] Month of the object [1-12]
    // 
    int month(void) const throw() 
              { return _month; };

    //
    // [Syn] Get the day of the month
    // 
    // [Ret] Day of the month [1-31]
    // 
    int mday(void) const throw() 
              { return _mday; };

    //
    // [Syn] Get the hour in a day
    // 
    // [Ret] The hour [0-23]
    // 
    int hour(void) const throw() 
              { return _hour; };

    //
    // [Syn] Get the minute in an hour
    // 
    // [Ret] Minute [0-59]
    // 
    int min(void) const throw() 
              { return _min; };

    //
    // [Syn] Get the second in a minute.
    //
    // [Ret] Second [0-61]
    // 
    int sec(void) const throw() 
              { return _sec; };

    //
    // [Syn] Get the day of the week
    // 
    // [Ret] Day of the week [0-6]
    // 
    int wday(void) const throw() 
              { return _wday; };

    //
    // [Syn] Get the day of the year
    // 
    // [Ret] Day of the year [0-365]
    // 
    int yday(void) const throw() 
              { return _yday; };

    //
    // [Syn] Get the ellapsed time of the object since 1970-1-1 due zero o'clock
    // 
    // [Ret] The ellapsed time of the object since 1970-1-1 due zero o'clock.
    //
    WxTime utc_time(void) const throw()
              { return _wxt; };

    //
    // [Syn] Assign date as the parameter WxDate indicate.
    // 
    void assign(const WxDate &rhs) throw()
              { return _assign(rhs); };

    //
    // [Syn] Assign date as parameter WxTime indicats from 1970-1-1 due zero o'clock.
    //
    // Note: Fractions less than one second is truncated to zero.
    //
    // [Exception] Wx_bad_alloc
    //             Wx_general_error
    //
    // [Ret] OK    
    //       WXM_ERANGE result not representable
    //
    // [Refer] std::localtime
    //
    WxRet assign(const WxTime &tm) throw(std::exception);

    //
    // [Syn] Assign the date object as parameters indicate.
    //       year = [A.D. years]
    //       month= [1-12]
    //       mday = [1-31]
    //       hour = [0-23]
    //       min  = [0-59]
    //       sec  = [0-61]
    //
    // Note: Normalization of the parameters are performed.
    // 
    // [Exception] Wx_bad_alloc
    //             Wx_general_error
    //
    // [Ret] OK    
    //       WXM_EINVAL invalid parameter
    //       WXM_ERANGE result not representable
    //
    // [Refer] std::mktime
    //
    WxRet assign(int year,int month,int mday,
                 int hour,int min, int sec) throw(std::exception);

    //
    // [Syn] Add time to the object
    //
    // [Exception] Wx_bad_alloc
    //             Wx_general_error
    //
    // [Ret] OK    
    //       WXM_ERANGE result not representable
    //
    WxRet add(const WxTime &tm) throw(std::exception);
/* not yet
    //
    // [Syn] Add date to the object
    //       (essentially different from adding time, mostly used in one item of parameter
    //        is non-zero, the others are zero)
    //
    // [Exception] Wx_bad_alloc
    //             Wx_general_error
    //
    // [Ret] OK    
    //       WXM_ERANGE result not representable
    //
    WxRet add(int year,int month,int mday, int hour,int min,int sec) throw(std::exception);
*/
    //
    // [Syn] Assign date as the parameter WxDate indicate.
    // 
    // [Ret] const Reference of this object
    //
    const WxDate & operator =(const WxDate &rhs) throw()
              { _assign(rhs); return *this; };

    //
    // [Syn] Assign date as parameter WxTime indicats from 1970-1-1 due zero o'clock.
    //
    // Note: Fractions less than one second is truncated to zero.
    //
    // [Exception] Wx_bad_alloc
    //             Wx_general_error
    //
    // [Exception] Fault
    //       WXM_ERANGE result not representable
    //
    // [Ret] const Reference of this object
    //
    // [Refer] std::localtime, assign
    //
    const WxDate &operator =(const WxTime &rhs) throw(std::exception,Fault);

    //
    // [Syn] Accumulate time of the object by right hand side time rhs
    //
    // [Exception] Wx_bad_alloc
    //             Wx_general_error
    //
    // [Exception] Fault
    //       WXM_ERANGE result not representable
    //
    // [Ret] const reference of this object
    //
    // [Refer] WxTime::operator+ , assign
    //
    const WxDate &operator +=(const WxTime &rhs) throw(std::exception,Fault);

    //
    // [Syn] These are operators for comparisons
    //
    // [Ret] bool result of the comparison
    //
    // [Refer] WxTime::operator ==,!=,<,<=,>,>=
    //
    bool operator ==(const WxDate &rhs) const throw() 
              { return _wxt==rhs._wxt; };
    bool operator !=(const WxDate &rhs) const throw()
              { return _wxt!=rhs._wxt; };
    bool operator > (const WxDate &rhs) const throw()
              { return _wxt> rhs._wxt; };
    bool operator >=(const WxDate &rhs) const throw()
              { return _wxt>=rhs._wxt; };
    bool operator < (const WxDate &rhs) const throw()
              { return _wxt< rhs._wxt; };
    bool operator <=(const WxDate &rhs) const throw()
              { return _wxt<=rhs._wxt; };

  private:
    inline void WxDate::_assign(const WxDate &src) throw()
              {
               if(&src==this) {
                 return;
               }
               _year = src._year;
               _month= src._month;
               _mday = src._mday;
               _hour = src._hour;
               _min  = src._min;
               _sec  = src._sec;
               _wday = src._wday;
               _yday = src._yday;
               _wxt  = src._wxt;
              };

    int _year;                // [A.D. year]
    int _month;               // [1,12]
    int _mday;                // [1,31]
    int _hour;                // [0,23]
    int _min;                 // [0,59]
    int _sec;                 // [0,61]

    // 
    // The following members are redundant, but quite efficient for
    // valid date checking and conversions. And the use of several C-library 
    // time functions could be saved.
    // 
    int _wday;                // [0-6] week day
    int _yday;                // [0,365] Days since January 1st
    WxTime _wxt;              // time since the epoch (1970-1-1 due zero o'clock)
};

//
// [Syn] Add WxDate with WxTime
//
// Note: Fractions less than one second is truncated to zero.
//
// [Exception] Wx_bad_alloc
//             Wx_general_error
//
// [Exception] WxDate::Fault
//       WXM_ERANGE result not representable
//
// [Ret] WxDate of dt+tm (or tm+dt)
//
// [Refer] std::localtime
//
WxDate operator +(const WxDate& dt,const WxTime& tm) throw(std::exception,WxDate::Fault);
WxDate operator +(const WxTime& tm,const WxDate& dt) throw(std::exception,WxDate::Fault);

namespace Wx {
 WxRet set_systime(const WxDate& t) throw(std::exception);
 WxStr what_is(const WxDate&) throw(std::exception);
};

#endif
