/* Copyright is licensed under GNU LGPL.                (I.J. Wang, 2003)

   This file should conform to the documentation in file wxbytetape.3wx
*/

#ifndef WXBYTETAPE_H__
#define WXBYTETAPE_H__
#define WXBYTETAPE_VERSION 8

#include "wx__handle.h"
#include "wxfilehandle.h"
#include "wxfilestat.h"
#include "wxpathname.h"

/*
  Note: This class does not have public assignment operation.

  [Avoided C Use] lseek, mkstemp

  Note: Some of the error messages are based on Linux Programmer's Manual.
*/
class WxByteTape {
  public:
    static const char class_name[];
    WX_DECLARE_FAULT;

    WxByteTape() throw() 
              : _pos(DefaultPos),_hdl() {};

    WxByteTape(const WxByteTape &src) throw(std::exception,WxRet);

    // No constructor from WxFileHandle, for not quite sensible

    virtual ~WxByteTape() throw(std::exception);

    bool is_default(void) const throw()
              { return _hdl.is_default(); };

    inline WxFileHandle fh(void) const throw()
              { return WxFileHandle(_hdl.fd()); };

    WxRet stat(WxFileStat& filestat) const throw(std::exception);

    virtual WxRet reset(void) throw(std::exception)
              { _pos=DefaultPos; return _hdl.reset(); };

    virtual WxRet read(void *buf, size_t count, size_t *n_read) throw(std::exception);

    virtual WxRet max_read(void* buf,size_t count,size_t* n_read) throw(std::exception);
    
    virtual WxRet write(const void *buf, size_t count, size_t *n_written) throw(std::exception);

    virtual WxRet drain(void) throw(std::exception)
              { return _hdl.fdatasync(); };

    off_t get_pos(void) const throw() 
              { return _pos; };
	    
    virtual void set_pos(off_t ofst) throw() 
              { _pos=ofst; };

    virtual WxRet seek_pos_end(void) throw(std::exception);
    
    virtual WxRet read_line(void *buf, size_t count, size_t *n_read) throw(std::exception);

    virtual WxRet read_line(WxStr &buf,size_t* n_read) throw(std::exception);

    virtual WxRet resize(off_t length) throw(std::exception)
              { return _hdl.truncate(length); };

    //
    // [Internal] Set file descriptor of this object to exactly the open fh
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    // 
    // Note: r/w position is not changed, reffer to wx_assign
    // 
    // [Ret] OK     Succeed
    //       WXM_EINVAL fd is not an open file descriptor
    //       WXM_NDEFAULT object is not default
    // 
    // [Refer] reset()
    //
    inline WxRet wx_set_fh(WxFileHandle fh) throw(std::exception)
              { return _hdl.set_fd(fh.fd()); };

  protected:
    //
    // [Syn] Construct object from the open fd
    //
    // Note: This is not a direct copy of fd but a duplicate by dup()
    // Note: r/w position is constructed with default value.
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    // 
    // [Exception] Fault
    //       WXM_EINVAL fd is not an open fd
    //       WXM_EBADF  fd is not valid for the operation
    //       WXM_EMFILE too many open fd
    //
    // Note: errmsg is converted from the errno ::dup returned
    //
    // [Refer] ::dup(int)
    //
    WxByteTape(WxFileHandle fh) throw(std::exception,WxRet);

    //
    // [Syn] Open by the given pathname and set the object to the fd.
    // 
    //      .flags= [O_RDONLY |O_WRONLY |O_RDWR]
    //              || [O_CREAT |O_EXECL |O_NOCTTY ...] the same as ::open(...)
    //       mode= specifies permissions, the same as ::open(...)
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    // 
    // [Ret] OK    
    //       WXM_NDEFAULT object is not default
    //       WXM_EACCES search permission is denied
    //       WXM_EEXIST pathname already exists
    //       WXM_EINTR  interrupted by a signal
    //       WXM_EISDIR object refers to a directory
    //       WXM_EMFILE too many open fd
    //       WXM_ENAMETOOLONG pathname too long
    //       WXM_ENFILE file table overflow
    //       WXM_ENOENT pathname does not exist
    //       WXM_ENOSPC device left no space for the operation
    //       WXM_ENOTDIR a component in pathname is not a directory
    //       WXM_ENXIO  no such device (device not ready)
    //       WXM_EROFS read-only file system
    //       WXM_ENODEV no such device
    //       WXM_ETXTBSY Text file busy (see man.)
    //       WXM_ELOOP too many symbolic links encountered
    //       WXM_ENOMEM not enough memory
    //
    // Note: errmsg is converted from the errno ::open returned
    //
    // [Refer] ::open(const char*,int,mode_t)
    //
    WxRet wx_open(const WxPathName& pathname,int f) throw(std::exception);
    WxRet wx_open(const WxPathName& pathname,int f,mode_t m) throw(std::exception);

    //
    // [Syn] Open object from the unique pathname derived from the given name_temp
    //       and appended with system generated (six) characters.
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    //
    // [Ret] OK         succeed, r/w position is set to zero. mode=0600
    //       WXM_NDEFAULT object is not default
    //       WXM_ENOMEM not enough memory
    //       WXM_EEXIST failed to create an unique temporary filename 
    //                  ($name_temp may be modified)
    //
    // Note: errmsg is converted from the errno ::mkstemp returned
    //
    // [Refer] ::mkstemp(...)
    //
    WxRet wx_open_tmpfile(WxPathName& name_temp) throw(std::exception);

    //
    // [Cancel Point]
    //
    // [Syn] Assign object by src (duplicate)
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    // 
    // [Ret] OK         object can be used interchangeably with src
    //       WXM_NDEFAULT object is not default
    //       WXM_EBADF  src is not valid for the operation
    //       WXM_EINTR  interrupted by a signal
    //       WXM_EMFILE too many open fd
    //
    // Note: errmsg is converted from the errno ::dup returned
    //
    // [Refer] ::dup(int)
    //
    WxRet wx_assignf(const WxByteTape& src) throw(std::exception);

    //
    // [Cancel Point]
    //
    // [Syn] Assign fh of this object by (duplicate) fh
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    // 
    // [Ret] OK         object can be used interchangeably with src
    //       WXM_NDEFAULT object is not default
    //       WXM_EBADF  src is not valid for the operation
    //       WXM_EINTR  interrupted by a signal
    //       WXM_EMFILE too many open fd
    //
    // Note: errmsg is converted from the errno ::dup returned
    //
    // [Refer] ::dup(int)
    //
    WxRet wx_assignf(WxFileHandle fh) throw(std::exception);

    //
    // cmd= F_GETFL: Get file descriptor flags as set by open()
    //      F_GETFD: Get the close-on-exec flag.
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    // 
    // [Ret] OK     *ret_f is set with descriptor flag(s)
    //       WXM_EACCES  lock held
    //       WXM_EAGAIN  file has been memory-mapped (see man.)
    //       WXM_EBADF  object is not valid for the operation
    //       WXM_EDEADLK dead lock would result
    //       WXM_EINTR  interrupted by a signal
    //       WXM_EINVAL object is not suitable
    //       WXM_EMFILE too many open fd
    //       WXM_ENOLCK no locks available
    //       WXM_EFAULT lock is not accessible
    //       WXM_EPERM  operation not permitted (see man.)
    //       
    // Note: if ref_f is zero, result is not stored.
    // Note: errmsg is converted from the errno ::fcntl returned
    //
    WxRet wx_fcntl(int cmd,int *ret_f) const throw(std::exception)
              { return _hdl.fcntl(cmd,ret_f); };

    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    // 
    // cmd= F_DUPFD Duplicate a file descriptor.
    //              .The call:  fcntl(fd,F_DUPFD,minfd);
    //               returns a new file descriptor with a numberical value 
    //               equal or greater than minfd.
    //              .The new file descriptor shares any locks with the original
    //                file. They can be used interchangebly.
    //              .The close-on-exec flag for the new descriptor is clear,
    //               meaning that it will not be closed on exec. They do not
    //               share the close-on-exec flag.
    //
    //              [Ret] OK     *rev_f contains duplicated file descriptor
    //                    ...... errmsg is converted from the errno ::fcntl returned
    //
    //      F_SETFD Set the close-on-exec flag. The following example sets the
    //              close-on-exec bit. All other bits are preserved.
    //                  flags=fcntl(fd,F_GETFD);
    //                  fcntl(fd,F_SETFD,flags| FD_CLOEXEC);
    //              The following example clears the close_on_exec bit. All
    //              other birs are preserved.
    //                  flags=fcntl(fd,F_GETFD);
    //                  fcntl(fd,F_SETFD,flags& ~FD_CLOEXEC);
    //              .F_SETFD applies only to a single file descriptor.
    //
    //              [Ret] OK    
    //                    ...... errmsg is converted from the errno ::fcntl returned
    //
    //      F_SETFL Two of the flags can be set in the open() call may be
    //              modified by
    //              the fcntl() function. The O_APPEND and O_NONBLOCK flags may
    //              be changed while the file is open. for example:
    //                   flags=fcntl(fd, F_GETFL);
    //                   flags|= O_NONBLOCK;
    //                   flags&=~(O_APPEND);
    //                   fcntl(fd,F_SETFL,flags);
    //              .F_SETFL applies to all file descriptors that share a common
    //               open file descriptor, either by inheritance through fork()
    //               or as the result of an F_DUPFD operation with fcntl();
    //               for example:
    //                   fd1= open(path, oflags);
    //                   fd2= dup(fd1);
    //                   fd3= open(path, oflags);
    //               .An F_SETFD on fd1 applies only to fd1. An F_SETFL on fd1
    //                 applies to fd1 and fd2 but not to fd3.
    //
    //              [Ret] OK    
    //                    ...... errmsg is converted from the errno ::fcntl returned
    //
    // Note: if ref_f is zero, result is not stored.
    //
    WxRet wx_fcntl(int cmd, long arg, int *ret_f) const throw(std::exception)
              { return _hdl.fcntl(cmd,arg,ret_f); };

    //
    // Locking region
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    // 
    // cmd= F_GETLK Search for a lock that would block the one in the struct
    //       flock
    //              [Ret] OK     The flock that will prevent from obtain the
    //                           lock, or l_type was set with F_UNLCK if no lock
    //                           found that would obstruct obtaining the lock.
    //                    ...... errmsg is converted from the errno ::fcntl returned
    //
    //      F_SETLK Set or clear(l_type=F_UNLCK) locks for a record 
    //
    //               [Ret] OK     region locked
    //                    ...... errmsg is converted from the errno ::fcntl returned
    //
    //      F_SETLKW [Cancel Point] identical to F_SETLK except would wait for
    //                lock to be released if the record is currently locked.
    //
    //               [Ret] OK         region locked
    //                    ...... errmsg is converted from the errno ::fcntl returned
    //
    // Note: if ref_f is zero, result is not stored.
    //
    WxRet wx_fcntl(int cmd, struct flock *lock,int *ret_f) const throw(std::exception)
              { return _hdl.fcntl(cmd,lock,ret_f); };

  private:
    //
    // [Cancel Point]
    //
    // [Syn] Seek the virtual head to offset ofst, save result to *res.
    //
    // Note: r/w position (_pos) is not changed.
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    //
    // [Ret] OK    
    //       WXM_EBADF  object is not valid for the operation
    //       WXM_ESPIPE Illegal seek (pipe,socket or FIFO)
    // 
    // Note: errmsg converted from the errno ::lseek(...) returned
    // 
    // [Refer] ::lseek(int,off_t,int)
    //
    WxRet seek_set(off_t ofst, off_t *res) const throw(std::exception);

    //
    // [Cancel Point]
    //
    // [Syn] Seek the virtual head to the current value + ofst, save result
    //       to *res.
    //
    // Note: r/w position (_pos) is not changed.
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    //
    // [Ret] OK    
    //       WXM_EBADF  object is not valid for the operation
    //       WXM_ESPIPE Illegal seek (pipe,socket or FIFO)
    // 
    // Note: errmsg converted from the errno ::lseek(...) returned
    // 
    // [Refer] ::lseek(int,off_t,int)
    //
    WxRet seek_cur(off_t ofst, off_t *res) const throw(std::exception);

    //
    // [Cancel Point]
    //
    // [Syn] Seek the virtual head to end offset + ofst, save result to
    //       *res.
    //
    //       end offset::= offset of the farest written byte + 1
    //       
    // Note: r/w position (_pos) is not changed.
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    //
    // [Ret] OK    
    //       WXM_EBADF  object is not valid for the operation
    //       WXM_ESPIPE Illegal seek (pipe,socket or FIFO)
    // 
    // Note: errmsg converted from the errno ::lseek(...) returned
    // 
    // [Refer] ::lseek(int,off_t,int)
    //
    WxRet seek_end(off_t ofst, off_t *res) const throw(std::exception);

  private:
    static const char DefaultEOL='\n';
    static const off_t DefaultPos=0;
    off_t _pos;
    Wx__Handle _hdl;

    const WxByteTape & operator =(const WxByteTape &); // not to use
    bool operator==(const WxByteTape&) const throw();  // not to use
    WxRet close(void); // not to use, see Wx__Handle::close(), yet to define
};

#endif
