/* Copyright is licensed under GNU LGPL.                (I.J. Wang, 2003)
*/

#include "wxbytetape.h"
#include <cstring>
#include <memory>

const char WxByteTape::class_name[]="WxByteTape";

WxRet WxByteTape::seek_set(off_t ofst, off_t *res) const throw(std::exception)
try {
 Wx__Base::cancel_point();
 const off_t v=::lseek(_hdl.fd(),ofst,SEEK_SET);
 Wx__Base::cancel_point();

 if(v!=(off_t)-1) {
   if(res!=0) {
     *res=v;
   }
   return(OK);
 }
 WX_RETURN(WxErrMsg(errno));
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet WxByteTape::seek_cur(off_t ofst, off_t *res) const throw(std::exception)
try {
 Wx__Base::cancel_point();
 off_t v=::lseek(_hdl.fd(),ofst,SEEK_CUR);
 Wx__Base::cancel_point();

 if(v!=(off_t)-1) {
   if(res!=0) {
     *res=v;
   }
   return(OK);
 }
 WX_RETURN(WxErrMsg(errno));
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet WxByteTape::seek_end(off_t ofst, off_t *res) const throw(std::exception)
try {
 Wx__Base::cancel_point();
 const off_t v=::lseek(_hdl.fd(),ofst,SEEK_END);
 Wx__Base::cancel_point();

 if(v!=(off_t)-1) {
   if(res!=0) {
     *res=v;
   }
   return(OK);
 }
 WX_RETURN(WxErrMsg(errno));
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

//-----------------------------------------------------------------------------
WxByteTape::WxByteTape(WxFileHandle fh) throw(std::exception,WxRet)
try : _pos(DefaultPos),_hdl(fh.fd()) {
}
catch(const WxByteTape::Fault& e) {
  throw;
}
catch(const Wx__Handle::Fault& e) {
  WX_THROW( Fault(e) );
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxByteTape::WxByteTape(const WxByteTape &src) throw(std::exception,WxRet)
try : _pos(src._pos),_hdl(src._hdl) {
}
catch(const Wx__Handle::Fault& e) {
  WX_THROW( Fault(e) );
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxByteTape::~WxByteTape() throw(std::exception)
try {
  _hdl.reset();
}
catch(const Wx__Handle::Fault& e) {
  WX_THROW( Fault(e) );
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet WxByteTape::stat(WxFileStat& filestat) const throw(std::exception)
try {
 //
 // Assert that _hdl.fstat failure should not modify filestat.
 //
 WX_RETURN( _hdl.fstat(filestat.wx_stt()) );
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet WxByteTape::wx_open(const WxPathName& pathname,int f) throw(std::exception)
try {
 const WxRet r=_hdl.open(pathname.pathname().c_str(),f);
 if(r!=OK) {
   WX_RETURN(r);
 }
 _pos=DefaultPos;
 return(OK);
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet WxByteTape::wx_open(const WxPathName& pathname,int f,mode_t m) throw(std::exception)
try {
 const WxRet r=_hdl.open(pathname.pathname().c_str(),f,m);
 if(r!=OK) {
   WX_RETURN(r);
 }
 _pos=DefaultPos;
 return(OK);
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet WxByteTape::wx_open_tmpfile(WxPathName& tmp_name) throw(std::exception)
try {
 if(_hdl.is_default()==false) {
   WX_RETURN(WXM_NDEFAULT);
 }

 {
  const size_t nsiz=tmp_name.pathname().size();
  std::auto_ptr<char> tnam(new(std::nothrow) char[nsiz+7]); 
  if(tnam.get()==0) {
    WX_RETURN(WXM_ENOMEM);
  }
  std::memcpy(tnam.get(),tmp_name.pathname().c_str(),nsiz);
  std::memset(tnam.get()+nsiz,'X',6);
  tnam.get()[nsiz+6]=0;
  const int v=::mkstemp(tnam.get());
  if(v>=0) {
    try {
      tmp_name=tnam.get();     // set argument with system returned name
    }
    catch(...) {
      ::close(v);
      WX_RETURN(WXM_ENOMEM);   // hope only ENOMEM possible
    }

    try {
      if(_hdl.set_fd(v)!=OK) {
        WX_THROW(Wx_general_error());
      }
    }
    catch(...) {
      ::close(v);
      throw;
    }
    _pos=DefaultPos;
    return(OK);
  }
 }
 WX_RETURN(WxErrMsg(errno));
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet WxByteTape::wx_assignf(const WxByteTape& src) throw(std::exception)
try {
 const WxRet r=_hdl.assign(src._hdl);
 if(r!=OK) {
   WX_RETURN(r);
 }
 _pos=src._pos;
 return(OK);
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet WxByteTape::wx_assignf(WxFileHandle fh) throw(std::exception)
try {
 if(_hdl.is_default()==false) {
   WX_RETURN(WXM_NDEFAULT);
 }
 if(fh.is_default()) {
   return(OK);
 }

 const WxRet r=_hdl.assign(fh.fd());
 if(r!=OK) {
   WX_RETURN(r);
 }
 return(OK);
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet WxByteTape::seek_pos_end(void) throw(std::exception)
try {
 off_t res;
 const WxRet r=seek_end(0, &res);
 if(r!=OK) {
   WX_RETURN(r);
 }
 _pos=res;
 return(OK);
}
catch(const WxByteTape::Fault&) {
  throw;
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet WxByteTape::read(void *buf, size_t count, size_t *n_read) throw(std::exception)
try {
 if(n_read!=0) {
   *n_read=0;
 }
 size_t n_rd;
 const WxRet r=_hdl.pread(buf,count,&n_rd,_pos);
 if(r!=OK) {
   WX_RETURN(r);
 }
 _pos+=n_rd;
 if(n_read!=0) {
   *n_read=n_rd;
 }
 return(OK);
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet WxByteTape::max_read(void* buf,size_t count,size_t* n_read) throw(std::exception)
try {
 if(n_read!=0) {
   *n_read=0;
 }
 size_t n_rd;
 const WxRet r=_hdl.max_pread(buf,count,&n_rd,_pos);
 if(r!=OK) {
   WX_RETURN(r);
 }
 _pos+=n_rd;        // failure not modify _pos
 if(n_read!=0) {
   *n_read=n_rd;
 }
 return(OK);
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet WxByteTape::write(const void *buf, size_t count, size_t *n_written) throw(std::exception)
try {
 if(n_written!=0) {
   *n_written=0;
 }
 size_t n_wr;
 WxRet r=_hdl.pwrite(buf,count,&n_wr,_pos);
 if(r!=OK) {
   WX_RETURN(r);
 }
 _pos+=n_wr;       // failure not modify _pos
 if(n_written!=0) {
   *n_written=n_wr;
 }
 return(OK);
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet WxByteTape::read_line(void *buf, size_t count, size_t *n_read) throw(std::exception)
try {
 char* const bptr=static_cast<char* const>(buf);
 //
 // set *n_read to 0 (inital setting)
 //
 if(n_read!=0) { 
   *n_read=0;
 }
 
 if(buf==0) {
   if(count==0) {
     return(OK);
   }
   WX_RETURN(WXM_EFAULT);
 }

 if(count<=1) {
   if(count!=0) {
     bptr[0]=0;
   }
   return(OK);
 }

 //
 // Read count-1 bytes data (buf can store)
 //
 WxRet r;
 size_t b_rd;

 r=_hdl.max_pread(buf,count-1,&b_rd,_pos);
 if(r!=OK) {
   WX_RETURN(r);
 }
 if(b_rd==0) {
   bptr[0]=0;
   return(OK);   // EOF
 }
 
 //
 // Scan for EOL and move back r/w position
 //
 try {
  char *sptr=static_cast<char *>(std::memchr(buf,DefaultEOL,b_rd));
  if(sptr==NULL) {
     bptr[b_rd]=0;
     if(n_read!=0) { 
       *n_read=b_rd;
     }
     _pos+=b_rd;
     return(OK);      // no LF found
  }

  ++sptr;
  *sptr=0;            // write the trailing zero
  b_rd=sptr-bptr;     // calculate the byte until LF (including)
 }
 catch(...) {
   //
   // Set unread anything
   //
   if(n_read!=0) { 
     *n_read=0;
   }
   bptr[0]=0;
   throw;
 }

 if(n_read!=0) { 
   *n_read=b_rd;
 }
 _pos+=b_rd;  // set up position the next r/w begins
 return(OK);
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet WxByteTape::read_line(WxStr &buf,size_t* n_read) throw(std::exception)
try {
 off_t org_pos=_pos;
 size_t org_siz=buf.size();

 if(n_read!=0) { 
   *n_read=0;
 }
 try {
   for(;;) {
     size_t n_rd;
     char lnbuf[128];
     WxRet r;
     if((r=read_line(lnbuf,sizeof(lnbuf),&n_rd))!=OK) {
       WX_RETURN(r);
     }
     if(n_rd==0) {
       break;
     }
     if(n_read!=0) {
       *n_read+=n_rd;
     }
     if((r=buf.append(lnbuf,n_rd))!=OK) {
       WX_RETURN(r);
     }
     if(lnbuf[n_rd-1]==DefaultEOL) {
       break;
     }
   }
 }
 catch(...) {
   //
   // Set unread anything
   //
   _pos=org_pos;
   if(n_read!=0) {
     *n_read=0;
   }
   buf.resize(org_siz); // shrink back to the original size. original data in buf 
                        // should not have been modified and no errors
   throw;
 }
 return(OK);
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};
