/* Copyright is licensed under GNU LGPL.                (I.J. Wang, 2003)

  Wx__Handle provides functions to handle the file descriptor.

  Note: Some of the error messages are based on Linux Programmer's Manual.
*/

#ifndef WX__HANDLE_H__
#define WX__HANDLE_H__
#define WX__HANDLE_VERSION 8

#include "wxret.h"
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include "wxstr.h"

/*
  [Internal] Wx__Handle wraps C functions that use file descriptor.
             The purpose is to group them and provide interface, not all
             applicable for any given specific file descriptor.

  [Avoided C Use] open,close,dup,read,write,pread,pwrite,fstat,fcntl,fdatasync,
                  ftruncate
*/
class Wx__Handle {
  public:
    static const char class_name[];
    WX_DECLARE_FAULT;  // declaraion of Fault

    //
    // [Syn] Construct default object (refer to nothing)
    //
    Wx__Handle() throw()
         : _fd(DefaultFd) {};

    //
    // [Syn] Construct object from the open fd
    //
    // Note: This is not a direct copy of fd but a duplicate by dup()
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    // 
    // [Exception] Fault
    //       WXM_EBADF  fd is not valid for the operation
    //       WXM_EMFILE too many open fd
    //
    // Note: errmsg is converted from the errno ::dup returned
    //
    // [Refer] ::dup(int)
    //
    Wx__Handle(Wx__TypeFD fd) throw(std::exception,Fault);

    //
    // [Syn] Construct object by copy from src
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    // 
    // [Exception] Fault
    //       WXM_EBADF  src is not valid for the operation
    //       WXM_EMFILE too many open fd
    //
    // Note: errmsg is converted from the errno ::dup returned
    //
    // [Refer] ::dup(int)
    //
    Wx__Handle(const Wx__Handle &src) throw(std::exception,Fault);

    //
    // [Syn] Destroy the object. Existing fd is reset.
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    //
    // [Exception] Fault
    //       WXM_EIO    low level I/O error (object default)
    // 
    // Note: errmsg is converted from the errno ::close returned
    //
    // [Refer] reset
    //
    ~Wx__Handle() throw(std::exception,Fault);

    //
    // [Syn] is object the state as default constructed
    //
    // [Ret] true  = object is the same as default constructed, 
    //       false = otherwise
    //
    bool is_default(void) const throw() 
              { return _fd==DefaultFd; };

    //
    // [Cancel Point]
    //
    // [Syn] Read maximum count bytes of data into memory pointed by buf.
    //       The number of bytes read and written to buf is stored 
    //       into *n_read if it is not zero, otherwise ignored.
    //       Zero byte read and written means end of file.
    // 
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    // 
    // [Ret] OK         *n_read stores the number of bytes read and written into buf.
    //       WXM_EINTR  interrupted by a signal before any data read
    //       WXM_EAGAIN non-blocking I/O (no data read)
    //       WXM_EIO    low level I/O error
    //       WXM_EISDIR object refers to a directory
    //       WXM_EBADF  object is not valid for the operation
    //       WXM_EINVAL object is not suitable for reading
    //       WXM_EFAULT buf is not accessible
    //       WXM_E2BIG  count too great (internal conversion overflow)
    //
    // Note: errmsg is converted from the errno ::read(int,void*,size_t) returned
    //       *n_read stores the number of bytes read and written into buf.
    //
    // Note: The exact behavior of operation depends on the refered device.
    //
    // [Refer] ::read(int,void*,size_t)
    //         POSIX Programmer's Guide,Donald Lewine,p89
    //
    WxRet read(void *buf, size_t count, size_t *n_read) const throw(std::exception);

    //
    // [Cancel Point] 
    //
    // [Syn] Write count bytes of data in buf to the device.
    //       The number of written bytes is stored into *n_written if not zero, 
    //       otherwise ignored.
    // 
    // Note: .If count is zero and object refers to a regular file, this function
    //        returns zero and has no other effect. If count is zero and the device 
    //        is not regular file or is special, the result is not portable.
    //       .If count is greater than SSIZWXM_EMAX, the result is unspecified.
    //       .Refer to ::write(int,const void*,size_t)
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    // 
    // [Ret] OK         *n_written stores the bytes written (if it is not zero).
    //       WXM_E2BIG  count too great (internal conversion overflow)
    //       WXM_EBADF  object is not valid or not open for writting
    //       WXM_EINVAL object attatched device is not suitable for writing
    //       WXM_EFAULT buf is not accessible
    //       WXM_EFBIG  attempt to write beyond the data size limit by the device
    //       WXM_EPIPE  attempt to write to a pipe or FIFO with no reader
    //       WXM_EAGAIN non-blocking I/O (write would block)
    //       WXM_EINTR  interrupted by a signal before any data written
    //       WXM_ENOSPC device left no space for the operation
    //       WXM_EIO    low level I/O error
    //
    // Note: errmsg is converted from the errno ::write(int,void*,size_t) returned
    //       *n_written stores the bytes written (if it is not zero).
    //
    // Note: the exact behavior of operation depends on the refered derive.
    //
    WxRet write(const void *buf, size_t count, size_t *n_written) const throw(std::exception);

    //
    // [Cancel Point]
    //
    // [Syn] Read maximum count bytes of data into memory pointed by buf.
    //       The number of bytes read and written to buf is stored 
    //       into *n_read if it is not zero, otherwise ignored.
    //       Zero byte read and written means end of file.
    //
    // Note: If there is currently not enough data to satisfy the amount requested
    //       (count), this function would block until satisfied or end of file. 
    //       This function is based on read(...), and provided for convenience.
    //       As many whiles, max_read(...) is meant to use.
    //
    // Warning: Device is not locked in the possible iterative read operations.
    //          This function is not thread safe.
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    // 
    // [Ret] OK         *n_read always stored with value identical to count.
    //       WXM_E2BIG  count too great (internal conversion overflow)
    //       WXM_EINTR  interrupted by a signal
    //       WXM_EAGAIN non-blocking I/O (no data read)
    //       WXM_EIO    low level I/O error
    //       WXM_EISDIR object refers to a directory
    //       WXM_EBADF  object is not valid for the operation
    //       WXM_EINVAL object is not suitable for reading
    //       WXM_EFAULT buf is not accessible
    //
    // Note: errmsg is converted from the errno ::read(int,void*,size_t) returned
    //       *n_read stores the number of bytes read and written into buf.
    //
    // Note: The exact behavior of operation depends on the refered device.
    //
    // [Refer] ::read(int,void*,size_t), Wx__Handle::read(...)
    //         POSIX Programmer's Guide,Donald Lewine,p89
    //
    WxRet max_read(void* buf,size_t count,size_t* n_read) const throw(std::exception);

    //
    // [Cancel Point]
    //
    // [Syn] Read maximum count bytes of data into memory pointed by buf.
    //       The number of bytes read and written to buf is stored 
    //       into *n_read if it is not zero, otherwise ignored.
    //       Zero byte read and written means end of file.
    // 
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    // 
    // [Ret] OK         *n_read stores the number of bytes read and written into buf.
    //       WXM_EINTR  interrupted by a signal before any data read
    //       WXM_EAGAIN non-blocking I/O (no data read)
    //       WXM_EIO    low level I/O error
    //       WXM_EISDIR object refers to a directory
    //       WXM_EBADF  object is not valid for the operation
    //       WXM_EINVAL object is not suitable for reading
    //       WXM_EFAULT buf is not accessible
    //       WXM_E2BIG  count too great (internal conversion overflow)
    //       WXM_ESPIPE Illegal seek (pipe,socket or FIFO)
    //
    // Note: errmsg is converted from the errno ::read(int,void*,size_t) returned
    //       *n_read stores the number of bytes read and written into buf.
    //
    // Note: The exact behavior of operation depends on the refered device.
    //
    // [Refer] ::read(int,void*,size_t)
    //         POSIX Programmer's Guide,Donald Lewine,p89
    //
    WxRet pread(void *buf, size_t count, size_t *n_read,off_t pos) const throw(std::exception);

    //
    // [Cancel Point] 
    //
    // [Syn] Write count bytes of data in buf to the device.
    //       The number of written bytes is stored into *n_written if not zero, 
    //       otherwise ignored.
    // 
    // Note: .If count is zero and object refers to a regular file, this function
    //        returns zero and has no other effect. If count is zero and the device 
    //        is not regular file or is special, the result is not portable.
    //       .If count is greater than SSIZWXM_EMAX, the result is unspecified.
    //       .Refer to ::write(int,const void*,size_t)
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    // 
    // [Ret] OK         *n_written stores the bytes written (if it is not zero).
    //       WXM_E2BIG  count too great (internal conversion overflow)
    //       WXM_EBADF  object is not valid or not open for writting
    //       WXM_EINVAL object attatched device is not suitable for writing
    //       WXM_EFAULT buf is not accessible
    //       WXM_EFBIG  attempt to write beyond the data size limit by the device
    //       WXM_EPIPE  attempt to write to a pipe or FIFO with no reader
    //       WXM_EAGAIN non-blocking I/O (write would block)
    //       WXM_EINTR  interrupted by a signal before any data written
    //       WXM_ENOSPC device left no space for the operation
    //       WXM_EIO    low level I/O error
    //       WXM_ESPIPE Illegal seek (pipe,socket or FIFO)
    //
    // Note: errmsg is converted from the errno ::write(int,void*,size_t) returned
    //       *n_written stores the bytes written (if it is not zero).
    //
    // Note: the exact behavior of operation depends on the refered derive.
    //
    WxRet pwrite(const void *buf, size_t count, size_t *n_written,off_t pos) const throw(std::exception);

    //
    // [Cancel Point]
    //
    // [Syn] Read maximum count bytes of data into memory pointed by buf.
    //       The number of bytes read and written to buf is stored 
    //       into *n_read if it is not zero, otherwise ignored.
    //       Zero byte read and written means end of file.
    //
    // Note: If there is currently not enough data to satisfy the amount requested
    //       (count), this function would block until satisfied or end of file. 
    //       This function is based on read(...), and provided for conveience.
    //       As many whiles, max_pread(...) is meant to use.
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    // 
    // [Ret] OK         *n_read always stored with value identical to count.
    //       WXM_E2BIG  count too great (internal conversion overflow)
    //       WXM_EINTR  interrupted by a signal
    //       WXM_EAGAIN non-blocking I/O (no data read)
    //       WXM_EIO    low level I/O error
    //       WXM_EISDIR object refers to a directory
    //       WXM_EBADF  object is not valid for the operation
    //       WXM_EINVAL object is not suitable for reading
    //       WXM_EFAULT buf is not accessible
    //       WXM_ESPIPE Illegal seek (pipe,socket or FIFO)
    //
    // Note: errmsg is converted from the errno ::read(int,void*,size_t) returned
    //       *n_read stores the number of bytes read and written into buf.
    //
    // Note: The exact behavior of operation depends on the refered device.
    //
    // [Refer] ::read(int,void*,size_t), Wx__Handle::read(...)
    //         POSIX Programmer's Guide,Donald Lewine,p89
    //
    WxRet max_pread(void* buf,size_t count,size_t* n_read,off_t pos) const throw(std::exception);

    //
    // [Syn] Get struct stat of fd and save to buf
    // 
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    // 
    // [Ret] OK    
    //       WXM_EBADF  object is not valid for the operation
    //       ......
    // Note: errmsg is converted from the errno ::fstat returned
    //       
    // [Refer] ::fstat(int,struct stat)
    //
    WxRet fstat(struct stat *buf) const throw(std::exception);

    //
    // cmd= F_GETFL: Get file descriptor flags as set by open()
    //      F_GETFD: Get the close-on-exec flag.
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    // 
    // [Ret] OK     *ret_f is set with descriptor flag(s)
    //       WXM_EACCES  lock held
    //       WXM_EAGAIN  file has been memory-mapped (see man.)
    //       WXM_EBADF  object is not valid for the operation
    //       WXM_EDEADLK dead lock would result
    //       WXM_EINTR  interrupted by a signal
    //       WXM_EINVAL object is not suitable
    //       WXM_EMFILE too many open fd
    //       WXM_ENOLCK no locks available
    //       WXM_EFAULT lock is not accessible
    //       WXM_EPERM  operation not permitted (see man.)
    //       
    // Note: if ref_f is zero, result is not stored.
    // Note: errmsg is converted from the errno ::fcntl returned
    //
    WxRet fcntl(int cmd,int *ret_f) const throw(std::exception);

    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    // 
    // cmd= F_DUPFD Duplicate a file descriptor.
    //              .The call:  fcntl(fd,F_DUPFD,minfd);
    //               returns a new file descriptor with a numberical value 
    //               equal or greater than minfd.
    //              .The new file descriptor shares any locks with the original
    //                file. They can be used interchangebly.
    //              .The close-on-exec flag for the new descriptor is clear,
    //               meaning that it will not be closed on exec. They do not
    //               share the close-on-exec flag.
    //
    //              [Ret] OK     *rev_f contains duplicated file descriptor
    //                    ...... errmsg is converted from the errno ::fcntl returned
    //
    //      F_SETFD Set the close-on-exec flag. The following example sets the
    //              close-on-exec bit. All other bits are preserved.
    //                  flags=fcntl(fd,F_GETFD);
    //                  fcntl(fd,F_SETFD,flags| FD_CLOEXEC);
    //              The following example clears the close_on_exec bit. All
    //              other birs are preserved.
    //                  flags=fcntl(fd,F_GETFD);
    //                  fcntl(fd,F_SETFD,flags& ~FD_CLOEXEC);
    //              .F_SETFD applies only to a single file descriptor.
    //
    //              [Ret] OK    
    //                    ...... errmsg is converted from the errno ::fcntl returned
    //
    //      F_SETFL Two of the flags can be set in the open() call may be
    //              modified by
    //              the fcntl() function. The O_APPEND and O_NONBLOCK flags may
    //              be changed while the file is open. for example:
    //                   flags=fcntl(fd, F_GETFL);
    //                   flags|= O_NONBLOCK;
    //                   flags&=~(O_APPEND);
    //                   fcntl(fd,F_SETFL,flags);
    //              .F_SETFL applies to all file descriptors that share a common
    //               open file descriptor, either by inheritance through fork()
    //               or as the result of an F_DUPFD operation with fcntl();
    //               for example:
    //                   fd1= open(path, oflags);
    //                   fd2= dup(fd1);
    //                   fd3= open(path, oflags);
    //               .An F_SETFD on fd1 applies only to fd1. An F_SETFL on fd1
    //                 applies to fd1 and fd2 but not to fd3.
    //
    //              [Ret] OK    
    //                    ...... errmsg is converted from the errno ::fcntl returned
    //
    // Note: if ref_f is zero, result is not stored.
    //
    WxRet fcntl(int cmd, long arg, int *ret_f) const throw(std::exception);

    //
    // Locking region
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    // 
    // cmd= F_GETLK Search for a lock that would block the one in the struct
    //       flock
    //              [Ret] OK     The flock that will prevent from obtain the
    //                           lock, or l_type was set with F_UNLCK if no lock
    //                           found that would obstruct obtaining the lock.
    //                    ...... errmsg is converted from the errno ::fcntl returned
    //
    //      F_SETLK Set or clear(l_type=F_UNLCK) locks for a record 
    //
    //               [Ret] OK     region locked
    //                    ...... errmsg is converted from the errno ::fcntl returned
    //
    //      F_SETLKW [Cancel Point] identical to F_SETLK except would wait for
    //                lock to be released if the record is currently locked.
    //
    //               [Ret] OK         region locked
    //                    ...... errmsg is converted from the errno ::fcntl returned
    //
    // Note: if ref_f is zero, result is not stored.
    //
    WxRet fcntl(int cmd, struct flock *lock,int *ret_f) const throw(std::exception);

#ifdef _POSIX_SYNCHRONIZED_IO
    //
    // [Syn] Write(flush) data buffered in system memory to the device
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    // 
    // [Ret] OK    
    //       WXM_EBADF  object is not valid for the operation
    //       WXM_EIO    low level I/O error
    //       WXM_EINVAL object is not suitable
    //       WXM_EROFS  read-only file system
    //
    // Note: errmsg is converted from the errno ::fdatasync returned
    //
    // [Refer] ::fdatasync(int))
    //
    WxRet fdatasync(void) const throw(std::exception);
#endif

    //
    // [Cancel Point]
    //
    // [Syn] Open fd by the given pathname and set the object to the fd.
    // 
    //      .flags= [O_RDONLY |O_WRONLY |O_RDWR]
    //              || [O_CREAT |O_EXECL |O_NOCTTY ...] the same as ::open(...)
    //       mode= specifies permissions, the same as ::open(...)
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    // 
    // [Ret] OK    
    //       WXM_NDEFAULT object is not default
    //       WXM_EACCES search permission is denied
    //       WXM_EEXIST pathname already exists
    //       WXM_EINTR  interrupted by a signal
    //       WXM_EISDIR object refers to a directory
    //       WXM_EMFILE too many open fd
    //       WXM_ENAMETOOLONG pathname too long
    //       WXM_ENFILE file table overflow
    //       WXM_ENOENT pathname does not exist
    //       WXM_ENOSPC device left no space for the operation
    //       WXM_ENOTDIR a component in pathname is not a directory
    //       WXM_ENXIO  no such device (device not ready)
    //       WXM_EROFS read-only file system
    //       WXM_ENODEV no such device
    //       WXM_ETXTBSY Text file busy (see man.)
    //       WXM_ELOOP too many symbolic links encountered
    //       WXM_ENOMEM not enough memory
    //
    // Note: errmsg is converted from the errno ::open returned
    //
    // [Refer] ::open(const char*,int,mode_t)
    //
    WxRet open(const char* pathname, int flags) throw(std::exception);
    WxRet open(const char* pathname, int flags, mode_t mode) throw(std::exception);

    //
    // [Cancel Point]
    //
    // [Syn] Close fd of the object. 
    // 
    // Note: Be cautious to use this function
    //       Reffer to POSIX Programmer's Guide,Donald Lewine,p242
    // 
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    // 
    // [Ret] OK     Succeed. The bytes written are flushed. Resources are reclaimed.
    //              The object is made default. Or the object was already default.
    //       WXM_EINTR  interrupted by a signal (fd intact)
    //                  The reffered device may or may not be closed.
    //       WXM_EIO    low level I/O error
    //
    // Note: Object is intact in WXM_EINTR error conditions.
    //
    // Note: errmsg is converted from the errno ::close returned
    //
    // [Refer] ::close(int)
    //
    WxRet close(void) throw(std::exception);

    //
    // [Syn] Reset the object to the state as default constructed.
    //       Existing fd is closed (see close).
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    // 
    // [Ret] OK     Succeed. The bytes written are flushed. Resources are
    //              are reclaimed.
    //       WXM_EIO    low level I/O error (object default)
    // 
    // Note: This function always returns with object set to default
    //       This function will retry in the internal loop for EINTR.
    // 
    // Note: errmsg is converted from the errno ::close returned
    //
    // [Refer] ::close(fd)
    //
    WxRet reset(void) throw(std::exception)
              { return _reset(); };

    //
    // [Cancel Point]
    //
    // [Syn] Assign object by src
    //       
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    // 
    // [Ret] OK         
    //       WXM_NDEFAULT object is not default
    //       WXM_EBADF  src is not valid for the operation
    //       WXM_EINTR  interrupted by a signal
    //       WXM_EMFILE too many open fd
    //
    // Note: errmsg is converted from the errno ::dup returned
    //
    // [Refer] ::dup(int)
    //
    WxRet assign(const Wx__Handle& src) throw(std::exception);
    WxRet assign(Wx__TypeFD src) throw(std::exception);

    //
    // [Syn] Truncate (make) the data length to exactly the size length.
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    // 
    // [Ret] OK    
    //       WXM_EBADF  object is not valid for the operation
    //       WXM_EFBIG  attempt to write beyond the data size limit by the device
    //       WXM_EINTR  interrupted by a signal before any data written
    //       WXM_EINVAL object attatched device is not suitable for writing
    //       WXM_EIO    low level I/O error
    //       WXM_EISDIR object refers to a directory
    //       WXM_EROFS read-only file system
    //       WXM_ETXTBSY Text file busy (see man.)
    //
    // Note: errmsg is converted from the errno ::ftruncate(int,off_t) returned
    //
    // [Refer] ::ftruncate(int,off_t)
    //
    WxRet truncate(off_t length) const throw(std::exception);

    //
    // [Syn] Is the object refer to a terminal
    //
    // [Ret] true= object refers to a terminal
    //      false= otherwise
    //
    // [Refer] ::isatty(int)
    //
    bool isatty(void) const throw()
              { return ::isatty(_fd)!=0; };

    //
    // [Syn] Get the pathname of the terminal device that is open on the file descriptor
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    // 
    // [Ret] OK    Succeed, pathname is stored with pathname the file descriptor is open 
    //             for a terminal
    //       WXM_EBADF  bad file descriptor
    //       WXM_ENOTTY file descriptor does not refer to a terminal device
    //       WXM_ENOMEM out of memory
    //
    // [Refer] ::ttyname_r(int,char*,size_t)
    //
    WxRet ttyname(WxStr& pathname) const throw(std::exception);

    //
    // [Syn] Get file descriptor of this object
    //
    // [Ret] file descriptor of this object
    //
    inline Wx__TypeFD fd(void) const throw() 
              { return _fd; };

    //
    // [Syn] Set file descriptor of this object to exactly the open fd
    //       This function differs from overwrite() in the checking of fd
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    // 
    // [Ret] OK     Succeed
    //       WXM_EINVAL fd is not an open file descriptor
    //       WXM_NDEFAULT object is not default
    // 
    // [Refer] reset()
    //
    WxRet set_fd(Wx__TypeFD fd) throw(std::exception);

    //
    // [Syn] Overwrite file descriptor of this object
    //
    // Waring: Object design is totally corrupted by using these functions
    //
    inline void overwrite() throw()
              { _fd=DefaultFd; };
    inline void overwrite(Wx__TypeFD fd) throw()
              { _fd=fd; };
    inline void overwrite(const Wx__Handle& h) throw()
              { _fd=h._fd; };

    //
    // [Syn] Is the given fd an open file descriptor
    //
    // [Ret] true= fd is an open file descriptor
    //      false= otherwise
    //
    inline static bool wx_is_open_fd(Wx__TypeFD fd) throw()
              { return (::fcntl(fd,F_GETFL)==-1)? false:true; };

  private:

    //
    // [Syn] Reset this object to default state
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    // 
    // [Ret] OK     Succeed. The bytes written are flushed. Resources are
    //              are reclaimed.
    //       WXM_EIO    low level I/O error (object default)
    // 
    // Note: All errmsg returned refer to the closing of previous fd
    // 
    // Note: Object is always default while returned.
    //       EINTR is retried in the internal loop to go away
    // 
    // Note: errmsg is converted from the errno ::close returned
    //
    // [Refer] ::close(int)
    //
    WxRet _reset(void) throw(std::exception);

    //
    // [Cancel Point]
    //
    // [Syn] Duplicate a file descriptor of this object, save to dest.
    //       The returned file descriptor can be used interchangeably.
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    // 
    // [Ret] OK      the duplicated fd is stored to dest
    //       WXM_EINVAL dest is not default
    //       WXM_EBADF  object is not valid for the operation
    //       WXM_EINTR  interrupted by a signal
    //       WXM_EMFILE too many open fd
    //
    // Note: errmsg is converted from the errno ::dup returned
    //
    // [Refer] ::dup(int)
    //
    WxRet _dup(Wx__Handle& dest) const throw(std::exception);

    static const Wx__TypeFD DefaultFd=WX__DEFAULT_FD;
    Wx__TypeFD _fd;

    const Wx__Handle& operator=(const Wx__Handle& rhs) throw(); // not to use
    bool operator==(const Wx__Handle& rhs) const throw(); // not to use
    bool operator!=(const Wx__Handle& rhs) const throw(); // not to use
};

namespace Wx {
 //
 // [Syn] Get descriptive string of this object.
 //
 // [Exception] Wx_general_error
 //             Wx_bad_alloc
 //
 // [Ret] The descriptive string of this object.
 //       e.g. "13" decimal number of the descriptor
 //
 extern WxStr what_is(const Wx__Handle& hdl) throw(std::exception);
}
#endif
