/* Copyright is licensed under GNU LGPL.                (I.J. Wang, 2003)
*/
#include "wx__handle.h"
#include <limits>
#include <pthread.h>
#include "wxnum.h"

const char Wx__Handle::class_name[]="Wx__Handle";

Wx__Handle::Wx__Handle(Wx__TypeFD fd) throw(std::exception,Fault)
try {
 if(fd==DefaultFd) {
   _fd=DefaultFd;
   return;
 }
 for(;;) {
   const int new_fd=::dup(fd);
   if(new_fd>=0) {
     _fd=new_fd;
     return;
   }
   if(errno==EINTR) {
     continue;
   }
   WX_THROW( Fault(WxErrMsg(errno)) );
 }
}
catch(const Wx__Handle::Fault&) {
  throw;
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

Wx__Handle::Wx__Handle(const Wx__Handle &src) throw(std::exception,Fault)
try {
 if(&src==this) {
   WX_THROW(Wx_general_error());  // Looking for trouble
 }
 _fd=DefaultFd;
 if(src._fd==DefaultFd) {
   return;
 }
 for(;;) {
   const WxRet r=src._dup(*this);
   if(r==OK) {
     return;
   }
   if(r==WXM_EINTR) {
     continue;
   }
   WX_THROW( Fault(r) );
 }
 return;
}
catch(const Wx__Handle::Fault&) {
  throw;
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

Wx__Handle::~Wx__Handle() throw(std::exception,Fault)
try {
  WxRet r=_reset();
  if(r!=OK) {
    WX_THROW( Fault(r) );
  }
  return;
}
catch(const Fault&) {
  throw;
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet Wx__Handle::assign(const Wx__Handle& src) throw(std::exception)
try {
 if(&src==this) {
   return(OK);
 }
 if(_fd!=DefaultFd) {
   WX_RETURN(WXM_NDEFAULT);
 }
 if(src.is_default()) {
   return(OK);
 }
 const WxRet r=src._dup(*this);
 WX_RETURN(r);
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet Wx__Handle::assign(Wx__TypeFD src) throw(std::exception)
try {
 if(_fd!=DefaultFd) {
   WX_RETURN(WXM_NDEFAULT);
 }
 if(src==DefaultFd) {
   return(OK);
 }
 const int new_fd=::dup(src);
 if(new_fd>=0) {
  _fd=new_fd;
  return(OK);
 }
 WX_RETURN(WxErrMsg(errno));
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet Wx__Handle::read(void *buf, size_t count, size_t *n_read) const throw(std::exception)
try {
 if(n_read!=0) {
   *n_read=0;   
 }

#ifdef __GLIBC__
 if(count>size_t(std::numeric_limits<ssize_t>::max())) {
   WX_RETURN(WXM_E2BIG);   // man. says unpredictable results, guess specific for glib
 }
#endif

 Wx__Base::cancel_point();
 ssize_t v=::read(_fd,buf,count);
 Wx__Base::cancel_point();
 if(v>=0) {
   if(size_t(v)>count) {
     WX_THROW(Wx_general_error());
   }
   if(n_read!=0) {
     *n_read=v;
   }
   return(OK);
 }
 WX_RETURN(WxErrMsg(errno));
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet Wx__Handle::write(const void *buf, size_t count, size_t *n_written) const throw(std::exception)
try {
 if(n_written!=0) {
   *n_written=0;
 }
#ifdef __GLIBC__
 if(count>size_t(std::numeric_limits<ssize_t>::max())) {
   WX_RETURN(WXM_E2BIG); // man. says unpredictable results, guess specific for glib
 }
#endif

 Wx__Base::cancel_point();
 ssize_t v=::write(_fd,buf,count);
 Wx__Base::cancel_point();
 if(v>=0) {
   if(size_t(v)>count) {
     WX_THROW(Wx_general_error());
   }
  if(n_written!=0) {
     *n_written=v;
   }
   return(OK);
 }
 WX_RETURN(WxErrMsg(errno));
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet Wx__Handle::max_read(void* buf,size_t count,size_t* n_read) const throw(std::exception)
try {
  if(n_read!=0) {
    *n_read=0;
  }
#ifdef __GLIBC__
 if(count>size_t(std::numeric_limits<ssize_t>::max())) {
   WX_RETURN(WXM_E2BIG);   // man. says unpredictable results, guess specific for glib
 }
#endif

  size_t btr=count;
  char *bptr=static_cast<char*>(buf);

  while(btr>0) {
    ssize_t n;

    Wx__Base::cancel_point();
    n=::read(_fd,bptr,btr);
    Wx__Base::cancel_point();

    if(n<=0) {
      if(n==0) {
        break;                    // EOF
      } else {
        WX_RETURN(WxErrMsg(errno));
      }
    }
    if(size_t(n)>btr) {
      WX_THROW(Wx_general_error());
    }

    if(n_read!=0) {
      *n_read+=n;
    }
    bptr+=n;
    btr-=n;
  }
  return(OK);
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet Wx__Handle::pread(void *buf, size_t count, size_t *n_read, off_t pos) const throw(std::exception)
try {
 if(n_read!=0) {
   *n_read=0;   
 }

#ifdef __GLIBC__
 if(count>size_t(std::numeric_limits<ssize_t>::max())) {
   WX_RETURN(WXM_E2BIG);   // man. says unpredictable results, guess specific for glib
 }
#endif

 Wx__Base::cancel_point();
 ssize_t v=::pread(_fd,buf,count,pos);
 Wx__Base::cancel_point();
 if(v>=0) {
   if(size_t(v)>count) {
     WX_THROW(Wx_general_error());
   }
   if(n_read!=0) {
     *n_read=v;
   }
   return(OK);
 }
 WX_RETURN(WxErrMsg(errno));
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet Wx__Handle::pwrite(const void *buf, size_t count, size_t *n_written,off_t pos) const throw(std::exception)
try {
 if(n_written!=0) {
   *n_written=0;
 }
#ifdef __GLIBC__
 if(count>size_t(std::numeric_limits<ssize_t>::max())) {
   WX_RETURN(WXM_E2BIG); // man. says unpredictable results, guess specific for glib
 }
#endif

 Wx__Base::cancel_point();
 ssize_t v=::pwrite(_fd,buf,count,pos);
 Wx__Base::cancel_point();
 if(v>=0) {
   if(size_t(v)>count) {
     WX_THROW(Wx_general_error());
   }
  if(n_written!=0) {
     *n_written=v;
   }
   return(OK);
 }
 WX_RETURN(WxErrMsg(errno));
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet Wx__Handle::max_pread(void* buf,size_t count,size_t* n_read,off_t pos) const throw(std::exception)
try {
  if(n_read!=0) {
    *n_read=0;
  }
#ifdef __GLIBC__
 if(count>size_t(std::numeric_limits<ssize_t>::max())) {
   WX_RETURN(WXM_E2BIG);   // man. says unpredictable results, guess specific for glib
 }
#endif

  size_t btr=count;
  off_t rpos=pos;
  char *bptr=static_cast<char*>(buf);

  while(btr>0) {
    ssize_t n;

    Wx__Base::cancel_point();
    n=::pread(_fd,bptr,btr,rpos);
    Wx__Base::cancel_point();

    if(n<=0) {
      if(n==0) {
        break;                    // EOF
      } else {
        WX_RETURN(WxErrMsg(errno));
      }
    }
    if(size_t(n)>btr) {
      WX_THROW(Wx_general_error());
    }

    if(n_read!=0) {
      *n_read+=n;
    }
    rpos+=n;
    bptr+=n;
    btr-=n;
  }
  return(OK);
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet Wx__Handle::fstat(struct stat *buf) const throw(std::exception)
try {
 //
 // Assert that ::fstat failure should not modify buf.
 //
 if(::fstat(_fd,buf)==0) {
   return(OK);
 }
 WX_RETURN(WxErrMsg(errno));
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet Wx__Handle::fcntl(int cmd,int *ret_f) const throw(std::exception)
try {
 const int v=::fcntl(_fd,cmd); 
 if(v>=0) {
   if(ret_f!=0) {
     *ret_f=v;
   }
   return(OK);
 }
 WX_RETURN(WxErrMsg(errno));
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet Wx__Handle::fcntl(int cmd, long arg, int *ret_f) const throw(std::exception)
try {
 const int v=::fcntl(_fd,cmd,arg); 
 if(v>=0) {
   if(ret_f!=0) {
     *ret_f=v;
   }
   return(OK);
 }
 WX_RETURN(WxErrMsg(errno));
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet Wx__Handle::fcntl(int cmd, struct flock *lock,int *ret_f) const throw(std::exception)
try {
 const int v=::fcntl(_fd,cmd,lock); 
 if(v>=0) {
   if(ret_f!=0) {
     *ret_f=v;
   }
   return(OK);
 }
 WX_RETURN(WxErrMsg(errno));
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

#ifdef _POSIX_SYNCHRONIZED_IO
WxRet Wx__Handle::fdatasync(void) const throw(std::exception)
try {
 if(::fdatasync(_fd)==0) {
   return(OK);
 }
 WX_RETURN(WxErrMsg(errno));
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};
#endif

WxRet Wx__Handle::open(const char* pathname, int flags, mode_t mode) throw(std::exception)
try {
 if(_fd!=DefaultFd) {
   WX_RETURN(WXM_NDEFAULT);
 }
 Wx__Base::cancel_point();
 const int v=::open(pathname,flags,mode);
 Wx__Base::cancel_point();

 if(v>=0) {
   _fd=v;
   return(OK);
 }
 WX_RETURN(WxErrMsg(errno));
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet Wx__Handle::open(const char* pathname, int flags) throw(std::exception)
try {
 if(_fd!=DefaultFd) {
   WX_RETURN(WXM_NDEFAULT);
 }
 Wx__Base::cancel_point();
 const int v=::open(pathname,flags);
 Wx__Base::cancel_point();
 if(v>=0) {
   _fd=v;
   return(OK);
 }
 WX_RETURN(WxErrMsg(errno));
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet Wx__Handle::close(void) throw(std::exception)
try {
 if(_fd==DefaultFd) {
   return(OK);
 }
 Wx__Base::cancel_point();
 const int v=::close(_fd);
 Wx__Base::cancel_point();
 if(v==0) {
   _fd=DefaultFd;
    return(OK);
 }
 if(errno==EINTR) {
   WX_RETURN(WXM_EINTR);
 }
 _fd=DefaultFd;
 WX_RETURN(WxErrMsg(errno));
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet Wx__Handle::set_fd(Wx__TypeFD fd) throw(std::exception)
try {
 if(_fd!=DefaultFd) {
   WX_RETURN(WXM_NDEFAULT);
 }
 if(wx_is_open_fd(fd)==false) {
   WX_RETURN(WXM_EINVAL);
 }
 _fd=fd;
 return(OK);
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxStr Wx::what_is(const Wx__Handle& hdl) throw(std::exception)
try {
 WxStr str;
 WxRet r=Wx::append_number(str,hdl.fd(),10);
 if(r!=OK) {
   WX_THROW( Wx_bad_alloc() );   // no others are sensible
 }
 return(str);
}
catch(const WxStr::Fault& e) {
 WX_THROW( Wx_general_error() );
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet Wx__Handle::_dup(Wx__Handle& dest) const throw(std::exception)
try {
 if(dest._fd!=DefaultFd) {
   WX_RETURN(WXM_EINVAL);
 }
 const int new_fd=::dup(_fd);
 if(new_fd>=0) {
  dest._fd=new_fd;
  return(OK);
 }
 WX_RETURN(WxErrMsg(errno));
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet Wx__Handle::truncate(off_t length) const throw(std::exception)
try {
 const int v=::ftruncate(_fd,length);
 if(v==0) {
  return(OK);
 }
 WX_RETURN(WxErrMsg(errno));
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet Wx__Handle::ttyname(WxStr& pathname) const throw(std::exception)
try {
  char buf[_POSIX_PATH_MAX];   // not sure this is right
  const int v=::ttyname_r(_fd,buf,sizeof(buf));
  if(v!=0) {
    if(v==ERANGE) {
      WX_THROW(Wx_general_error()); // buf is too small
    }
    WX_RETURN(WxErrMsg(v));
  }
  WxRet r=pathname.assign(buf);
  if(r!=OK) {
    WX_RETURN(r);
  }
  return(OK);
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet Wx__Handle::_reset(void) throw(std::exception)
try {
  for(;;) {
    WxRet r=close();
    if(r==OK) {
      _fd=DefaultFd;
      return(OK);
    }
    if(r==WXM_EINTR) {
      continue;
    }
    _fd=DefaultFd;
    WX_RETURN(r);
  }
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};
