/* Copyright is licensed under GNU LGPL.                (I.J. Wang, 2003)

   This file should conform to the documentation in file wxdirfile.3wx
*/

#ifndef WX__DIRHANDLE_H__
#define WX__DIRHANDLE_H__
#define WX__DIRHANDLE_VERSION 8

#include "wxmutex.h"
#include "wxpathname.h"
#include "wxfilestat.h"
#include "wxdirent.h"
#include <sys/types.h>
#include <dirent.h>
#include <vector>

//
// [Internal] Wx__DirHandEnt defines the class of system directory handle.
//            This class exists because no dup/fstat equivalent 
//            operation for ::DIR* as with file descriptor
//
class Wx__DirHandEnt {
  public:
    WX_DECLARE_FAULT;  // declaraion of Fault

    //
    // [Syn] Construct default object 
    //       (ref_count() == 0)
    //
    Wx__DirHandEnt() throw()
              : _rcnt(0) {};

    //
    // [Syn] Construct object by copy  (only default object)
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //
    // Note: This constructor is for std::vector to expand size.
    //       Copy source must be default.
    //       No two Wx__DirHandEnt objects can be copied except the default ones.
    //
    Wx__DirHandEnt(const Wx__DirHandEnt& src) throw(std::exception);

    //
    // [Syn] Destroy object
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    //
    //  Note: reset error is ignored
    //
    ~Wx__DirHandEnt() throw(std::exception)
              { _reset(); };

    //
    // [Syn] is object the state as default constructed
    //
    // [Ret] true  = object is the same as default constructed, 
    //       false = otherwise
    //
    bool is_default(void) const throw()
              { return _rcnt==0; };

    //
    // [Syn] Open the object from the directory dirname and set the reference
    //       count to one.
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    //
    // [Ret] OK 
    //       WXM_NDEFAULT not a default object
    //       WXM_EACCES search permission is denied
    //       WXM_ENAMETOOLONG pathname too long
    //       WXM_ENOENT pathname does not exist
    //       WXM_ENOTDIR a component in pathname is not a directory
    //       WXM_EMFILE too many open fd
    //       WXM_ENFILE file table overflow
    //       WXM_ENOMEM not enough memory
    //       ...        errmsg converted from ::opendir(...) or open(int)
    //
    WxRet open(const char* dirname) throw(std::exception);

    //
    // [Syn] Check in the use of the object and increase the reference count
    //       by one. The object should have been open to call this function.
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //
    // [Ret] OK         reference of the object counted
    //       WXM_ENOENT object is not open yet
    //       WXM_ENFILE counter overflow
    //
    WxRet check_in(void) throw(std::exception);

    //
    // [Syn] Check out the use of the object and decrease the reference count
    //       by one. If the reference count reaches zero, the object is reset.
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    //
    // [Ret] OK
    //       WXM_ENOENT object is default
    //       WXM_EIO   Input/output error 
    //       ...       errmsg converted from the call of ::closedir(...)
    //
    // Note: failed or not, reference count is decreased by one (until zero).
    //
    WxRet check_out(void) throw(std::exception);

    //
    // [Syn] Get the ::DIR pointer of this object
    //
    // [Ret] ::DIR pointer of this object
    //
    // Note: The returned value is valid while object is open or checked-in.
    //       And is dedicated for use by readdir/rewind
    //
    inline ::DIR* dirp(void) throw()
              { return _dirp; };

    //
    // [Syn] Get the file descriptorof this object
    //
    // [Ret] file descriptor of this object
    //
    // Note: The returned value is valid while object is open or checked-in.
    //       And is dedicated for use by fstat
    //
    inline Wx__TypeFD fd(void) throw()
              { return _fd; };

    //
    // [Syn] Assign operator 
    //
    // [Exception] Wx_general_error
    //
    // Note: This function is for use by std::vector to expand size.
    //       This object and source object must be default
    //
    const Wx__DirHandEnt& operator =(const Wx__DirHandEnt& rhs) throw(std::exception)
              {
               if(&rhs==this) {
                 return(*this);
               }
               if(rhs._rcnt!=0) {
                 WX_THROW(Wx_general_error());   // rhs not default
               }
               if(_rcnt!=0) {
                 WX_THROW(Wx_general_error());   // object not default
               }
               return(*this);
              };
 
    //
    // [Syn] Get the reference count
    //
    // [Ret] reference count
    //
    size_t ref_count(void) const throw()
              { return _rcnt; };
      
  private:
    //
    // [Syn] Reset object to the default state
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    //
    // [Ret] OK
    //       WXM_EIO   Input/output error 
    //       ...       errmsg converted from the returned errno of closedir(...)
    //
    // Note: Object is always default while returned.
    //       EINTR is retried in the internal loop to go away
    //
    WxRet _reset(void) throw(std::exception);

    size_t _rcnt;       // reference count
    ::DIR *_dirp;       // points to the ::DIR this object is using
    Wx__TypeFD _fd;     // file descriptor this object is using
};

/*
  [Internal] Wx__DirHandle defines the class of the system directory.
*/
class Wx__DirHandle {
  public:
    static const char class_name[];
    WX_DECLARE_FAULT;

    //
    // [Syn] Construct default object. (Object refers to nothing)
    //
    Wx__DirHandle() throw()
              : _idx(DefaultIdx) {};

    //
    // [Syn] Construct object by copy from src
    //
    // Note: The copy object associates to the same node as src does.
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    // 
    // [Exception] Fault
    //             WXM_ENFILE counter overflow
    // 
    // Note: This copy constructed share the same information with src
    //       Operations of one object is the same as that on the other object.
    // 
    // [Refer] ::dup(int)
    //
    Wx__DirHandle(const Wx__DirHandle&) throw(std::exception,Fault);

    //
    // [Syn] Destroy the object.
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    // 
    //  Note: reset error is ignored
    //
    // [Refer] ::closrdir(...),close(int)
    //
    ~Wx__DirHandle() throw(std::exception);

    //
    // [Syn] is object the state as default constructed
    //
    // [Ret] true  = object is the same as default constructed, 
    //       false = otherwise
    //
    bool is_default(void) const throw()
              { return _idx==DefaultIdx; };

    //
    // [Syn] Read the filestat of this object and save to filestat
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    // 
    // [Ret] OK    
    //       WXM_EBADF  object is default
    //       ......
    // 
    // Note: errmsg is converted from the errno ::fstat returned
    //       
    // [Refer] ::fstat(int,struct stat)
    // 
    WxRet fstat(WxFileStat& filestat) const throw(std::exception);

    //
    // [Syn] Reset the object to the state as default constructed.
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    // 
    // [Ret] OK    Succeed
    //       WXM_EIO   Input/output error 
    //       ...   errmsg converted from the errno ::close/::closedir returned
    //
    // Note: Object is always default while returned.
    //       EINTR is retried in the internal loop to go away
    //
    // [Refer] ::closedir(...),::close(int)
    //
    WxRet reset(void) throw(std::exception);

    //
    // [Syn] Read the directory entry of the next position
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    // 
    // [Ret] OK        success, dirent is stored with the directory entry read.
    //                 dirent is set to default if no more entry to read.
    //       ...       errmsg converted from ::readdir_r
    // 
    // Note: Multiple processes may access exactly the same directory at the
    //       same time.
    // 
    // [Refer] ::readdir_r
    // 
    WxRet read(WxDirEnt& dirent) throw(std::exception);

    //
    // [Syn] Reset the read position
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    // 
    // [Ret] OK           success
    //       WXM_BADF  object is default
    // 
    WxRet rewind(void) throw(std::exception);

    //
    // [Syn] Open the the directory node by the given pathname (read-only)
    // 
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    //
    // [Ret] OK         succeed
    //       WXM_NDEFAULT object is not default
    //       WXM_EACCES search permission is denied
    //       WXM_EEXIST pathname already exists
    //       WXM_EINTR  interrupted by a signal
    //       WXM_EISDIR object refers to a directory
    //       WXM_EMFILE too many open fd
    //       WXM_ENAMETOOLONG pathname too long
    //       WXM_ENFILE file table overflow
    //       WXM_ENOENT pathname does not exist
    //       WXM_ENOSPC device left no space for the operation
    //       WXM_ENOTDIR a component in pathname is not a directory
    //       WXM_ENXIO  no such device (device not ready)
    //       WXM_EROFS read-only file system
    //       WXM_ENODEV no such device
    //       WXM_ETXTBSY Text file busy (see man.)
    //       WXM_EFAULT pathname is not accessible
    //       WXM_ELOOP too many symbolic links encountered
    //       WXM_ENOMEM not enough memory
    //       ...        errmsg converted from the errno ::opendir and ::open(...)
    //                  returned
    //
    // [Refer] ::opendir(const char*),::open(...)
    //
    WxRet open(const WxPathName& dirname) throw(std::exception);

    //
    // [Syn] Assign object to handle the device handled by src
    //
    // [Exception] Wx_general_error
    //             Wx_bad_alloc
    //             Wx_bad_errno
    // 
    // [Ret] OK         object can be used interchangeably with src
    //       WXM_NDEFAULT object is not default
    //       WXM_ENFILE counter overflow
    //
    // Note: This assigned object share the same information with src
    //       Operations of one object is the same as that on the other object.
    //
    // [Refer]
    // 
    WxRet assignf(const Wx__DirHandle& src) throw(std::exception);

    //
    // [Internal]
    //
    size_t ref_count(void) const throw(std::exception);
  private:
    enum { DefaultIdx=-1 };

    // Handle table
    static std::vector<Wx__DirHandEnt> wx_tab; 
    static WxMutex wx_tab_mtx;            

    int _idx;   // index of the entry in wx_tab
};

#endif
