
#include "wx__dirhandle.h"
#include <sys/stat.h>
#include <fcntl.h>

WxRet Wx__DirHandEnt::_reset(void) throw(std::exception)
try {
  if(_rcnt<=0) {
    return(OK);
  }
  _rcnt=0;
  if(::closedir(_dirp)!=0) {
    const int en=errno;
    while(::close(_fd)!=0) {
      if((errno==EINTR)||(errno==EAGAIN)) {
      } else {
        break;     // errno ignored, for already had one
      }
    };
    if((en==EBADF)||(en==0)) {  // closedir(_dirp) error should not return this
      WX_THROW( Wx_bad_errno(en) );
    }
    WX_RETURN(WxErrMsg(en));
  }
  while(::close(_fd)!=0) {
    const int en=errno;
    if((en==EINTR)||(en==EAGAIN)) {
    } else {
      if((en==EBADF)||(en==0)) {  // close(_fd) error should not return this
        WX_THROW( Wx_bad_errno(en) );
      }
      WX_RETURN(WxErrMsg(en));
    }
  };
  return(OK);
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_general_error&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
  WX_THROW(Wx_general_error());
};

Wx__DirHandEnt::Wx__DirHandEnt(const Wx__DirHandEnt& src) throw(std::exception)
try {
  if(&src==this) {
    WX_THROW(Wx_general_error());
  }
  _rcnt=0;
  if(src._rcnt!=0) {
    // +---------+
    // | Warning |
    // +---------+
    //
    // Execution might get here but not sure.
    //  (RH Linex 9.0, stl_vector.h, line 904)
    //   _M_insert_aux(iterator,const _Tp&)
    //   .....
    //   uninitialized_copy(...) // should not have this constructor called
    // The workaround is by preallocating wx_tab to some size to prevent
    // this call. (see wx_tab)
    //
    WX_THROW(Wx_general_error());
  }
}
catch(const Wx_general_error&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
  WX_THROW(Wx_general_error());
};

WxRet Wx__DirHandEnt::open(const char* dirname) throw(std::exception)
try {
  if(_rcnt>0) {
    WX_RETURN(WXM_NDEFAULT);
  }
  _dirp=NULL;
  _fd=WX__DEFAULT_FD;
  try {
   // open _dirp
   _dirp=::opendir(dirname);
   if(_dirp==NULL) {
     if(errno==0) {
       WX_THROW( Wx_bad_errno(errno) );
     }
     WX_RETURN(WxErrMsg(errno));
   }

   // open _fd
   _fd=::open(dirname,O_RDONLY|O_DIRECTORY);
   if(_fd<0) {
     ::closedir(_dirp);
     WX_RETURN(WxErrMsg(errno));
   }
  }
  catch(...) {
    if(_dirp!=NULL) {
      ::closedir(_dirp);
    }
    if(_fd>=0) {
      while(::close(_fd)!=0) {
        if((errno==EINTR)||(errno==EAGAIN)) {
        } else {
          break;     // errno ignored, for already had one
        }
      };
    }
    _rcnt=0;
    throw;
  }
  _rcnt=1;
  return(OK);
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
  WX_THROW(Wx_general_error());
};

WxRet Wx__DirHandEnt::check_in(void) throw(std::exception)
try {
  if(_rcnt<=0) {
    return(WXM_ENOENT);
  }
  if(_rcnt>=std::numeric_limits<size_t>::max()) {
    return(WXM_ENFILE);
  }
  ++_rcnt;
  return(OK);
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
  WX_THROW(Wx_general_error());
};

WxRet Wx__DirHandEnt::check_out(void) throw(std::exception)
try {
  if(_rcnt<=1) {
    if(_rcnt==0) {
      WX_RETURN(WXM_ENOENT);
    }
    WX_RETURN( _reset() );  // reset will zero the count
  }
  --_rcnt;
  return(OK);
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
  WX_THROW(Wx_general_error());
};

//--------------------------------------------------------------------

const char Wx__DirHandle::class_name[]="Wx__DirHandle";

std::vector<Wx__DirHandEnt> Wx__DirHandle::wx_tab(256);
WxMutex Wx__DirHandle::wx_tab_mtx;

Wx__DirHandle::Wx__DirHandle(const Wx__DirHandle& src) throw(std::exception,Fault)
try {
 if(&src==this) {
   WX_THROW(Wx_general_error());
 }
 if(src._idx==DefaultIdx) {  // src is defaut
   _idx=DefaultIdx;
   return;
 }

 WxLock aa(wx_tab_mtx);
 _idx=src._idx;
 if(_idx>=int(wx_tab.size())) {
   WX_THROW(Wx_general_error());  // src._idx out of range
 }

 const WxRet r=wx_tab[_idx].check_in();
 if(r!=OK) {
   if(r==WXM_ENFILE) {
     WX_THROW( Fault(r) ); // only this possible
   }
   WX_THROW(Wx_general_error());
 }
}
catch(const WxLock::Fault& e) {
  WX_THROW(Wx_general_error());
}
catch(const Wx__DirHandle::Fault&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
  WX_THROW(Wx_general_error());
};

Wx__DirHandle::~Wx__DirHandle() throw(std::exception)
try {
 if(_idx==DefaultIdx) {
   return;
 }

 WxLock aa(wx_tab_mtx);

 if(_idx>=int(wx_tab.size())) {
   WX_THROW(Wx_general_error());  // _idx out of range
 }

 wx_tab[_idx].check_out();    // error ignored
}
catch(const WxLock::Fault& e) {
  WX_THROW(Wx_general_error());
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet Wx__DirHandle::fstat(WxFileStat& filestat) const throw(std::exception)
try {
 if(_idx==DefaultIdx) {
   WX_RETURN(WXM_EBADF);
 }
 
 WxLock aa(wx_tab_mtx);

 if(_idx>=int(wx_tab.size())) {
   WX_THROW(Wx_general_error());  // _idx out of range
 }

 if(::fstat(wx_tab[_idx].fd(),filestat.wx_stt())!=0) {
   if(errno==0) {
     WX_THROW( Wx_bad_errno(errno) );
   }
   WX_RETURN(WxErrMsg(errno));
 }
 return(OK);
}
catch(const WxLock::Fault& e) {
  WX_THROW(Wx_general_error());
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet Wx__DirHandle::reset(void) throw(std::exception)
try {
 if(_idx==DefaultIdx) {
   return(OK);
 }
 WxLock aa(wx_tab_mtx);

 if(_idx>=int(wx_tab.size())) {
   WX_THROW(Wx_general_error());  // _idx out of range
 }

 WxRet r=wx_tab[_idx].check_out();
 _idx=DefaultIdx;

 if((r==WXM_ENOENT)||(r==WXM_EBADF)) {
   WX_THROW(Wx_general_error());
 }
 WX_RETURN(r);
}
catch(const WxLock::Fault& e) {
  WX_THROW(Wx_general_error());
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet Wx__DirHandle::read(WxDirEnt& dirent) throw(std::exception)
try {
 if(_idx==DefaultIdx) {
   WX_RETURN(WXM_EBADF);
 }
 
 WxLock aa(wx_tab_mtx);

 if(_idx>=int(wx_tab.size())) {
   WX_THROW(Wx_general_error());  // _idx out of range
 }

 struct ::dirent* ptr=&dirent._ent; 
 const int v=::readdir_r(wx_tab[_idx].dirp(),ptr,&ptr);
 if(v!=0) {
   if((v==EBADF)||(v==0)) {
     WX_THROW(Wx_general_error());  // should not get this
   }
   WX_RETURN(WxErrMsg(v));
 }
 if(ptr==NULL) {   // end of directory
   dirent.reset();
 }
 return(OK);
}
catch(const WxLock::Fault& e) {
  WX_THROW(Wx_general_error());
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet Wx__DirHandle::rewind(void) throw(std::exception)
try {
 if(_idx==DefaultIdx) {
   WX_RETURN(WXM_EBADF);
 }
 
 WxLock aa(wx_tab_mtx);

 if(_idx>=int(wx_tab.size())) {
   WX_THROW(Wx_general_error());  // _idx out of range
 }

 ::rewinddir(wx_tab[_idx].dirp());
 return(OK);
}
catch(const WxLock::Fault& e) {
  WX_THROW(Wx_general_error());
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet Wx__DirHandle::open(const WxPathName& pathname) throw(std::exception)
try {
  if(_idx!=DefaultIdx) {
    WX_RETURN(WXM_NDEFAULT);
  }
  WxLock aa(wx_tab_mtx);
  // 
  // Locate an empty entry of wx_tab (from 0)
  //
  size_t nidx;
  for(nidx=0; nidx<wx_tab.size(); ++nidx) {
    if(wx_tab[nidx].is_default()) {
      break;
    }
  }
  if(nidx>=wx_tab.size()) {
    wx_tab.push_back(Wx__DirHandEnt()); // push a default entry and use it
  }

  //
  // Open the entry wx_tab[_idx];
  //
  const WxRet r=wx_tab[nidx].open(pathname.pathname().c_str());
  if(r!=OK) {
    WX_RETURN(r);
  }
  _idx=nidx;
  return(OK);
}
catch(const WxLock::Fault& e) {
  WX_THROW(Wx_general_error());
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

WxRet Wx__DirHandle::assignf(const Wx__DirHandle& src) throw(std::exception)
try {
 if(&src==this) {
   return(OK);
 }
 if(_idx!=DefaultIdx) {
   WX_RETURN(WXM_NDEFAULT);
 }
 if(src._idx==DefaultIdx) {  // src is defaut
   _idx=DefaultIdx;
   return(OK);
 }

 WxLock aa(wx_tab_mtx);
 _idx=src._idx;
 if(_idx>=int(wx_tab.size())) {
   WX_THROW(Wx_general_error());  // src._idx out of range
 }

 const WxRet r=wx_tab[_idx].check_in();
 if(r!=OK) {
   if(r==WXM_ENFILE) {
     WX_RETURN(r); // only this possible
   }
   WX_THROW(Wx_general_error());
 }
 return(OK);
}
catch(const WxLock::Fault& e) {
  WX_THROW(Wx_general_error());
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};

size_t Wx__DirHandle::ref_count(void) const throw(std::exception)
try {
 if(_idx==DefaultIdx) {
   return(0);
 }
 
 WxLock aa(wx_tab_mtx);

 if(_idx>=int(wx_tab.size())) {
   WX_THROW(Wx_general_error());  // _idx out of range
 }

 return(wx_tab[_idx].ref_count());
}
catch(const WxLock::Fault& e) {
  WX_THROW(Wx_general_error());
}
catch(const Wx_general_error&) {
  throw;
}
catch(const Wx_bad_errno&) {
  throw;
}
catch(const Wx_bad_alloc&) {
  throw;
}
catch(const std::bad_alloc&) {
  WX_THROW( Wx_bad_alloc() );
}
catch(...) {
 WX_THROW(Wx_general_error());
};
