/* Copyright is licensed under GNU LGPL.                (I.J. Wang, 2003)

   Build: make chk_time
*/
#include "wxtime.h"
#include "wxstr.h"
#include <iostream>
#include <limits>
#include <unistd.h>
#include <sys/types.h>

#if WXTIME_VERSION!=8
#error Test code is for WXTIME_VERSION 8
#endif

#define TEST_SLEEP_TILL

//
// Constant limits use for test
//
static const long LongMax=std::numeric_limits<long>::max();
static const long LongMin=std::numeric_limits<long>::min();
static const long long LLongMax=std::numeric_limits<long long>::max();
static const long long LLongMin=std::numeric_limits<long long>::min();

static bool chk_default(const WxTime& tm)
{
const static WxTime T0;
 if(tm.is_default()==false) {
   return(false);
 }
 if((tm.secs()!=0)||(tm.nano()!=0)) {
   return(false);
 }
 if(tm==T0) {
 } else {
   return(false);
 }
 if(tm!=T0) {
   return(false);
 }
 return(true);
};

static bool chk_not_default(const WxTime& tm)
{
const static WxTime T0;
 if(tm.is_default()) {
   return(false);
 }
 if((tm.secs()==0)&&(tm.nano()==0)) {
   return(false);
 }
 if(tm!=T0) {
 } else {
   return(false);
 }
 if(tm==T0) {
   return(false);
 }
 return(true);
};

static bool chk_equ(const WxTime& tm1,const WxTime& tm2)
{
 if(tm1==tm2) {
 } else {
   return(false);
 }
 if(tm1!=tm2) {
   return(false);
 }
 if(tm1.secs()!=tm2.secs()) {
   return(false);
 }
 if(tm1.nano()!=tm2.nano()) {
   return(false);
 }
 return(true);
};

#define CHK_EQU(expr1,expr2) if(chk_equ(expr1,expr2)==false) { WX_THROW( Wx_general_error() ); };
#define CHK_NOT_EQU(expr1,expr2) if(chk_not_equ(expr1,expr2)==false) { WX_THROW( Wx_general_error() ); };

#define CHK_DEFAULT(expr) if(chk_default(expr)==false) { WX_THROW( Wx_general_error() ); };
#define CHK_NOT_DEFAULT(expr) if(chk_not_default(expr)==false) { WX_THROW( Wx_general_error() ); };

/*
   Test WxTime public interface (set_systime is not tested)
*/
static void t1(void)
{
 WxRet r;

 //
 // Test WxTime constructors and basic test
 //
 WxTime a1;
 WxTime a2(100,100);
 WxTime a3(a2);
 WxTime a4(99,1000000100);

 CHK_DEFAULT(a1);
 CHK_NOT_DEFAULT(a2);
 CHK_NOT_DEFAULT(a3);
 CHK_NOT_DEFAULT(a4);

 if((a1==a2)||(a1==a3)||(a1==a4)) {
   WX_THROW( Wx_general_error() ); 
 }
 if((a2!=a3)||(a2!=a4)||(a3!=a4)) {
   WX_THROW( Wx_general_error() ); 
 }
 if((a2.secs()!=a3.secs())||(a2.nano()!=a3.nano())) {
   WX_THROW( Wx_general_error() ); 
 }

 //
 // Test reset
 //
 a3.reset();
 CHK_DEFAULT(a3);

 a3.reset(a4);
 CHK_EQU(a3,a4);

 if((r=a3.reset(1,2))!=OK) {
    WX_HERE(r); throw(r);
 }
 CHK_EQU(a3,WxTime(1,2));

 //
 // Test assign
 //
 if((r=a3.assign(300,LongMax))!=OK) {
   WX_THROW( Wx_general_error() ); 
 }
 if((a3.secs()!=300+LongMax/1000000000)||(a3.nano()!=LongMax%1000000000)) {
   WX_THROW( Wx_general_error() );   // not rounded-up
 }
 a2.assign(a3);
 CHK_EQU(a2,a3);

 if((r=a3.assign(300,LongMin))!=OK) {
   WX_THROW( Wx_general_error() ); 
 }
 a2.assign(a3);
 CHK_EQU(a2,a3);

 //
 // Test abs()
 //
 CHK_EQU(a1.abs(),WxTime(0,0));

 if((r=a2.assign(-300,-300))!=OK) {
   WX_THROW( Wx_general_error() ); 
 }
 CHK_EQU(a2.abs(),WxTime(300,300));

 //
 // Test add
 //
 if((r=a2.assign(-400,-400))!=OK) {
   WX_THROW( Wx_general_error() ); 
 }
 a3.assign(a2);
 if((r=a2.add(500,500))!=OK) {
   WX_THROW( Wx_general_error() ); 
 }
 if((r=a3.add(WxTime(500,500)))!=OK) {
   WX_THROW( Wx_general_error() ); 
 }
 CHK_EQU(a2,a3);

 if((a2.nano()!=100)||(a2.secs()!=100)) {
   WX_THROW( Wx_general_error() ); 
 }

 //
 // Test operator =
 //
 a2.assign(-100,-100);
 a3.assign( 100, 100);
 a4=a3;
 CHK_EQU(a4,a3);

 a4=a2;
 CHK_EQU(a4,a2);

 //
 // Test operator -()
 //
 CHK_EQU(-a1,WxTime(0,0));

 a2.assign(-111,-111);
 CHK_EQU(-a2,WxTime(111,111));

 a2.assign(-111,-LongMax);
 CHK_EQU(-a2,WxTime(111,LongMax));

 //
 // Test operator +/-, +=
 //
 a2.assign(100,100);
 a3.assign(200,200);
 a4=a2+a3;
 if((a4.secs()!=300)||(a4.nano()!=300)) {
   WX_THROW( Wx_general_error() ); 
 }
 a4=a2-a3;
 CHK_EQU(a4,WxTime(-100,-100));

 a4+=a3;
 CHK_EQU(a4,a2);

 //
 // Test operator <,<=, >,>=
 //
 a2.assign(100,100);
 a3.assign(200,200);
 a4=a2+a3;
 if((a4<a2)||(a4<a3)) {
   WX_THROW( Wx_general_error() ); 
 }
 if((a4<=a2)||(a4<=a3)) {
   WX_THROW( Wx_general_error() ); 
 }
 a2.assign(-100,-100);
 a3.assign(-200,-200);
 a4=a2+a3;
 if((a4>a2)||(a4>a3)) {
   WX_THROW( Wx_general_error() ); 
 }
 if((a4>=a2)||(a4>=a3)) {
   WX_THROW( Wx_general_error() ); 
 }

 //
 // Simple test for of Wx::now
 //
 {
  WxTime tm1=Wx::now();
  WxTime tm2=tm1+WxTime(15,0);
  if((tm1-tm2).abs()!=WxTime(15,0)) {
    WX_THROW( Wx_general_error() ); 
  }
 }

 //
 // check the default time refers to 1970-1-1 0000
 //
 {
   WxTime tm1;
   time_t tm_t=tm1.secs()-28800;       // time_t is based on 1970-1-1 0800
   struct tm* tsp=::localtime(&tm_t);
   if(tsp==NULL) {
     WX_THROW( Wx_general_error() ); 
   }
   if(tsp->tm_year!=70) {
     WX_THROW( Wx_general_error() ); 
   }
   if(tsp->tm_mon!=0) {
     WX_THROW( Wx_general_error() ); 
   }
   if(tsp->tm_mday!=1) {
     WX_THROW( Wx_general_error() ); 
   }
   if(tsp->tm_hour!=0) {     // this have nothing to with the Epoch WxTime based
     WX_THROW( Wx_general_error() ); 
   }
   if(tsp->tm_min!=0) {
     WX_THROW( Wx_general_error() ); 
   }
   if(tsp->tm_sec!=0) {
     WX_THROW( Wx_general_error() ); 
   }
   if(tsp->tm_wday!=4) {
     WX_THROW( Wx_general_error() ); 
   }
   if(tsp->tm_yday!=0) {
     WX_THROW( Wx_general_error() ); 
   }
 }

 //
 // Test set_systime
 //
 if(::getuid()==0) {
   // this is root
   WxTime tm0=Wx::now()-WxTime(10,0);   // tm to set systime
   WxTime tm1;
   WxTime tm_now=Wx::now();
   if(tm0==tm_now) {
     WX_THROW( Wx_general_error() );
   }
   if((r=Wx::set_systime(tm0))!=OK) {   // set default time
     WX_HERE(r); throw(r);
   }
   tm1=Wx::now();                       // read back
   if((r=Wx::set_systime(tm_now))!=OK) {// recover time
     WX_HERE(r); throw(r);
   }
   if((tm0-tm1).abs()>WxTime(0,60000000)) { // diff>0.06sec
     std::cerr << Wx::what_is(tm0) << "!=" << Wx::what_is(tm1) << std::endl;
     WX_THROW( Wx_general_error() ); 
   }

 } else {
   // non-superuser cannot set systeim time
   WxTime tm_now=Wx::now();
   if((r=Wx::set_systime(tm_now))!=WXM_EPERM) {
     WX_HERE(r); throw(r);  
   }
 }

 // test what_is(...)
 {
  if(Wx::what_is(WxTime(120,450))!=WxStr("120.00000045")) {
   WX_THROW( Wx_general_error() ); 
  }
  if(Wx::what_is(WxTime(0,450))!=WxStr("0.00000045")) {
   WX_THROW( Wx_general_error() ); 
  }
  if(Wx::what_is(WxTime(0,0))!=WxStr("0.0")) {
   WX_THROW( Wx_general_error() ); 
  }
 }
};

/*
   Test errors (set_systime is not tested)
*/
static void t2(void)
{
 
 //
 // constructor WXM_ERANGE
 //
 try {
   WxTime ttm(LLongMax,LongMax);   // should overflow
   WX_THROW( Wx_general_error() ); 
 }
 catch(const WxTime::Fault& e) {
   if(e!=WXM_ERANGE) {
     WX_THROW( Wx_general_error() );
   }
 };

 //
 // constructor WXM_ERANGE
 //
 try {
   WxTime ttm(LLongMin,LongMin);  // should underflow
   WX_THROW( Wx_general_error() ); 
 }
 catch(const WxTime::Fault& e) {
   if(e!=WXM_ERANGE) {
     WX_THROW( Wx_general_error() );
   }
 };

 //
 // Test assign(long long,long) overflow
 //
 {
  WxTime ttm;
  WxRet r;
  if((r=ttm.assign(LLongMax,999999999))!=OK) {  // this should be the max value
     WX_HERE(r); throw(r);
  }
  if((r=ttm.assign(LLongMax,1000000000))!=WXM_ERANGE) { // this should overflow
     WX_THROW( Wx_general_error() );
  }
 }

 //
 // Test add(long long,long) overflow
 //
 {
  WxTime ttm(LLongMax,999999999);   // maximum value
  WxRet r;
  if((r=ttm.add(0,0))!=OK) {
     WX_THROW( Wx_general_error() );
  }
  if((r=ttm.add(1,0))!=WXM_ERANGE) {
     WX_THROW( Wx_general_error() );
  }
  if((r=ttm.add(0,1))!=WXM_ERANGE) {
     WX_THROW( Wx_general_error() );
  }
 }

 //
 // Test add(WxTime) overflow
 //
 {
  WxTime ttm(LLongMax,999999999);   // maximum value
  WxRet r;
  if((r=ttm.add(0,1))!=WXM_ERANGE) {
     WX_HERE(r); throw(r);
  }
 }

 //
 // Test +=WxTime overflow
 //
 try {
   WxTime ttm(LLongMax,999999999);
   ttm+=WxTime(0,1);
   WX_THROW( Wx_general_error() ); 
 }
 catch(const WxTime::Fault& e) {
   if(e!=WXM_ERANGE) {
     WX_THROW( Wx_general_error() );
   }
 };

 //
 // Test WxTime + WxTime overflow
 //
 try {
   WxTime ttm(LLongMax,999999999);
   ttm=ttm+WxTime(0,1);
   WX_THROW( Wx_general_error() ); 
 }
 catch(const WxTime::Fault& e) {
   if(e!=WXM_ERANGE) {
     WX_THROW( Wx_general_error() );
   }
 };

 //
 // Test WxTime - WxTime underflow
 //
 try {
   WxTime ttm(LLongMin,-999999999);
   ttm=ttm-WxTime(0,1);
   WX_THROW( Wx_general_error() ); 
 }
 catch(const WxTime::Fault& e) {
   if(e!=WXM_ERANGE) {
     WX_THROW( Wx_general_error() );
   }
 };

 // 
 // Test operator -() failure condition
 //
 try {
   WxTime ttm(LLongMin,0);   // negate LLongMin should fail
   ttm=-ttm;
   WX_THROW( Wx_general_error() ); 
 }
 catch(const WxTime::Fault& e) {
   if(e!=WXM_MATHNEG) {
     WX_THROW( Wx_general_error() );
   }
 };

 // 
 // Test operator -() failure condition
 //
 try {
   WxTime ttm(LLongMin,0);   // negate LLongMin should fail
   ttm=ttm.abs();          
   WX_THROW( Wx_general_error() ); 
 }
 catch(const WxTime::Fault& e) {
   if(e!=WXM_MATHNEG) {
     WX_THROW( Wx_general_error() );
   }
 };

 // don't know how to generate get_time/set_time error
};

//
// Test random time arithmatic operations
//
static void t3(void)
{
 WxRet r;
 WxTime t1,t2,t3,t4,tt;
 long i;
 for(i=100000L; i>0; --i) {
   t1=WxTime(std::rand(),std::rand());
   t2=WxTime(std::rand(),std::rand());
   t3=t1+t2;
   t4=t1-t2;
   tt=t3+t4;
   if(tt!=t1+t1) {
     WX_HERE(r);
     break;
   }
   tt=t4-t3;
   if(tt!=-t2-t2) {
     WX_HERE(r);
     break;
   }
 }
 if(i>0) { // premature break, error happened
   std::cout << " t1= " << Wx::what_is(t1)
             << "\n t2= " << Wx::what_is(t2)
             << "\n t3= " << Wx::what_is(t3)
             << "\n t4= " << Wx::what_is(t4)
             << "\n tt= " << Wx::what_is(tt)
             << std::endl;
   WX_HERE(r); throw(r);
 }
};

//
//   Test the ensured range LLONG_MAX+0.999999999
//
static void t4(void) 
{
const WxTime WorkMax(LLongMax,999999999L);
const WxTime WorkMin(-LLongMax,-999999999L);
WxTime t1,t2,t3;

 //
 // Test -WorkMin
 //
 t1=-WorkMin;
 if(t1!=WorkMax) {
   WX_THROW( Wx_general_error() );
 }

 //
 // Test -WorkMax
 //
 t1=-WorkMax;
 if(t1!=WorkMin) {
   WX_THROW( Wx_general_error() );
 }

 //
 // Test WorkMin.abs()
 //
 t1=WorkMin.abs();
 if(t1!=WorkMax) {
   WX_THROW( Wx_general_error() );
 }

 //
 // Test WorkMax - 1nano
 //
 t1=WorkMax-WxTime(0,1);
 if(t1>=WorkMax) {
   WX_THROW( Wx_general_error() );
 }
 t1+=WxTime(0,1);
 if(t1!=WorkMax) {
   WX_THROW( Wx_general_error() );
 }

 //
 // Test WorkMax - 1sec
 //
 t1=WorkMax-WxTime(1,0);
 if(t1>=WorkMax) {
   WX_THROW( Wx_general_error() );
 }
 t1+=WxTime(1,0);
 if(t1!=WorkMax) {
   WX_THROW( Wx_general_error() );
 }

 //
 // Test WorkMin + 1nano
 //
 t1=WorkMin+WxTime(0,1);
 if(t1<=WorkMin) {
   WX_THROW( Wx_general_error() );
 }
 t1-=WxTime(0,1);
 if(t1!=WorkMin) {
   WX_THROW( Wx_general_error() );
 }

 //
 // Test WorkMin + 1sec
 //
 t1=WorkMin+WxTime(1,0);
 if(t1<=WorkMin) {
   WX_THROW( Wx_general_error() );
 }
 t1-=WxTime(1,0);
 if(t1!=WorkMin) {
   WX_THROW( Wx_general_error() );
 }

 //
 // Constructor WXM_ERANGE
 //
 try {
   WxTime ttm(LLongMax,LongMax);
   WX_THROW( Wx_general_error() ); 
 }
 catch(const WxTime::Fault& e) {
   if(e!=WXM_ERANGE) {
     WX_THROW( Wx_general_error() );
   }
 };
}

//
// Check sleep_till
//
#ifdef TEST_SLEEP_TILL
static void t5(void)
{
 WxTime t0=Wx::now();
 WxRet r=Wx::sleep_till(t0+WxTime(1,0)); // delay one second
 if(r!=OK) {
   WX_HERE(r); throw(r);
 }
 WxTime t1=Wx::now()-t0;
 if(  (t1<WxTime(0,940000000))      // 0.94 sec
    ||(t1>WxTime(1, 60000000))) {   // 1.06 sec
   WX_THROW( Wx_general_error() );  // error> 0.06 sec
 }
};
#endif

static const WxStr chdr(
                  "+-------------------------+\n"
                  "| main() caught exception:|\n"
                  "+-------------------------+\n");
int main(void) throw()
try {
 std::cout << "Checking wxtime.h ...\n";
 if(std::strcmp(WxTime::class_name,"WxTime")!=0) {
   WX_THROW( Wx_general_error() );
 }
 t1();
 t2();
 t3();
 t4();
#ifdef TEST_SLEEP_TILL
 t5();
#endif
 std::cout << "It is now()= " << Wx::what_is(Wx::now()) << std::endl;
 std::cout << "Checked OK\n";
 return(0);
}
catch(const WxRet& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_general_error& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_bad_errno& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_except& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_bad_alloc& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const std::exception& e) {
 std::cerr << chdr << "std::exception" << std::endl;
 return(-1);
}
catch(...) {
 std::cerr << chdr << "unknown exception" << std::endl;
 return(-1);
};
